#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/07/28
@file: check_es3000_version.py
@function:
"""
import re

from Common.base import context_util
from Common.base import entity
from Common.base.constant import MsgKey
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import Compare, ResultFactory
from Common.base.entity import DeployException
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
DRIVER_VERSION_START = "version:"
DISK_NAME_START = "|---- "
FIRMWARE_VERSION_START = "firmware version"
FIRMWARE_MODEL_START = "model number"
TOOL_VERSION_KEY = "ES_3000_Tool_Ver"
DRIVER_VERSION_KEY = "ES_3000_Driver_Ver"
FIRMWARE_VERSION_KEY = "ES_3000_{}_FW_Ver"
# 当前支持的是v5、v6盘、v6盘441，443型号、v6盘447型号
ES3000_NUMS = ("5", "6", "6SP", "6-447")
ES3000_NUM_REG = re.compile(r"HWE(\d)")
DRIVER_VERSION_CMD = "cat /sys/module/nvme/version"
MODEL_NUMS = {
    'HWE62P441T6M005N,HWE62P443T2M005N': "SP",
    'HWE62P447T6N005N,HWE62P447T6L00BN,HWE62P441T6M00BN,HWE62P441T9L00BN,'
    'HWE62P443T8L00BN,HWE62P443T2M00BN,HWE62P446T4M00BN': "-447",
}
ES3000_V5 = "HSSD-D6|HWE5"
ES3000_V6 = "HSSD-D7|HWE6"


def execute(task):
    return Es3000VersionCheck(task).do_check()


def get_ver_num_after_fix(model):
    for key, value in MODEL_NUMS.items():
        if model in key:
            return value
    return ""


class Es3000VersionCheck(object):

    def __init__(self, task):
        self.task = task
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._err_msgs = list()
        self._ssh_rets = list()
        self._tool_mapping_ver = ""
        self._tool_mapping_url = ""
        self._driver_mapping_ver = ""
        self._driver_mapping_url = ""
        self._firmware_mapping_ver_dict = dict()
        self._firmware_mapping_url_dict = dict()
        self.not_support_msg = entity.create_msg(MsgKey.NOT_INVOLVE)
        self._check_firmware_method = self._init_method()
        self._check_tool_flag = True
        self._check_fw_flag = True

    def do_check(self):
        version_keys = [TOOL_VERSION_KEY, DRIVER_VERSION_KEY]
        if not context_util.contain_need_check_key(PY_JAVA_ENV, version_keys):
            return ResultFactory.create_pass()
        self._obtain_match_versions()
        match_msg = self._get_match_msg()
        try:
            if not self._contains_es3000_disk():
                return ResultFactory.create_pass(self._ssh_rets, match_msg)
            driver_match = self._check_mapping_driver_version()
            self._check_firmware_method.get(context_util.get_scene(PY_JAVA_ENV), self._check_in_deploy_scene)()
            self._err_msgs.append(match_msg)
            if driver_match and self._check_fw_flag and self._check_tool_flag:
                return ResultFactory.create_pass(self._ssh_rets, self._err_msgs)
            self._err_msgs.insert(0, entity.build_driver_tool_tips())
            return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
        except DeployException as e:
            self._ssh_rets.append(e.origin_info)
            self._logger.error(e.message)
            self._err_msgs.append(e.err_msg)
            self._err_msgs.insert(0, match_msg)
            if e.may_info_miss():
                self.task.openAutoRetry()
            return ResultFactory.create_not_pass(self._ssh_rets,
                                                 self._err_msgs)

    def _init_method(self):
        return {"Distributed Deploy": self._check_mapping_firmware_with_storage}

    def _check_in_deploy_scene(self):
        self._logger.info("default firmware method executed")
        self._check_mapping_tool_version()
        self._check_mapping_firmware_version()

    def _obtain_match_versions(self):
        self._driver_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, DRIVER_VERSION_KEY)
        self._driver_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, DRIVER_VERSION_KEY)
        self._tool_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, TOOL_VERSION_KEY)
        self._tool_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, TOOL_VERSION_KEY)
        for num in ES3000_NUMS:
            mapping_ver = get_mapping_attribute(
                PY_JAVA_ENV, FIRMWARE_VERSION_KEY.format(num))
            if mapping_ver:
                self._firmware_mapping_ver_dict[num] = mapping_ver
                self._firmware_mapping_url_dict[num] = get_mapping_attribute_url(
                    PY_JAVA_ENV, FIRMWARE_VERSION_KEY.format(num))

    def _get_match_msg(self):
        driver_mapping_msg = entity.build_url_error_msg(self._driver_mapping_url, self._driver_mapping_ver) if \
            self._driver_mapping_ver else self.not_support_msg
        tool_mapping_msg = entity.build_url_error_msg(self._tool_mapping_url, self._tool_mapping_ver) if \
            self._tool_mapping_ver else self.not_support_msg
        if self._firmware_mapping_ver_dict:
            firmware_mapping_msg = ""
            for num, version in self._firmware_mapping_ver_dict.items():
                url = self._firmware_mapping_url_dict.get(num)
                num = self._get_special_model_desc(num)
                firmware_mapping_msg += entity.build_url_error_msg(url, "[V{}: {}] ".format(num, version))
        else:
            firmware_mapping_msg = self.not_support_msg
            self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_FW.getKey(), self.not_support_msg)
        if not self._tool_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_TOOL.getKey(), self.not_support_msg)
        if not self._tool_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_DRIVER.getKey(),
                                        self.not_support_msg)
        match_msg = entity.create_msg("es3000.match.version").format(
            driver_mapping_msg, tool_mapping_msg, firmware_mapping_msg)
        return "\n" + entity.create_source_file_msg(PY_JAVA_ENV, match_msg)

    def _contains_es3000_disk(self):
        cmd = "lspci -mm -n -D | grep 0108|tr -d '\"'|awk '{print $3}'"
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
        self._ssh_rets.append(ssh_ret)
        for line in ssh_ret.splitlines():
            if line.strip() == "19e5":
                return True
        return False

    def _check_mapping_driver_version(self):
        if not self._driver_mapping_ver:
            self._logger.info("not need check driver version")
            return True
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, DRIVER_VERSION_CMD)

        self._ssh_rets.append(ssh_ret)
        if "No such file" in ssh_ret:
            raise DeployException(
                "not found driver version",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        ret_str = ssh_ret.splitlines()
        fw_vers = []
        for index in range(len(ret_str)):
            if DRIVER_VERSION_CMD in ret_str[index] and index + 1 < len(ret_str):
                ret_ver = ret_str[index + 1]
                fw_vers.append(ret_ver)
                if Compare.compare_digital_version(ret_ver, self._driver_mapping_ver) < 0:
                    self._err_msgs.append(entity.create_msg(DRIVER_VERSION_KEY).format(ret_ver))
                    self.deploy_node.putResult(context_util.get_version_key_enum().ES3000_DRIVER.getKey(),
                                               context_util.get_not_pass_key())
                    return False
                self._err_msgs.append(entity.create_msg(DRIVER_VERSION_KEY).format(ret_ver))
        self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_DRIVER.getKey(), ";".join(set(fw_vers)))
        return True

    def _check_mapping_tool_version(self):
        if not self._tool_mapping_ver:
            self._logger.info("not need check tool version")
            return
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "hioadm -v")
        self._ssh_rets.append(ssh_ret)
        res = re.findall(r"hioadm version (.+) Copyright", ssh_ret)
        if res:
            self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_TOOL.getKey(), res[0])
            if Compare.compare_digital_version(
                    res[0], self._tool_mapping_ver) < 0:
                self.deploy_node.putResult(context_util.get_version_key_enum().ES3000_TOOL.getKey(),
                                           context_util.get_not_pass_key())
                self._check_tool_flag = False
            self._err_msgs.append(entity.create_msg(TOOL_VERSION_KEY).format(res[0]))
            return
        raise DeployException(
            "match tool version failed",
            err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _check_mapping_firmware_with_storage(self):
        try:
            self._logger.info("storage node firmware method executed")
            if not self._firmware_mapping_ver_dict:
                self._logger.info("not need check firmware version")
                return
            self._check_mapping_ver_by_firmware_num("5", self._query_disk_fireware_info(ES3000_V5))

            self._check_mapping_ver_by_firmware_num("6", self._query_disk_fireware_info(ES3000_V6))

        except DeployException as exception:
            # 因为场景无法区分安装前检查还是安装后检查，所以在没有show_disk.sh脚本时，按照开局前处理，用工具检查版本
            if DeployException.ErrCode.MAY_INFO_MISS != exception.err_code:
                raise exception
            self._check_in_deploy_scene()

    def _check_mapping_ver_by_firmware_num(self, firmware_num, firmware_list):
        if not firmware_list:
            self._logger.info("have no nvme disk for firmware num :{}".format(firmware_num))
            return
        fw_vers = []
        for firmware in firmware_list:
            # 特殊铭牌的盘根据配套表版本后缀处理
            version_key = firmware_num + get_ver_num_after_fix(firmware.get("model"))
            ver = firmware.get("version")
            fw_vers.append(ver)
            self._compare_mapping_ver(version_key, firmware.get("name"), firmware_num, ver)
        self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_FW.getKey(), ';'.join(set(fw_vers)))

    def _query_disk_fireware_info(self, filter_str):
        """
        查询硬盘固件信息
        :param filter_str: 过滤条件
        :return: 命中的盘信息列表
        """
        firmware_list = list()
        cmd = "/opt/fusionstorage/persistence_layer/agent/tool/show_disk.sh | grep -E '{}'".format(filter_str)
        origin = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
        if ssh_util.is_invalid_cmd(origin):
            # 因为场景无法区分安装前检查还是安装后检查，所以在没有show_disk.sh脚本时，按照开局前处理，用工具检查版本
            raise DeployException("no show_disk.sh", err_code=DeployException.ErrCode.MAY_INFO_MISS)
        self._ssh_rets.append(origin)
        for line in origin.splitlines():
            nvme_infos = line.split("|")
            if len(nvme_infos) < 6:
                continue
            self._build_nvme_info(firmware_list, nvme_infos, filter_str)
        self._logger.info("filter disks :{} for filter_str:{}".format(firmware_list, filter_str))
        return firmware_list

    def _build_nvme_info(self, firmware_list, nvme_infos, filter_str):
        name = nvme_infos[1].strip()
        firmware_message = nvme_infos[5]
        firmware_messages = firmware_message.split(",")
        firmware_version = firmware_messages[2]
        firmware_model = firmware_messages[1]
        if not self._is_sata_disk(firmware_model) or filter_str == ES3000_V5:
            model_firmware = {
                "model": firmware_model,
                "name": name,
                "type": "ES3000",
                "version": firmware_version
            }
            firmware_list.append(model_firmware)

    def _is_sata_disk(self, firmware_model):
        # HWE系列Model号的6-7位是盘片接口类型，SATA盘标志是“ST”
        if firmware_model.startswith("HWE"):
            return firmware_model[5:6] == "ST"
        # HSSD系列Model号的11位是盘片接口类型，SATA盘标志是“S”
        if firmware_model.startswith("HSSD"):
            return firmware_model[10] == "S"
        return False

    def _check_mapping_firmware_version(self):
        if not self._firmware_mapping_ver_dict:
            self._logger.info("not need check firmware version")
            return
        disk_names = self._get_disk_names()
        self._logger.info("disk names: {}".format(disk_names))
        nvme_disk_names = self._filter_nvme_disk_names(disk_names)
        fw_vers = []
        for name in nvme_disk_names:
            num = self._get_firmware_num(name)
            if num in self._firmware_mapping_ver_dict:
                ver = self._get_firmware_version(name)
                fw_vers.append(ver)
                self._compare_mapping_ver(num, name, num, ver)
            self._logger.info("not support disk: {}".format(name))
        self.deploy_node.putVersion(context_util.get_version_key_enum().ES3000_FW.getKey(), ';'.join(set(fw_vers)))

    def _compare_mapping_ver(self, version_key, name, num, ver):
        mapping_ver = self._firmware_mapping_ver_dict.get(version_key)
        if not mapping_ver:
            self._logger.info("key :{} num:{} not support disk: {}".format(version_key, mapping_ver, name))
            return
        if Compare.compare_digital_version(ver, mapping_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().ES3000_FW.getKey(),
                                       context_util.get_not_pass_key())
            self._check_fw_flag = False
        self._err_msgs.append(entity.create_msg("es3000.fw.ver.not.match").format(name, num, ver))

    def _get_disk_names(self):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, "hioadm info")
        self._ssh_rets.append(ssh_ret)
        disk_names = list()
        if "Executed Error" in ssh_ret:
            raise DeployException(
                "Scan disk failed",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        for line in ssh_ret.splitlines():
            if line.startswith(DISK_NAME_START):
                # |---- nvme0 (032WEUFSK8007786)
                disk_name = line.split(DISK_NAME_START)[1].split("(")[0]
                disk_names.append(disk_name.strip())
        return disk_names

    def _filter_nvme_disk_names(self, disk_names):
        return filter(lambda disk_name: disk_name.startswith("nvme"),
                      disk_names)

    def _get_firmware_version(self, disk_name):
        ssh_ret = ssh_util.exec_ssh_cmd(
            PY_JAVA_ENV, "hioadm info -d {}".format(disk_name))
        self._ssh_rets.append(ssh_ret)
        for line in ssh_ret.splitlines():
            if line.startswith(FIRMWARE_VERSION_START):
                return line.split(":")[1].strip()
        raise DeployException(
            "not found FW {} version".format(disk_name),
            err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _get_firmware_num(self, disk_name):
        """
        获取ES3000盘的系列数字数字，v5，v6等
        :return: 系列数字 str
        """
        model = self._get_firmware_model(disk_name)
        match = ES3000_NUM_REG.search(model)
        if match:
            ver_num = match.groups()[0]
            return ver_num + get_ver_num_after_fix(model)

    def _get_firmware_model(self, disk_name):
        ssh_ret = ssh_util.exec_ssh_cmd(
            PY_JAVA_ENV, "hioadm info -d {} -a".format(disk_name))
        self._ssh_rets.append(ssh_ret)
        for line in ssh_ret.splitlines():
            if line.startswith(FIRMWARE_MODEL_START):
                return line.split(":")[1].strip()
        raise DeployException(
            "not found FW {} model".format(disk_name),
            err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _get_special_model_desc(self, num):
        num_no_sp = num
        # 改成V6(HWE62P441T6M005N, HWE62P443T2M005N)格式
        for key, value in MODEL_NUMS.items():
            if value not in num:
                continue
            num_no_sp = num.replace(value, '')
            num_no_sp += '({})'.format(key)
        return num_no_sp
