#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
"""
@time: 2024/10/30
@file: check_expansion_module_version.py
@function: 检查9550的SAS级联板的固件和驱动版本号

"""
import json
import re

from Common.base import context_util
from Common.base import entity
from Common.base.entity import Compare
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
LOGGER = entity.create_logger(__file__)
LINE_BREAK_STR = "\n\n"
MAPPING_KEY_SAS1_FW_VER = "SAS_Expansion_Module_FW_Ver_10"
MAPPING_KEY_SAS2_FW_VER = "SAS_Expansion_Module_FW_Ver_20"
MAPPING_KEY_SAS_DRIVER_VER = "SAS_Expansion_Module_Driver_Ver"
SUPPORT_PRODUCTS = (
    "STL6SPCM", "STL6SPCO", "STL6SPCX", "STL6SPCY", "STL6SPCY55", "STL6SPCY44", "STL6SPCH01", "STL6SPCH02"
)
CMD_QUERY_TYPE_EAST_SEA = "ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x27 0x58 0x00 0x06 0x00 0x02"
CMD_QUERY_TYPE_PACIFIC = "ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x5b 0x31 0x00 0x00 0x00"
CMD_QUERY_TYPE_DICT = {
    # 9546 9346 9146
    "eastSea": CMD_QUERY_TYPE_EAST_SEA,
    #  9550 9350 9150
    "pacific": CMD_QUERY_TYPE_PACIFIC
}

# 命令回显信息
ret_info = list()
# 错误信息
err_msgs = list()
# 提示信息(当前配备和设备版本号，不涉及信息)
driver_tip_info = list()
fw_tip_info = list()


def execute(task):
    try:
        # 检查SAS驱动版本号
        check_sas_driver_version()
        # 检查SAS固件版本号
        check_sas_fw_version()
        # 检查是否报错，返回检查状态
        if err_msgs:
            return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), __build_failed_result_msgs())
        return ResultFactory.create_pass(LINE_BREAK_STR.join(ret_info), "\n".join(__build_tips_info()))
    except DeployException as ex:
        LOGGER.error(ex.message)
        LOGGER.error("query sas version failed:{}.".format(ex.message))
        return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), ex.err_msg)


def __build_failed_result_msgs():
    result_msgs = __build_tips_info()
    if result_msgs:
        result_msgs.append("\n")
    result_msgs.extend(err_msgs)
    return entity.create_source_file_msg(PY_JAVA_ENV, "\n".join(result_msgs))


def __build_tips_info():
    """
    生成最终的提示信息，包括版本信息，不涉及等。先展示驱动再展示固件。
    :return: 提示信息列表
    """
    return driver_tip_info + fw_tip_info


def get_table_sas_config_version(attribute_key):
    """
    从版本配套表中获取配置推荐版本
    :param attribute_key: 版本号的KEY
    :return: 配套版本号，配套的升级包地址
    """
    table_version = context_util.get_mapping_attribute(PY_JAVA_ENV, attribute_key)
    table_url = context_util.get_mapping_attribute_url(PY_JAVA_ENV, attribute_key)
    return table_version, table_url


def check_sas_fw_version():
    """
    检查SAS级联板的固件版本号是否满足版本配套表的要求
    """
    # 检查SAS级联板类型
    sas_version = check_sas_expansion_module_type()
    LOGGER.info("current sas version : {}.".format(sas_version))
    if sas_version == 1:
        check_sas1_fw_version()
    elif sas_version == 2:
        check_sas2_fw_version()
    else:
        LOGGER.error("check sas type failed.")
        err_msgs.append(entity.create_msg("check.expansion.module.not.ipmitool"))


def check_sas_expansion_module_type():
    """
    查询级联板的类型，通过ipmitool命令查询，执行命令如下：
    [root@FSM0 ~]# ipmitool raw 0x30 0x93 0xdb 0x07 0x00 0x5b 0x31 0x00 0x00 0x00
    db 07 00 00
    [root@FSM0 ~]#
    :return: 如果是SAS1.0的级联板，返回1，如果是SAS2.0的级联板，返回2，否则返回0
    """
    platform_id = context_util.get_platform_id(PY_JAVA_ENV)
    LOGGER.info("current platform id : {}.".format(platform_id))
    if platform_id not in CMD_QUERY_TYPE_DICT.keys():
        LOGGER.error("current platform not support.")
        return 0
    query_cmd = CMD_QUERY_TYPE_DICT.get(platform_id)
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, query_cmd)
    ret_info.append(ssh_ret)
    for line in ssh_ret.splitlines():
        line = line.strip()
        if not line.startswith("db 07"):
            continue
        # SAS1.0的级联板
        if 'db 07 00 00' == line or 'db 07 00 00 0d 00' == line:
            return 1
        # SAS2.0的级联板
        if 'db 07 00 01' == line or 'db 07 00 00 0a 00' == line:
            return 2
    return 0


def check_sas1_fw_version():
    """
    检查SAS1.0的级联板的固件版本号
    :return: 检查结果
    """
    # 查询设备的ID和版本号的字典
    device_fw_versions = query_sas1_fw_versions()
    LOGGER.info("current sas1 device fw versions : {}.".format(json.dumps(device_fw_versions)))
    check_sas_fw_versions(MAPPING_KEY_SAS1_FW_VER, "1.0", device_fw_versions)


# ====================================SAS1的固件检查（开始）====================================
def query_sas1_fw_versions():
    """
    查询各个Host ID的固件版本号
    :return: 字典，{Device ID ： FW Version}
    """
    # 解析设备符
    device_ids = query_sas1_device_ids()
    LOGGER.info("current sas1 device ids : {}.".format(",".join(device_ids)))
    # 查询每个设备的版本号
    return query_sas1_fw_version_by_device_id(device_ids)


def query_sas1_device_ids():
    """
    查询SAS1.0版本的设备符，执行命令如下：
    [root@FSM0 ~]# lsscsi -g | grep enclosu | awk -F' ' '{print $NF}'
    /dev/sg2
    /dev/sg3
    [root@FSM0 ~]#
    :return:
    """
    cmd = "lsscsi -g | grep enclosu | awk -F' ' '{print $NF}'"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    device_ids = set()
    # 解析设备符
    for line in ssh_ret.splitlines():
        if line.startswith('/dev/'):
            device_ids.add(line)
    return device_ids


def query_sas1_fw_version_by_device_id(device_ids):
    """
    解析SAS1.0版本级联板的固件版本号
    执行命令如下：
    [root@FSM0 ~]# sg_ses -p 0x97 /dev/sg2
    PMCSIERA  SXP 36x12G        RevB
    Cannot decode response from diagnostic page: dpage 0x97
     00     97 00 00 20 36 30 2e 30  33 2e 30 36 54 30 31 00    ... 60.03.06T01.
     10     00 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00    ................
     20     00 00 00 00
    [root@FSM0 ~]#
    :param device_ids: 设备符列表
    :return: 设备符对应的版本号的字典(Device Id : FW Version)
    """
    device_fw_version = dict()
    # 查询每个设备符
    cmd = "sg_ses -p 0x97 {}"
    for device_id in device_ids:
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd.format(device_id))
        ret_info.append(ssh_ret)
        fw_version = ""
        for line in ssh_ret.splitlines():
            # 通过正则解析版本号，固定格式：XX.XX.XXTXX 或者 XX.XXTXX
            match_result = re.search(r"(\d+\.){1,2}\d+T\d+", line.strip())
            if match_result:
                fw_version = match_result.group()
                break
        LOGGER.info("query sas1 fw version : {} -> {}".format(device_id, fw_version))
        if fw_version:
            device_fw_version[device_id] = fw_version
    return device_fw_version


# ====================================SAS1的固件检查（结束）====================================
def check_sas2_fw_version():
    """
    检查SAS2.0的级联板的固件版本号
    :return: 检查结果
    """
    if not check_sas2_hw_envir():
        # 没有环境或不支持
        LOGGER.error("current env not found sas fw.")
        err_msgs.append(entity.create_msg("check.expansion.module.not.install").format("2.0"))
        return
    # 查询设备的ID和版本号的字典
    device_fw_versions = query_sas2_fw_versions()
    LOGGER.info("current sas2 device fw versions : {}.".format(json.dumps(device_fw_versions)))
    check_sas_fw_versions(MAPPING_KEY_SAS2_FW_VER, "2.0", device_fw_versions)


def check_sas_fw_versions(mapping_key, version, device_fw_versions):
    """
    检查sas的固件版本是否满足要求
    :param mapping_key: 当前SAS固件版本号的KEY
    :param version:SAS的版本号，1.0还是2.0
    :param device_fw_versions: 设备上固件的版本，字典：{"Device Id" : "FW Version"}
    :return: 无
    """
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    if not device_fw_versions:
        LOGGER.error("not found fw version.")
        err_msgs.append(entity.create_msg("check.expansion.module.not.install").format(version))
        deploy_node.putVersion(mapping_key, "--")
        return
    # 读取配套的固件版本号
    table_version, table_url = get_table_sas_config_version(mapping_key)
    LOGGER.info("current table sas fw version : {}.".format(table_version))
    if not table_version:
        fw_tip_info.append(entity.create_msg("check.expansion.module.fw.not.support").format(version))
        return
    not_match_device_ids = list()
    format_versions = ""
    for device_id, fw_version in device_fw_versions.items():
        if Compare.compare_digital_version(format_fw_version(fw_version), format_fw_version(table_version)) >= 0:
            fw_tip_info.append(
                entity.create_msg("check.expansion.module.current.fw.version").format(version, device_id, fw_version))
            format_versions = format_versions + "{}:{}\n".format(device_id, fw_version)
        else:
            # 不匹配，版本过低
            LOGGER.error("sas device not match, device id : {}, version : {}.".format(device_id, fw_version))
            not_match_device_ids.append(device_id)
            fw_tip_info.append(
                entity.create_msg("check.expansion.module.module.fw.version").format(version, device_id, fw_version,
                    table_version))
    deploy_node.putVersion(mapping_key, format_versions)
    # 不匹配的设备ID列表
    if not_match_device_ids:
        device_id_desc = ",".join(not_match_device_ids)
        url_desc = entity.create_msg("check.expansion.module.fw.not.match").format(version, device_id_desc)
        if table_url:
            url_desc = entity.build_url_error_msg(table_url, url_desc)
        err_msgs.append(url_desc)
        deploy_node.putResult(mapping_key, context_util.get_not_pass_key())


def format_fw_version(version):
    """
    格式化固件的版本，因固件版本格式为："(\d+\.){1,2}\d+T\d+"，存在两种格式：XX.XX.XXTXX或者XX.XXTXX
    并且65.XX的版本号小于60.XX.XX的格式，因为为了补齐版本号比较大小，统一处理为XX.XX.XXTXX的格式
    :param version: 原始版本号
    :return: 格式化后的三位版本号
    """
    if version.count(".") == 1:
        return "00.{}".format(version)
    return version


# ====================================SAS2的固件检查（开始）====================================
def check_sas2_hw_envir():
    """
    检查当前环境是否有SAS2的级联板
    :return: 是否需要检查
    """
    product_name = query_sas2_product_name()
    LOGGER.info("query sas2 product name : {}.".format(product_name))
    if product_name not in SUPPORT_PRODUCTS:
        return False
    chip_name = query_sas2_chip_num()
    LOGGER.info("query sas2 chip number : {}.".format(chip_name))
    return chip_name > 0


def query_sas2_fw_versions():
    """
    查询各个Host ID的固件版本号
    :return: 字典，{Host ID ： FW Version}
    """
    host_id_fw_version_dict = dict()
    host_ids = query_sas2_host_ids()
    LOGGER.info("current sas2 hosts ids : {}.".format(",".join(host_ids)))
    for host_id in host_ids:
        if not check_sas2_host_id_belongs(host_id):
            continue
        fw_version = query_sas2_host_id_fw_version(host_id)
        if fw_version:
            host_id_fw_version_dict[host_id] = fw_version
    return host_id_fw_version_dict


def parse_num_lines_in_ssh_ret(ssh_ret):
    """
    解析SSH回显中的纯数字行，如果没有返回空列表
    :param ssh_ret: SSH回显
    :return: 只包含数字的字符串类型的列表，如果没有，则返回空列表
    """
    numbers = list()
    for line in ssh_ret.splitlines():
        line = line.strip()
        if re.match(r"^\d+$", line):
            numbers.append(line)
    return numbers


def query_sas2_chip_num():
    """
    查询SAS2.0的CHIP数量，查询命令回显：
    [root@FSM0 ~]# lspci -n|grep -E '1e81:37d8|19e5:37d8' |wc -l
    8
    [root@FSM0 ~]#
    :return: 数量
    """
    cmd = "lspci -n|grep -E '1e81:37d8|19e5:37d8' |wc -l"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    number_lines = parse_num_lines_in_ssh_ret(ssh_ret)
    return int(number_lines[0]) if number_lines else 0


def query_sas2_product_name():
    """
    查询当前环境的产品名称，查询命令回显：
    [root@FSM0 ~]# dmidecode -t 2 |grep -i "Product Name" -i | awk -F ': ' '{print $2}'
    STL6SPCM
    [root@FSM0 ~]#
    :return: 产品名称
    """
    cmd = "dmidecode -t 2 |grep -i \"Product Name\" | awk -F ': ' '{print $2}'"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    return ssh_ret.splitlines()[-2].strip()


def query_sas2_host_ids():
    """
    查询SAS2.0的主机ID，查询命令回显：
    [root@FSM0 ~]# lsscsi -H |grep "sal_layer" | awk -F"[][]"  '{print $2}'
    8
    9
    [root@FSM0 ~]#
    :return: Host ID集合列表
    """
    cmd = "lsscsi -H |grep \"sal_layer\" | awk -F\"[][]\"  '{print $2}'"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    return parse_num_lines_in_ssh_ret(ssh_ret)


def check_sas2_host_id_belongs(host_id):
    """
    查询当前的HOST ID是否合法，查询命令回显：
    [root@FSM0 ~]# cat /sys/class/scsi_host/host8/host_belongs'
    1
    [root@FSM0 ~]#
    :param host_id: HOST ID
    :return: 是否合法
    """
    cmd = "cat /sys/class/scsi_host/host{}/host_belongs".format(host_id)
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    number_lines = parse_num_lines_in_ssh_ret(ssh_ret)
    return number_lines[0] == "1" if number_lines else False


def query_sas2_host_id_fw_version(host_id):
    """
    查询当前的Host Id的固件版本号，查询命令回显：
    [root@FSM0 ~]# cat /sys/class/scsi_host/host8/fw_version
    Firmware:1.8.1.23
    [root@FSM0 ~]#
    :param host_id: Host ID
    :return: 版本号
    """
    cmd = "cat /sys/class/scsi_host/host{}/fw_version".format(host_id)
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    for line in ssh_ret.splitlines():
        if "Firmware" in line:
            return line.split(":")[1].strip()
    return ""


# ====================================SAS2的固件检查（结束）====================================
def check_sas_driver_version():
    """
    检查SAS驱动版本号是否满足配套表要求
    :return: 无
    """
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    driver_version = query_sas_driver_version()
    LOGGER.info("query sas driver version : {}.".format(driver_version))
    if not driver_version:
        err_msgs.append(entity.create_msg("check.expansion.module.not.version"))
        deploy_node.putVersion(MAPPING_KEY_SAS_DRIVER_VER, "--")
        return
    table_version, table_url = get_table_sas_config_version(MAPPING_KEY_SAS_DRIVER_VER)
    LOGGER.info("current sas driver version : {}.".format(table_version))
    if not table_version:
        driver_tip_info.append(entity.create_msg("check.expansion.module.driver.not.support"))
        return
    driver_tip_info.append(entity.create_msg("check.expansion.module.current.driver.version").format(driver_version))
    deploy_node.putVersion(MAPPING_KEY_SAS_DRIVER_VER, driver_version)
    # 不匹配，版本过低
    if Compare.compare_digital_version(driver_version, table_version) < 0:
        LOGGER.error("sas driver not match, driver version: {}..".format(driver_version))
        url_desc = entity.create_msg("check.expansion.module.driver.not.match")
        if table_url:
            url_desc = entity.build_url_error_msg(table_url, url_desc)
        err_msgs.append(url_desc)
        deploy_node.putResult(MAPPING_KEY_SAS_DRIVER_VER, context_util.get_not_pass_key())


# ====================================SAS1/SAS2的驱动检查（开始）====================================
def query_sas_driver_version():
    """
    查询SAS驱动版本号，回显示例：
    [root@FSM0 ~]# rpm -qi pangea-hisisas-driver-euler | grep -i version
    Version     : 1.99.32
    [root@FSM0 ~]#
    :return:
    """
    cmd = "rpm -qi pangea-hisisas-driver-euler  |  grep -i version"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    for line in ssh_ret.splitlines():
        if "Version" in line:
            return line.split(":")[1].strip()
    return ""
# ====================================SAS1/SAS2的驱动检查（结束）====================================
