#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/12/26
@file: check_ibmc_firmware_version.py
@function:
"""

from Common.base import context_util
from Common.base import entity
from Common.base.constant import VE_SUPPORTED_NUM, V6_SUPPORTED_NUM
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory, Compare
from Common.protocol.redfish import redfish_util
from Common.service import check_main_board_uids_service

PY_JAVA_ENV = py_java_env
IBMC_FW_VER_KEY = "Main_Bord_iBMC_Ver_{}"
OCEANS_IBMC_FW_VER_KEY = "Oceans_Main_Bord_iBMC_Ver"
ATLANTIC_SSM_IBMC_FW_VER_KEY = "Atlantic_Ssm_iBMC_Ver"


def execute(task):
    login_info = context_util.get_login_info(PY_JAVA_ENV)
    logger = entity.create_logger(__file__)
    err_msgs = list()
    origin_infos = list()
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    try:
        mapping_url, mapping_version = init_mapping_msg(logger, login_info, origin_infos)

        # 获取设备ibmc版本进行判断
        resource = redfish_util.get_managers_info(login_info, logger)
        dev_ibmc_fw_version = resource.get("FirmwareVersion")
        logger.info("iBMC fw version: {}".format(dev_ibmc_fw_version))
        # 去掉其中的括号的描述
        dev_ibmc_fw_version = dev_ibmc_fw_version.split("(")[0].strip()
        origin_infos.append(str(resource))
        version_match = True
        if Compare.compare_digital_version(dev_ibmc_fw_version, mapping_version) < 0:
            version_match = False
        err_msgs.append(entity.create_msg("ibmc.fw.current.version").format(dev_ibmc_fw_version))
        deploy_node.putVersion(context_util.get_version_key_enum().IBMC_FW.getKey(), dev_ibmc_fw_version)
        err_msgs.append(entity.build_url_error_msg(mapping_url,
                                                   entity.create_msg("ibmc.match.fw.version").format(mapping_version)))
        err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, ""))
        if version_match:
            return ResultFactory.create_pass(origin_infos, err_msgs)
        err_msgs.insert(0, entity.build_fw_tool_tips())
        deploy_node.putResult(context_util.get_version_key_enum().IBMC_FW.getKey(), context_util.get_not_pass_key())
        return ResultFactory.create_not_pass(origin_infos, err_msgs)
    except DeployException as e:
        origin_infos.append(e.origin_info)
        logger.error(e.message)
        err_msgs.append(e.err_msg)
        return ResultFactory.create_not_pass(origin_infos, err_msgs)


def init_mapping_msg(logger, login_info, origin_infos):
    # 查询主板类型
    if context_util.is_ocean_platform(PY_JAVA_ENV):
        mapping_version = context_util.get_mapping_attribute(PY_JAVA_ENV, OCEANS_IBMC_FW_VER_KEY)
        mapping_url = context_util.get_mapping_attribute_url(PY_JAVA_ENV, OCEANS_IBMC_FW_VER_KEY)
    elif context_util.is_atlantic_ssm_platform(PY_JAVA_ENV):
        mapping_version = context_util.get_mapping_attribute(PY_JAVA_ENV, ATLANTIC_SSM_IBMC_FW_VER_KEY)
        mapping_url = context_util.get_mapping_attribute_url(PY_JAVA_ENV, ATLANTIC_SSM_IBMC_FW_VER_KEY)
    else:
        main_bord_num, resource = redfish_util.get_ibmc_main_board_num(
            login_info, logger)
        origin_infos.append(resource)
        # 判断是不是VE机型主板（1620s 2.0主板）
        if check_main_board_uids_service.is_ve_supported_main_board(login_info, logger):
            main_bord_num = main_bord_num + VE_SUPPORTED_NUM
        elif check_main_board_uids_service.is_v6_supported_main_board(login_info, logger):
            main_bord_num += V6_SUPPORTED_NUM
        logger.info("iBMC fw main_bord_num: {}".format(main_bord_num))
        # 根据ibmc主板类型获取配套版本
        mapping_version = context_util.get_mapping_fw_version_with_haiyan(
            PY_JAVA_ENV, IBMC_FW_VER_KEY.format(main_bord_num))
        mapping_url = context_util.get_mapping_fw_url_with_haiyan(
            PY_JAVA_ENV, IBMC_FW_VER_KEY.format(main_bord_num))
    return mapping_url, mapping_version
