#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
"""
@time: 2024/11/12
@file: check_net_driver_version.py
@function: 检查NET驱动版本

"""

from Common.base import context_util
from Common.base import entity
from Common.base.entity import Compare
from Common.base.entity import ResultFactory
from Common.base.entity import DeployException
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env

LOGGER = entity.create_logger(__file__)
MAPPING_KEY_NET_DRIVER_VER = "NET_driver_version"
LINE_BREAK_STR = "\n\n"

# 命令回显信息
ret_info = list()
# 错误信息
err_msgs = list()
# 提示信息
tip_info = list()


def get_table_net_driver_config_version(attribute_key):
    """
    从版本配套表中获取配置推荐版本
    :param attribute_key: 版本号的KEY
    :return: 配套版本号，配套的升级包地址
    """
    table_version = context_util.get_mapping_attribute(PY_JAVA_ENV, attribute_key)
    table_url = context_util.get_mapping_attribute_url(PY_JAVA_ENV, attribute_key)
    return table_version, table_url


def execute(task):
    """
    执行检查逻辑
    :return: 无
    """
    try:
        deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        # 首先查询版本配套信息再查询设备版本信息
        table_version, table_url = get_table_net_driver_config_version(MAPPING_KEY_NET_DRIVER_VER)
        LOGGER.info("current config driver version : {}.".format(table_version))
        if not table_version:
            err_msgs.append(entity.create_msg("check.net.driver.not.support"))
            return ResultFactory.create_pass(LINE_BREAK_STR.join(ret_info), __build_result_msgs())
        tip_info.append(entity.create_msg("check.net.driver.config.driver.version").format(table_version))
        driver_version = query_net_driver_version()
        LOGGER.info("query net driver version : {}.".format(driver_version))
        if not driver_version:
            err_msgs.append(entity.create_msg("check.net.driver.not.file"))
            deploy_node.putVersion(MAPPING_KEY_NET_DRIVER_VER, "--")
            return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), __build_result_msgs())
        tip_info.append(entity.create_msg("check.net.driver.current.version").format(driver_version))
        deploy_node.putVersion(MAPPING_KEY_NET_DRIVER_VER, driver_version)
        # 不匹配，版本过低
        if Compare.compare_digital_version(driver_version, table_version) < 0:
            LOGGER.error("net driver not match, driver version: {}..".format(driver_version))
            url_desc = entity.create_msg("check.net.driver.not.match")
            if table_url:
                url_desc = entity.build_url_error_msg(table_url, url_desc)
            err_msgs.append(url_desc)
            deploy_node.putResult(MAPPING_KEY_NET_DRIVER_VER, context_util.get_not_pass_key())
            return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), __build_result_msgs())
        return ResultFactory.create_pass(LINE_BREAK_STR.join(ret_info), __build_result_msgs())
    except DeployException as ex:
        LOGGER.error("query net version failed:{}.".format(ex.message))
        return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), ex.err_msg)


def __build_result_msgs():
    return entity.create_source_file_msg(PY_JAVA_ENV, "\n".join(tip_info + err_msgs))


def query_net_driver_version():
    """
    查询NET驱动版本号，回显示例：
    [root@FSM0 ~]# cat /opt/net/commit_file | grep NET_VERSION
    NET_VERSION : 888_0
    [root@FSM0 ~]#
    :return:
    """
    cmd = "cat /opt/net/commit_file | grep NET_VERSION"
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    ret_info.append(ssh_ret)
    if "No such file or directory" in ssh_ret :
        return ""
    for line in ssh_ret.splitlines():
        if line.startswith("NET_VERSION") :
            version_info = line.split(":")
            if len(version_info) < 2 :
                return ""
            return version_info[1].strip()
    return ""