#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2021/04/15
@file: check_nic_normalized_driver_version.py
@function:
"""
import re

from Common.base import context_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory, Compare
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
CX_1822_NORMALIZED_DRIVER_KEY = "1822_mlnx_normalized_driver_ver"
RDMA_NIC_DRIVER_KEY = "RDMA_nic_driver_version"
VERSION_PATTERN = r'[\d.]+-\d+-(pacific|atlantic|arm|x86)-(\d|[a-f]|[A-F])+-\d+'


def execute(task):
    return CheckNicVersion(task).check()


class CheckNicVersion(object):

    def __init__(self, task):
        self.task = task
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._ssh_rets = list()
        self._err_msgs = list()
        self._mapping_driver_url = ''
        self._cur_key = CX_1822_NORMALIZED_DRIVER_KEY
        self._mapping_msg = ""

    def check(self):
        if context_util.is_ocean_platform(PY_JAVA_ENV):
            self._cur_key = RDMA_NIC_DRIVER_KEY
        # 判断cur_key有没有在版本配套表中，若不存在，则直接通过。
        if not context_util.contain_need_check_key(
                PY_JAVA_ENV, [self._cur_key]):
            return ResultFactory.create_pass()
        try:
            self._get_mapping_msg()
            # 判断是否需要检查网卡驱动版本，若不需要，直接通过
            if not self._need_check_cx_nic():
                not_need_check_msgs = [entity.create_msg("no.need.check.nic.card"), self._mapping_msg]
                return ResultFactory.create_pass(self._ssh_rets,
                                                 not_need_check_msgs)
            res = self._check_driver_version()
            self._err_msgs.append(self._mapping_msg)
            if not res:
                self.deploy_node.putResult(context_util.get_version_key_enum().NIC_DRIVER.getKey(),
                                           context_util.get_not_pass_key())
                self._err_msgs.insert(0, entity.build_driver_tool_tips())
                return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
            return ResultFactory.create_pass(self._ssh_rets, self._err_msgs)
        except DeployException as de:
            self._logger.error(de.message)
            self._ssh_rets.append(de.origin_info)
            self._err_msgs.insert(0, self._mapping_msg)
            self._err_msgs.append(de.err_msg)
            if de.may_info_miss():
                self.task.openAutoRetry()
            return ResultFactory.create_not_pass(self._ssh_rets,
                                                 self._err_msgs)

    def _contain_check_key(self):
        check_keys = [self._cur_key]
        return context_util.contain_need_check_key(PY_JAVA_ENV, check_keys)

    def _get_mapping_msg(self):
        self._mapping_driver_ver = context_util. \
            get_mapping_attribute(PY_JAVA_ENV, self._cur_key)
        self._mapping_driver_url = context_util.get_mapping_attribute_url(PY_JAVA_ENV, self._cur_key)
        url_msg = entity.build_url_error_msg(self._mapping_driver_url,
                                             entity.create_msg("nic.match.version").format(self._mapping_driver_ver))
        self._mapping_msg = entity.create_source_file_msg(PY_JAVA_ENV, url_msg)

    def _contains_cx_nic_card(self):
        return self._contains_target_nic_card("Mellanox")

    def _contains_1822_nic_card(self):
        return self._contains_target_nic_card("1822")

    def _contains_target_nic_card(self, card_key):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(
            PY_JAVA_ENV, "lspci | grep {}".format(card_key))
        self._ssh_rets.append(ssh_ret)
        cmd_and_end_line_num = 2
        return len(ssh_ret.splitlines()) > cmd_and_end_line_num

    def _need_check_cx_nic(self):
        self.driver_ssh_ret = ssh_util.exec_ssh_cmd_disable_none(
            PY_JAVA_ENV, "rdma_ver")
        self._ssh_rets.append(self.driver_ssh_ret)
        # 判断返回值中是否存在“command not found”、“no such file”等信息
        if ssh_util.is_invalid_cmd(self.driver_ssh_ret):
            # 判断当前环境上是否存在cx网卡或1822网卡，若存在，抛出异常
            if self._contains_cx_nic_card() or self._contains_1822_nic_card():
                raise DeployException("invalid cmd", err_msg=entity.
                                      create_msg("cmd.not.invalid"))
            # 当前环境上不存在cx网卡或1822网卡，则无须检查它们的驱动版本
            return False
        return True

    def _check_driver_version(self):
        """
        检查驱动版本，分为两种驱动，cx驱动和1882cx归一驱动
        """

        def trans_normalized_driver_version_2_digital_version(driver_version):
            # 4.19-26-pacific-47f-0
            # 归一驱动中带有字符pacific、atlantic、arm和x86，需要替换掉再进行判断
            replace_version = driver_version.strip()
            for driver_key in context_util.NORMALIZED_DRIVER_KEYS:
                replace_version = replace_version.replace(driver_key, "0")
            # 倒数第二位为16进制需要转换成10进制
            versions = replace_version.split("-")
            versions[-2] = str(int(versions[-2], 16))
            return "-".join(versions)

        version_match = True
        # 版本在第二行,执行命令判断有长度大于2，所以不会越界
        # 不一定在第二行，需要兼容异常情况
        ver_info = ""
        version_pattern = re.compile(VERSION_PATTERN)
        for line in self.driver_ssh_ret.splitlines():
            result = version_pattern.search(line) if len(line) <= context_util.VERSION_MAX_LINE_LENGTH else None
            if result:
                ver_info = result.group()
                break
        if not context_util.contain_normalized_driver_key(ver_info) or Compare.compare_digital_version(
                trans_normalized_driver_version_2_digital_version(
                    ver_info),
                trans_normalized_driver_version_2_digital_version(
                    self._mapping_driver_ver)) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().NIC_DRIVER.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(entity.create_msg("nic.driver.current.version").format(ver_info))
        self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_DRIVER.getKey(), ver_info)
        return version_match
