#  coding=UTF-8
#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/06/08
@file: bmc_ip_config.py
@function:
"""
import os.path
import re

from Common.base import context_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util
from Common.service.acpi_spcr_service import AcpiSpcrService
from Business.adaptor.java_adaptor import get_tool_exception

ToolException = get_tool_exception()

PY_JAVA_ENV = py_java_env
EULER_OS_KEY = "euleros"
OS_VERSION_KEY = "OS_Image_Ver"
OS_VERSION_PASS_KEY = "OS_Image_Ver"

EULER_INFO_REG = r'[\d+.\d+.].*euleros[a-zA-Z.0-9_]+'


def find_euler_info(ssh_ret):
    version = re.search(EULER_INFO_REG, ssh_ret)
    if not version:
        return ""
    return version.group(0)


def execute(task):
    logger = entity.create_logger(__file__)
    os_mapping_ver = context_util.get_mapping_attribute(
        PY_JAVA_ENV, OS_VERSION_KEY)
    os_mapping_url = context_util.get_mapping_attribute_url(
        PY_JAVA_ENV, OS_VERSION_KEY)
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    err_msgs = []
    url_msg = entity.build_url_error_msg(os_mapping_url, entity.create_msg("os.match.version").format(os_mapping_ver))
    try:
        check_spcr(logger, deploy_node)

        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "cat /proc/version")
        if EULER_OS_KEY not in ssh_ret:
            err_msgs.append(entity.create_msg("os.version.not.support"))
            return ResultFactory.create_not_pass(ssh_ret, err_msgs)
        euler_info = find_euler_info(ssh_ret)
        logger.info("FusionStorage OS Version: {}".format(euler_info))
        os_version = parse_os_version(euler_info)
        os_little_version = parse_os_little_version(euler_info)
        deploy_node.putVersion(context_util.get_version_key_enum().OS.getKey(), euler_info)
        if os_version in os_mapping_ver and os_little_version \
                >= parse_os_little_version(os_mapping_ver):
            err_msgs.append(entity.build_url_error_msg(os_mapping_url, entity.create_msg(OS_VERSION_PASS_KEY)
                                                       .format(euler_info)))
            err_msgs.append(url_msg)
            err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, ""))
            return ResultFactory.create_pass(ssh_ret, err_msgs)
        err_msgs.append(entity.create_msg(OS_VERSION_KEY).format(euler_info))
        err_msgs.append(url_msg)
        err_msgs.insert(0, entity.build_os_tool_tips())
        deploy_node.putResult(context_util.get_version_key_enum().OS.getKey(), context_util.get_not_pass_key())
        err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, ""))
        return ResultFactory.create_not_pass(ssh_ret, err_msgs)
    except DeployException as e:
        logger.error(e.message)
        err_msgs.append(e.err_msg)
        if e.may_info_miss():
            task.openAutoRetry()
        err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, ""))
        return ResultFactory.create_not_pass(e.origin_info, err_msgs)


def check_spcr(logger, deploy_node):
    # 在评估场景，不应该通过配置变成并发起设备重启动作，此处只检查配置，开关没开则报错
    ip = deploy_node.getIp()
    spcr_service = AcpiSpcrService(PY_JAVA_ENV)
    if not spcr_service.need_open_spcr():
        logger.info("not need open spcr for :{}".format(ip))
        return
    # 在spcr没有开启的情况下，尝试获取连接，获取连接成功则不影响后续执行，若获取连接失败，则提示串口未开启，联系技术工程师
    try:
        ssh_util.get_ssh(PY_JAVA_ENV)
        logger.info("sprc not open but can connect to os for :{}".format(ip))
        return
    except (Exception, ToolException) as e:
        logger.error(e)
    raise DeployException("try ssh failed", origin_info=entity.create_msg("sol.need.manu.confirm.reboot"),
                          err_msg=entity.create_msg("switch.os.failed"))


def parse_os_little_version(version_info):
    match = re.findall(r".h(\d{3,}).", version_info)
    if match:
        return int(match[0])
    raise DeployException(
        "parsed os little version failed",
        err_code=DeployException.ErrCode.MAY_INFO_MISS)


def parse_os_version(version_info):
    match = re.findall(r".(eulerosv\d+r\d+).", version_info)
    if match:
        return match[0]
    raise DeployException(
        "parsed os version failed",
        err_code=DeployException.ErrCode.MAY_INFO_MISS)
