#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

"""
@time: 2023/12/25
@file: check_raid_card_config_version.py
@function:
"""

from Common.base import context_util
from Common.base import entity
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import Compare
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env

LOGGER = entity.create_logger(__file__)

CONFIG_MAPPING_KEY = "{}_RAID_Config_Ver"

LINE_BREAK_STR = "\n\n"

# 错误信息
err_msgs = list()
# 不涉及
not_supports = list()
# 原始信息
res_info = list()
# 命令回显
ret_info = list()


def execute(task):
    try:
        check()
        if err_msgs:
            res_info.append("\n")
            result_msgs = res_info + err_msgs
            return ResultFactory.create_not_pass(LINE_BREAK_STR.join(ret_info), "\n".join(result_msgs))
        if not_supports:
            not_supports.extend(res_info)
            return ResultFactory.create_pass(LINE_BREAK_STR.join(ret_info), "\n".join(not_supports))
        return ResultFactory.create_pass(LINE_BREAK_STR.join(ret_info), "\n".join(res_info))
    except DeployException as ex:
        return ResultFactory.create_not_pass(ex.origin_info, ex.err_msg)


def get_device_raid_type():
    return CONFIG_MAPPING_KEY.format(query_system_arch())


def is_need_install_raid_config():
    """  查询设备是否支持安装RAID卡配置工具   """
    cmd = 'lspci | grep -E "LSI|3758|3858" | grep -v "3008" | wc -l'
    try:
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
        ret_info.append(ssh_ret)
        if int(ssh_ret.splitlines()[-2]) == 0:
            # 不涉及，通过
            not_supports.append(entity.create_msg("check.raid.config.not.support"))
            return False
        return True
    except DeployException as e:
        err_msgs.append(e.err_msg)
        LOGGER.error(e.message)
        return False


def query_system_arch():
    """  查询设备系统架构   """
    cmd = 'arch'
    try:
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
        ret_info.append(ssh_ret)
        return "X86" if 'x86' in ssh_ret.splitlines()[-2].lower() else "ARM"
    except DeployException as e:
        err_msgs.append(e.err_msg)
        LOGGER.error(e.message)
        return ""


def get_table_raid_config_version(attribute_key):
    """  从版本配套表中获取RAID配置推荐版本  """
    table_version = get_mapping_attribute(PY_JAVA_ENV, attribute_key)
    if not table_version:
        not_supports.append(entity.create_msg("check.raid.config.not.support"))
        return "", ""
    res_info.append(entity.create_msg("check.raid.config.table.raid.config.version").format(table_version))
    return table_version, get_mapping_attribute_url(PY_JAVA_ENV, attribute_key)


def get_device_raid_config_version():
    """   从设备上查询RAID配置版本  """
    try:
        cmd = 'rpm -qi RAID-CONFIG-Euler | grep -i "version"'
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
        ret_info.append(ssh_ret)
        version_result = ssh_ret.splitlines()[-2]
        if 'version' in version_result.lower() and len(version_result.split(":")) == 2:
            current_version = version_result.split(":")[1].strip()
            res_info.append(
                entity.create_msg("check.raid.config.current.raid.config.version").format(current_version))
            return current_version
        err_msgs.append(entity.create_msg("check.raid.config.not.install"))
        return ""
    except DeployException as e:
        err_msgs.append(e.err_msg)
        LOGGER.error(e.message)
        return ""


def check():
    if not is_need_install_raid_config():
        return
    raid_card_type = get_device_raid_type()
    table_version, table_url = get_table_raid_config_version(raid_card_type)
    if not table_version:
        return
    current_version = get_device_raid_config_version()
    if not current_version:
        return
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    deploy_node.putVersion(context_util.get_version_key_enum().RAID_CONFIG.getKey(), current_version)
    # 版本比较，大于返回1，等于返回0，小于返回-1
    if Compare.compare_digital_version(current_version, table_version) != -1:
        return
    deploy_node.putResult(context_util.get_version_key_enum().RAID_CONFIG.getKey(), context_util.get_not_pass_key())
    if table_url:
        url_desc = entity.build_url_error_msg(table_url, entity.create_msg("check.raid.config.not.match"))
    else:
        url_desc = entity.create_msg("check.raid.config.not.match")
    err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, url_desc))
