#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/08/07
@file: check_raid_card_version.py
@function:
"""
import re

from Common.base import context_util
from Common.base import entity
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory, Compare
from Common.protocol import ssh_util
from Common.service import check_main_board_uids_service

PY_JAVA_ENV = py_java_env

LOGGER = entity.create_logger(__file__)

STORCLI_TOOL = "/opt/MegaRAID/storcli/storcli64"

STORCLI_TOOL_ARM = "/opt/MegaRAID/storcli/storcli64_arm"


def execute(task):
    check_items = (AvagoVendorRaidCardVersionCheckItem,
                   UnionVendorRaidCardVersionCheckItem,
                   MsccVendorRaidCardVersionCheckItem)
    match_msgs = list()
    ssh_rets = list()
    err_msgs = list()
    res_list = []
    deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
    try:
        for item in check_items:
            res_list.append(item(task, ssh_rets, err_msgs, match_msgs, deploy_node).check())
    except DeployException as de:
        LOGGER.error(de.message)
        ssh_rets.append(de.origin_info)
        err_msgs.append(de.err_msg)
        err_msgs.insert(0, entity.create_source_file_msg(PY_JAVA_ENV, "; ".join(match_msgs) + "\n"))
        if de.may_info_miss():
            task.openAutoRetry()
        return ResultFactory.create_not_pass(ssh_rets, err_msgs)
    err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, "; ".join(match_msgs) + "\n"))
    if not all(res_list):
        err_msgs.insert(0, entity.build_driver_tool_tips())
        return ResultFactory.create_not_pass(ssh_rets, err_msgs)
    return ResultFactory.create_pass(ssh_rets, err_msgs)


def append_mapping_url(fw_url, driver_url):
    mapping_url = []
    if driver_url:
        mapping_url.append(driver_url)
    if fw_url:
        mapping_url.append(fw_url)
    return mapping_url


class MsccVendorRaidCardVersionCheckItem(object):
    FW_MAPPING_KEY = "MSCC_RAID_{}_FW_Ver"
    DRIVER_MAPPING_KEY = "MSCC_RAID_{}_Driver_Ver"
    MSCC_CARD_REG = re.compile(r"SmartRAID (\d{4})")
    VERSION_REG = re.compile(r"([.\d-]+)")

    def __init__(self, task, ssh_rets, err_msgs, match_msgs, deploy_node):
        self.task = task
        self.deploy_node = deploy_node
        self._ssh_rets = ssh_rets
        self._err_msgs = err_msgs
        self._match_msgs = match_msgs
        self._fw_mapping_ver = None
        self._driver_mapping_ver = None
        self._fw_mapping_url = None
        self._driver_mapping_url = None
        self._raid_card_type = None

    def check(self):
        if self._obtain_match_versions():
            return self._check_version()
        return True

    def _obtain_match_versions(self):
        raid_card_type = self._get_raid_card_type()
        LOGGER.info("Mscc vendor card type: {}".format(raid_card_type))
        if not raid_card_type:
            return False
        self._raid_card_type = raid_card_type
        self.init_match_version(raid_card_type)
        no_support_msg = entity.create_msg("not.support")
        fw_mapping_msg = self._fw_mapping_ver if self._fw_mapping_ver \
            else no_support_msg
        driver_mapping_msg = self._driver_mapping_ver if \
            self._driver_mapping_ver else no_support_msg
        url_msg = entity.build_url_error_msg(
            "\n".join(append_mapping_url(self._fw_mapping_url, self._driver_mapping_url)),
            entity.create_msg("raid.card.match.version").format(
                raid_card_type, driver_mapping_msg, fw_mapping_msg))
        self._match_msgs.append(url_msg)
        self.set_mscc_vendor_raid_not_support(no_support_msg)
        return self._fw_mapping_ver or self._driver_mapping_ver

    def set_mscc_vendor_raid_not_support(self, no_support_msg):
        if not self._driver_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), no_support_msg)
        if not self._fw_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), no_support_msg)

    def init_match_version(self, raid_card_type):
        self._fw_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, self.FW_MAPPING_KEY.format(raid_card_type))
        self._driver_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, self.DRIVER_MAPPING_KEY.format(raid_card_type))
        self._fw_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, self.FW_MAPPING_KEY.format(raid_card_type))
        self._driver_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, self.DRIVER_MAPPING_KEY.format(raid_card_type))

    def _get_raid_card_type(self):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(
            PY_JAVA_ENV, "lspci -m | grep -i Adaptec")
        self._ssh_rets.append(ssh_ret)
        match = self.MSCC_CARD_REG.search(ssh_ret)
        if match:
            return match.groups()[0]

    def _get_card_version(self):
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "arcconf getversion")
        self._ssh_rets.append(ssh_ret)
        if "Command completed successfully" not in ssh_ret:
            raise DeployException(
                "get raid card version failed",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        fw_version = None
        driver_version = None
        for line in ssh_ret.splitlines():
            if line.startswith("Firmware"):
                fw_version = self._search_version(line)
            if line.startswith("Driver"):
                driver_version = self._search_version(line)
        LOGGER.info("fw_version: {}, driver_version: {}".format(
            fw_version, driver_version))
        if fw_version and driver_version:
            return fw_version, driver_version
        raise DeployException(
            "get raid card version failed",
            err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _search_version(self, line):
        match = self.VERSION_REG.search(line)
        if match:
            return match.groups()[0]

    def _check_version(self):
        version_match = True
        fw_version, driver_version = self._get_card_version()
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), fw_version)
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), driver_version)
        if entity.Compare.compare_digital_version(
                fw_version, self._fw_mapping_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_FW.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(
            entity.create_msg("raid.card.fw.current.version").format(self._raid_card_type, fw_version))
        if entity.Compare.compare_digital_version(
                driver_version, self._driver_mapping_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_DRIVER.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(
            entity.create_msg("raid.card.driver.current.version").format(self._raid_card_type, driver_version))
        return version_match


class AvagoVendorRaidCardVersionCheckItem(object):
    FW_MAPPING_KEY = "AVAGO_RAID_{}_{}_FW_Ver"
    DRIVER_MAPPING_KEY = "AVAGO_RAID_{}_{}_Driver_Ver"
    FW_MAPPING_KEY_V6 = "AVAGO_RAID_{}_{}_FW_Ver_V6"
    DRIVER_MAPPING_KEY_V6 = "AVAGO_RAID_{}_{}_Driver_Ver_V6"
    # 扣卡
    BUCKLE_CARD_KEY = "BC"
    # 标卡
    LABEL_CARD_KEY = "LC"

    def __init__(self, task, ssh_rets, err_msgs, match_msgs, deploy_node):
        self.task = task
        self.deploy_node = deploy_node
        self._ssh_rets = ssh_rets
        self._mapping_versions_dict = dict()
        self._err_msgs = err_msgs
        self._match_msgs = match_msgs
        self.login_info = context_util.get_login_info(PY_JAVA_ENV)
        self.is_v6_type = check_main_board_uids_service.is_v6_supported_main_board(self.login_info, LOGGER)
        self._check_method = self.init_check_method()
        self._fw_mapping_ver = ''
        self._driver_mapping_ver = ''
        self._fw_mapping_url = ''
        self._driver_mapping_url = ''
        self._raid_card_type = ''

    def check(self):
        raid_card_type, ssh_ret = ssh_util.get_raid_card_type(PY_JAVA_ENV)
        self._ssh_rets.append(ssh_ret)
        if not self._obtain_match_versions(raid_card_type):
            return True
        default_check_method = [self._check_driver_version_with_storcli_tool, self._check_fw_version_with_storcli_tool]
        res = []
        for check_method in self._check_method.get(raid_card_type, default_check_method):
            res.append(check_method())
        return all(res)

    def init_check_method(self):
        return {'3008': [self._check_3008_card_driver_version, self._check_3008_card_fw_version],
                'SP686C-M-40i/SP686C-M-16i': [self._check_1880_card_driver_version, self._check_1880_card_fw_version],
                'SP186-M-8i': [self._check_1880_card_driver_version, self._check_1880_card_fw_version]}

    def _obtain_match_versions(self, raid_card_type):
        if not raid_card_type:
            return False
        self._raid_card_type = raid_card_type
        detail_key = self._get_detail_card_key(raid_card_type)
        current_fw_key = self.FW_MAPPING_KEY
        current_driver_key = self.DRIVER_MAPPING_KEY
        if self.is_v6_type:
            current_fw_key = self.FW_MAPPING_KEY_V6
            current_driver_key = self.DRIVER_MAPPING_KEY_V6
        self.init_match_version(current_driver_key, current_fw_key, detail_key, raid_card_type)
        self._obtain_match_msg(raid_card_type)

        return self._fw_mapping_ver or self._driver_mapping_ver

    def init_match_version(self, current_driver_key, current_fw_key, detail_key, raid_card_type):
        self._fw_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, current_fw_key.format(
                raid_card_type, detail_key))
        self._driver_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, current_driver_key.format(
                raid_card_type, detail_key))
        self._fw_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, current_fw_key.format(
                raid_card_type, detail_key))
        self._driver_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, current_driver_key.format(
                raid_card_type, detail_key))

    def _get_detail_card_key(self, raid_card_type):
        if raid_card_type == "3008":
            return self.BUCKLE_CARD_KEY
        if raid_card_type in ('SP686C-M-40i/SP686C-M-16i', 'SP186-M-8i'):
            return self.LABEL_CARD_KEY
        ssh_ret = self.exec_storcli_tool_cmd('/c0 show | grep -i "Product Name"')
        self._ssh_rets.append(ssh_ret)
        if re.findall(r"= SAS\d+$", ssh_ret.splitlines()[-2].strip()):
            return self.BUCKLE_CARD_KEY
        return self.LABEL_CARD_KEY

    def _obtain_match_msg(self, raid_card_type):
        no_support_msg = entity.create_msg("not.support")
        fw_mapping_ver = self._fw_mapping_ver if self._fw_mapping_ver \
            else no_support_msg
        driver_mapping_ver = self._driver_mapping_ver if \
            self._driver_mapping_ver else no_support_msg
        url_msg = entity.build_url_error_msg(
            "\n".join(append_mapping_url(self._fw_mapping_url, self._driver_mapping_url)),
            entity.create_msg(
                "raid.card.match.version").format(
                raid_card_type, driver_mapping_ver, fw_mapping_ver))
        self.set_avago_vendor_raid_not_support(no_support_msg)
        self._match_msgs.append(url_msg)

    def set_avago_vendor_raid_not_support(self, no_support_msg):
        if not self._driver_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), no_support_msg)
        if not self._fw_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), no_support_msg)

    def _check_raid_version_by_type(self, ssh_ret, raid_card_type):
        version = re.findall(r"(\d+.\d+.\d+.\d+)", ssh_ret)
        if not version:
            raise DeployException("match {} driver version failed".format(raid_card_type),
                                  origin_info=ssh_ret,
                                  err_code=DeployException.ErrCode.MAY_INFO_MISS)
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), version[0])
        self._err_msgs.append(entity.create_msg("raid.card.driver.current.version")
                              .format(self._raid_card_type, version[0]))
        return self._compare_version_with_mapping_key(version[0], self._driver_mapping_ver)

    def _check_3008_card_driver_version(self):
        if not self._driver_mapping_ver:
            return True
        ssh_ret = self._execute_3008_raid_card_version_cmd()
        self._ssh_rets.append(ssh_ret)
        return self._check_raid_version_by_type(ssh_ret, '3008')

    def _execute_3008_raid_card_version_cmd(self):
        try:
            return ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "modinfo mpt3sas | grep -iw version")
        except DeployException:
            LOGGER.error("execute modinfo mpt3sasa failed, use rpm cmd try again")
            return ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "rpm -qa|grep -i mpt3sas")

    def _check_3008_card_fw_version(self):
        if not self._fw_mapping_ver:
            return True
        tool_name = "sas3flash"
        version_match = True
        if context_util.get_platform_id(PY_JAVA_ENV) != "x86_64":
            tool_name = "sas3flash_arm"
        cmd = './tools/{} -list | grep -i "Firmware Version"'.format(tool_name)
        ssh_ret = ssh_util.exec_ssh_cmd_disable_none(PY_JAVA_ENV, cmd)
        self._ssh_rets.append(ssh_ret)
        if ssh_util.is_invalid_cmd(ssh_ret):
            self.task.openManualIgnored()
            self._err_msgs.append(entity.create_msg(
                "need.install.3008.driver.tool"))
            return False
        version = ssh_ret.splitlines()[-2].split(":")[-1].strip()
        if Compare.compare_digital_version(version, self._fw_mapping_ver) < 0:
            version_match = False
        self._err_msgs.append(entity.create_msg("raid.card.fw.current.version").format(self._raid_card_type, version))
        return version_match

    def _check_fw_version_with_storcli_tool(self):
        is_mapping, version = self._is_mapping_version(self._fw_mapping_ver,
                                                       "FW")
        self._err_msgs.append(entity.create_msg("raid.card.fw.current.version").format(self._raid_card_type, version))
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), version)
        if not is_mapping:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_FW.getKey(),
                                       context_util.get_not_pass_key())
        return is_mapping

    def _check_driver_version_with_storcli_tool(self):
        is_mapping, version = self._is_mapping_version(
            self._driver_mapping_ver, "Driver")
        self._err_msgs.append(entity.create_msg("raid.card.driver.current.version").format(self._raid_card_type,
                                                                                           version))
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), version)
        if not is_mapping:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_DRIVER.getKey(),
                                       context_util.get_not_pass_key())
        return is_mapping

    def _is_mapping_version(self, mapping_version, version_key):
        if not mapping_version:
            return True, ""
        ssh_ret = self.exec_storcli_tool_cmd('/c0 show | grep -i "{} Version"'.format(version_key))
        version = ssh_ret.splitlines()[-2].split("=")[-1].strip()
        LOGGER.info("version: {}".format(version))
        return Compare.compare_digital_version(
            version, mapping_version) >= 0, version

    def exec_storcli_tool_cmd(self, cmd_param):
        cmd = "{} {}".format(STORCLI_TOOL, cmd_param)
        try:
            ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
        except DeployException as exception:
            ssh_ret = self.run_arm_util(cmd_param, exception)
        self._ssh_rets.append(ssh_ret)
        return ssh_ret

    def run_arm_util(self, cmd_param, exception):
        self._ssh_rets.append(exception.origin_info)
        cmd = "{} {}".format(STORCLI_TOOL_ARM, cmd_param)
        try:
            ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
        except DeployException as exception:
            self.update_error_msg(exception)
            raise exception
        return ssh_ret

    def update_error_msg(self, exception):
        if exception.err_msg == entity.create_msg("cmd.not.invalid"):
            exception.err_msg = entity.create_msg("cmd.not.invalid.storcli")

    def _check_1880_card_driver_version(self):
        if not self._driver_mapping_ver:
            return True
        cmd = 'cat /sys/module/hiraid/version'
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
        self._ssh_rets.append(ssh_ret)
        return self._check_raid_version_by_type(ssh_ret, '1880')

    def _check_1880_card_fw_version(self):
        if not self._fw_mapping_ver:
            return True
        controller_ids = self._get_raid_id_list()
        cmd = 'hiraidadm c{} show status'
        # 事实上存储设备只会有一张RAID卡
        res = []
        for controller_id in controller_ids:
            ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd.format(controller_id))
            self._ssh_rets.append(ssh_ret)
            version = self._get_version(ssh_ret)
            res.append(self._compare_version_with_mapping_key(version, self._fw_mapping_ver))
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), version)
            self._err_msgs.append(entity.create_msg("raid.card.fw.current.version").format(
                self._raid_card_type, version))
        return all(res)

    def _compare_version_with_mapping_key(self, version, mapping_ver):
        if Compare.compare_digital_version(version, mapping_ver) >= 0:
            return True
        return False

    def _get_raid_id_list(self):
        cmd = 'hiraidadm show allctrl'
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
        self._ssh_rets.append(ssh_ret)
        if 'Status = Success' not in ssh_ret:
            raise DeployException('query 1880 controller id failed', origin_info=ssh_ret)
        return list(map(lambda line: line.split("|")[1].strip(),
                        re.findall(r"Controller Id[\s]+\|[\s]+[\d]+", ssh_ret)))

    def _get_version(self, ssh_ret):
        version = re.findall(r"FW\s+Version0\s+\|\s+\d+.\d+.\d+.\d+", ssh_ret)
        if not version:
            raise DeployException('query 1880 fw version failed', origin_info=ssh_ret)
        return version[0].split("|")[1].strip()


class UnionVendorRaidCardVersionCheckItem(object):
    FW_MAPPING_KEY = "UNION_RAID_{}_FW_Ver"
    DRIVER_MAPPING_KEY = "UNION_RAID_{}_Driver_Ver"
    UNION_CARD_REG = re.compile(r"Smart Storage.*Union Memory")
    VERSION_REG = re.compile(r"([.\d-]+)")

    def __init__(self, task, ssh_rets, err_msgs, match_msgs, deploy_node):
        self.task = task
        self.deploy_node = deploy_node
        self._ssh_rets = ssh_rets
        self._err_msgs = err_msgs
        self._match_msgs = match_msgs
        self._fw_mapping_ver = None
        self._driver_mapping_ver = None
        self._fw_mapping_url = None
        self._driver_mapping_url = None
        self._raid_card_type = None

    def check(self):
        if self._obtain_match_versions():
            return self._check_version()
        return True

    def _obtain_match_versions(self):
        raid_card_type = self._get_raid_card_type()
        LOGGER.info("UNIOS vendor card type: {}".format(raid_card_type))
        if not raid_card_type:
            return False
        self._raid_card_type = raid_card_type
        self.init_match_version(raid_card_type)
        no_support_msg = entity.create_msg("not.support")
        fw_mapping_msg = self._fw_mapping_ver if self._fw_mapping_ver else no_support_msg
        driver_mapping_msg = self._driver_mapping_ver if self._driver_mapping_ver else no_support_msg
        url_msg = entity.build_url_error_msg(
            "\n".join(append_mapping_url(self._fw_mapping_url, self._driver_mapping_url)),
            entity.create_msg("raid.card.match.version")
            .format(raid_card_type, driver_mapping_msg, fw_mapping_msg))
        self._match_msgs.append(url_msg)
        self.set_union_vendor_not_support(no_support_msg)
        return self._fw_mapping_ver or self._driver_mapping_ver

    def set_union_vendor_not_support(self, no_support_msg):
        if not self._driver_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), no_support_msg)
        if not self._fw_mapping_ver:
            self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), no_support_msg)

    def init_match_version(self, raid_card_type):
        self._fw_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, self.FW_MAPPING_KEY.format(raid_card_type))
        self._driver_mapping_ver = get_mapping_attribute(
            PY_JAVA_ENV, self.DRIVER_MAPPING_KEY.format(raid_card_type))
        self._fw_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, self.FW_MAPPING_KEY.format(raid_card_type))
        self._driver_mapping_url = get_mapping_attribute_url(
            PY_JAVA_ENV, self.DRIVER_MAPPING_KEY.format(raid_card_type))

    def _get_raid_card_type(self):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, "lspci -m | grep -i Adaptec")
        self._ssh_rets.append(ssh_ret)
        match = self.UNION_CARD_REG.search(ssh_ret)
        # 当前该厂商raid卡只涉及3152，后续依据演进规则，此处需要对应修改
        if match:
            return "FBGF-RAD-R1-S1"
        return ""

    def _get_card_version(self):
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, "arcconf getversion")
        self._ssh_rets.append(ssh_ret)
        if "Command completed successfully" not in ssh_ret:
            raise DeployException("get raid card version failed", err_code=DeployException.ErrCode.MAY_INFO_MISS)
        fw_version = None
        driver_version = None
        for line in ssh_ret.splitlines():
            if line.startswith("Firmware"):
                fw_version = self._search_version(line)
            if line.startswith("Driver"):
                driver_version = self._search_version(line)
        LOGGER.info("fw_version: {}, driver_version: {}".format(fw_version, driver_version))
        if fw_version and driver_version:
            return fw_version, driver_version
        raise DeployException("get raid card version failed", err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _search_version(self, line):
        match = self.VERSION_REG.search(line)
        if match:
            return match.groups()[0]
        return ""

    def _check_version(self):
        version_match = True

        fw_version, driver_version = self._get_card_version()
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_FW.getKey(), fw_version)
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_DRIVER.getKey(), driver_version)
        if entity.Compare.compare_digital_version(fw_version, self._fw_mapping_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_FW.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(
            entity.create_msg("raid.card.fw.current.version").format(self._raid_card_type, fw_version))
        if entity.Compare.compare_digital_version(driver_version, self._driver_mapping_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().RAID_DRIVER.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(entity.create_msg("raid.card.driver.current.version").format(self._raid_card_type,
                                                                                           driver_version))
        return version_match
