#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

"""
@time: 2023/05/25
@file: check_whether_os_has_installed.py
@function:
"""
import time

from Business.adaptor import java_adaptor
from Common.base import entity, context_util
from Common.base.entity import ResultFactory
from Common.protocol.redfish import redfish_util
from Common.protocol.ssh_util import get_ssh, SOL_TO_OS_CMD, OFF_SOL_CMD, SOL_TO_OS_STATE_DISABLED, \
    SOL_TO_OS_STATE_ENABLED, SOL_CMD_TIME_OUT, get_ssh_try_new_password

PY_JAVA_ENV = py_java_env
IPMIConnection = java_adaptor.get_ipmi_connection()
JException = java_adaptor.get_j_exception()
logger = entity.create_logger(__file__)
login_info = context_util.get_login_info(PY_JAVA_ENV)
origin_infos = []
err_msgs = []


def execute(task):
    entity.AutoBrushProgressTask(task, 5400).start()
    context = task.getJythonContext()
    return check_os_installation_status_cyclically(context)


def check_os_installation_status(context):
    """
    根据回显是否以“login:“结束操作系统是否安装完成
    :param context: 上下文
    :return:返回True代表安装完成
    """
    try:
        if not check_image_is_connect():
            err_msgs.append(entity.create_msg("image.is.not.connected"))
            return False
        # 开局部署创建的ssh连接，结束符已经制定包括"]"
        # OS灌装时不能发送"3"中断连接，可能会影响选择镜像安装模式，因此不能使用已有的ssh_util公共方法
        ssh = get_ssh(context, switch_root=False)
        return SwitchToSshService(context, ssh).switch_to_ssh()
    finally:
        ssh_release(context)


def check_os_installation_status_cyclically(context):
    """
    循环检查操作系统是否安装完成，每3分钟检查一次，每次检查耗时1分钟左右，总超时时间设为2小时
    :param context: 上下文
    :return: 检查通过/不通过
    """
    for _ in range(30):
        try:
            if check_os_installation_status(context):
                return ResultFactory.create_pass(origin_infos, entity.create_msg("progress.finish"))
            if err_msgs:
                return ResultFactory.create_not_pass(origin_infos, err_msgs)
            time.sleep(3 * 60)
        except JException:
            time.sleep(3 * 60)
            logger.info("SSH connection is interrupted abnormally, reconnect the connection.")
    return ResultFactory.create_not_pass(origin_infos, entity.create_msg("check.os.installation.status.timeout"))


def check_image_is_connect():
    virtual_media = redfish_util.current_virtual_media(login_info, logger)
    return virtual_media.get("Inserted")


def ssh_release(context):
    dev_node = context_util.get_dev_node(context)
    platform_id = context_util.get_platform_id(context)
    if platform_id == "TaiShan V1":
        java_adaptor.get_connection_manager().releaseConnection(dev_node, platform_id)
    else:
        java_adaptor.get_ssh_connection_manager().releaseConnection(context_util.get_dev_node(context))


class SwitchToSshService(object):
    def __init__(self, context, ssh):
        self.context = context
        self._ssh = ssh

    def switch_to_ssh(self):
        if self.get_switch_to_ssh_ret().strip().endswith("login:"):
            return True
        self.re_login_bmc()
        self.off_and_on_sol()
        return self.get_switch_to_ssh_ret().strip().endswith("login:")

    def get_switch_to_ssh_ret(self):
        if not isinstance(self._ssh, IPMIConnection):
            ssh_ret = self._ssh.execCmdWithTimout(SOL_TO_OS_CMD, 10, None, False)
            origin_infos.append(ssh_ret)
            ssh_ret = self.handle_private_model(ssh_ret)
            # SOL切换命令执行结果不是超时且没有回显成功，则认为切换失败，直接报错。
            if ssh_ret != "TOOLKIT_SEND_CMD_TIME_OUT" and "successfully" not in ssh_ret:
                err_msgs.append(entity.create_msg("switch.os.failed"))
                return ssh_ret
        ssh_ret = self._ssh.execCmdWithTimout("\n", 10, None, False).strip()
        origin_infos.append(ssh_ret)
        return ssh_ret

    def handle_private_model(self, ssh_ret):
        if "Current SOL session is in private mode" in ssh_ret:
            origin_infos.append(self._ssh.execCmdWithTimout(OFF_SOL_CMD, 10, None, False))
            tmp_ssh_ret = self._ssh.execCmdWithTimout(SOL_TO_OS_CMD, 10, None, False)
            origin_infos.append(tmp_ssh_ret)
            return tmp_ssh_ret
        return ssh_ret

    def off_and_on_sol(self):
        # 关闭SOL
        ssh_ret = self._ssh.execCmdWithTimout(SOL_TO_OS_STATE_DISABLED, SOL_CMD_TIME_OUT, None, False).strip()
        if "successfully" not in ssh_ret.strip().lower():
            time.sleep(5)
            self._ssh.execCmdWithTimout(SOL_TO_OS_STATE_DISABLED, SOL_CMD_TIME_OUT, None, False).strip()
        time.sleep(10)
        # 开启SOL
        ssh_ret = self._ssh.execCmdWithTimout(SOL_TO_OS_STATE_ENABLED, SOL_CMD_TIME_OUT, None, False).strip()
        if "successfully" not in ssh_ret.strip().lower():
            time.sleep(5)
            self._ssh.execCmdWithTimout(SOL_TO_OS_STATE_ENABLED, SOL_CMD_TIME_OUT, None, False).strip()
        time.sleep(60)

    def re_login_bmc(self):
        dev_node = context_util.get_dev_node(self.context)
        platform_id = context_util.get_platform_id(self.context)
        ssh_release(self.context)
        self._ssh = get_ssh_try_new_password(dev_node, platform_id, self.context)
