# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.

from Common.base import context_util, entity
from Common.base.entity import DeployException, ResultFactory, Compare
from Common.protocol.redfish.entity.resource_client import SystemsResourceClient
from Common.protocol.redfish import redfish_util

# OceanStor A310,OceanStor Pacific 9950 要求最低的iBMC版本号，才支持修改BIOS密码
MIN_BMC_VERSION = "3.15.05.08"
# OceanStor A310(0x020dc300), OceanStor Pacific 9950(0x020d1b00) 的设备UID，用于检查最低版本号
NEED_CHECK_BMC_VERSION_DEV_UID = ["0x020dc300", "0x020d1b00"]


def execute(task):
    config_bios_pwd = ConfigBiosPassword(task.getJythonContext())
    if not config_bios_pwd.check_bmc_version_valid():
        err_msg = entity.create_msg("check.ibmc.version.lower.config.bios.pwd").format(
            MIN_BMC_VERSION)
        return ResultFactory.create_not_pass(origin_info=config_bios_pwd.origin_info, err_msg=err_msg)
    return config_bios_pwd.post_info(task.getParam())


def build_param(param):
    bios_infos = param.getBiosInfos()
    bios_params = []
    for bios_info in bios_infos:
        bios_params.append(
            {
                "OldPassword": bios_info.getOldPassword(),
                "NewPassword": bios_info.getNewPassword(),
                "PasswordName": bios_info.getPasswordName()
            }
        )
    return bios_params


class ConfigBiosPassword(object):
    def __init__(self, context):
        self._context = context
        self._logger = entity.create_logger(__file__)
        self._login_info = context_util.get_login_info(context)
        self._client = SystemsResourceClient(self._login_info, self._logger)
        self.origin_info = list()

    def post_info(self, param):
        try:
            for bios_param in build_param(param):
                self.origin_info.append(str(self._client.patch_bios_password(bios_param)))
            return ResultFactory.create_pass(entity.create_msg("config.bmc.password.success"))
        except DeployException as exception:
            self.origin_info.append(exception.origin_info)
            return ResultFactory.create_not_pass(origin_info=self.origin_info, err_msg=exception.err_msg)

    def check_bmc_version_valid(self):
        # 检查设备型号
        dev_uid, resource = redfish_util.get_product_unique_id(self._login_info, self._logger)
        self.origin_info.append(resource)
        if dev_uid not in NEED_CHECK_BMC_VERSION_DEV_UID:
            self._logger.info("not need to check bmc version, uid : {}.".format(dev_uid))
            return True
        # 检查A310和9950的设备iBMC版本号
        active_bmc_ver, resource = redfish_util.get_firmware_inventory_version(
            self._login_info, "ActiveBMC", self._logger)
        self.origin_info.append(resource)
        self._logger.info("check bmc version, uid:{}, version:{}, min version : {}.".format(dev_uid, active_bmc_ver,
                                                                                            MIN_BMC_VERSION))
        return Compare.compare_digital_version(active_bmc_ver, MIN_BMC_VERSION) >= 0
