#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2024. All rights reserved.

"""
@time: 2021/03/10
@file: config_bmc_ip_by_input_ip.py
@function:
"""
import time
import traceback

from Common.base import context_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.util import modify_bmc_pwd_util
from Common.protocol.redfish import redfish_util
from Business.adaptor.java_adaptor import get_ip_util

from Common.protocol.redfish.entity.http import HttpClient

PY_JAVA_ENV = py_java_env
LOGGER = entity.create_logger(__file__)


def check_target_ip_reachable(target_ip):
    """
    检查目标ip连通性，bmc ip设置后，存在目标ip不会及时生效的问题
    配置完成后，循环五次，间隔5秒，只要ip能连通则返回true
    或最终返回false，认为连通性校验失败
    :param target_ip: 目标ip
    :return:
    """
    for i in range(5):
        LOGGER.info("current ip :{} reachable test retry :{}".format(target_ip, i))
        if get_ip_util().isReachable(target_ip):
            LOGGER.info("current ip :{} reachable test success :{}".format(target_ip, i))
            return True
        time.sleep(5)
    return False


def execute(task):
    login_info = context_util.get_login_info(PY_JAVA_ENV)
    login_info.ip = task.getNowIp()

    # 检查一下当前BMC IP地址是否可达，用于拦截本地多网卡IPv6配置导致的第一次访问Redfish接口不通的问题。
    if not check_target_ip_reachable(task.getNowIp()):
        LOGGER.error("current ip :{} not reachable.".format(task.getNowIp()))
        return ResultFactory.create_not_pass("", entity.create_msg("bmc.source.ip.not.reached").format(task.getNowIp()))

    msgs = list()
    origin_infos = list()
    had_modify_default_pwd = False
    try:
        had_modify_default_pwd = redfish_util.modify_bmc_default_password(PY_JAVA_ENV, LOGGER, ip=login_info.ip)
    except DeployException as de:
        LOGGER.error(de.message)
        LOGGER.error(str(traceback.format_exc()))
        # 修改默认密码失败时，尝试释放缓存链接
        HttpClient.release_session(login_info, LOGGER)
        return ResultFactory.create_not_pass(de.origin_info, de.err_msg)

    config_success = True
    if not had_modify_default_pwd:
        dev_type = context_util.get_deploy_node(PY_JAVA_ENV).getDevType()
        result, origin_info, err_msg = modify_bmc_pwd_util.handle_dev_modify_bmc_new_password(login_info, dev_type)
        origin_infos.append(origin_info)
        if not result:
            msgs.append(err_msg)
            config_success = False
    host_name = context_util.get_config_os_name(PY_JAVA_ENV)
    if host_name:
        if config_bmc_host_name(login_info, host_name):
            msgs.append(entity.create_msg("bmc.hostname.config.success"))
        else:
            config_success = False
            msgs.append(entity.create_msg("bmc.hostname.config.failed"))

    is_success, origin_info = config_bmc_ip(login_info)
    origin_infos.append(origin_info)
    if is_success:
        task.updateNowIp2NodeIp()
        msgs.append(entity.create_msg("bmc.ip.config.success"))
    else:
        config_success = False
        msgs.append(entity.create_msg("bmc.ip.config.failed"))

    if not config_success:
        return ResultFactory.create_not_pass(origin_infos, msgs)
    return ResultFactory.create_pass(origin_infos, msgs)


def target_ip_reachable(ip_address, msgs, base_entity):
    if not ip_address or check_target_ip_reachable(ip_address.ip):
        return True
    msgs.append(base_entity.create_msg("ip.config.pass.not.reachable").format(ip_address.ip))
    return False


def config_bmc_host_name(login_info, host_name):
    try:
        redfish_util.modify_bmc_host_name(login_info, host_name, LOGGER)
    except DeployException:
        LOGGER.error(str(traceback.format_exc()))
        return False
    return True


def config_bmc_ip(login_info):
    ip_address = context_util.get_target_bmc_ip_address(PY_JAVA_ENV)
    ip_address_ds = context_util.get_target_bmc_ip_address_ds(PY_JAVA_ENV)
    try:
        redfish_util.set_bmc_ip(login_info, ip_address, ip_address_ds, LOGGER)
    except DeployException as e:
        LOGGER.error(str(traceback.format_exc()))
        return False, e.origin_info
    finally:
        HttpClient.release_session(login_info, LOGGER)
    return True, ""
