# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import re

from Common.base import entity, context_util
from Common.base.entity import ResultFactory, DeployException
from Common.protocol import ssh_util
from Common.protocol.redfish.entity.resource_client import ManagerResourceClient
from Common.protocol.ssh_util import BmcConnection


def execute(task):
    if task.getParam().getLldpEnable() is None and task.getParam().getHttpService() is None:
        return ResultFactory.create_pass(origin_info=entity.create_msg('not.support'))
    context = task.getJythonContext()
    origin_info = []
    try:
        config_services = [LldpService(context), HttpService(context)]
        for service in config_services:
            service.patch_info(task.getParam())
            origin_info.append(service.get_origin_info())
        return ResultFactory.create_pass(origin_info)
    except DeployException as exception:
        origin_info.append(str(exception.origin_info))
        return ResultFactory.create_not_pass(origin_info=origin_info, err_msg=exception.err_msg)


class LldpService(object):
    RESOURCE_PATH = "/LldpService"

    def __init__(self, context):
        self._login_info = context_util.get_login_info(context)
        self._client = ManagerResourceClient(self._login_info, entity.create_logger(__file__))
        self._resource_path = self._client.get_assign_resource_path() + LldpService.RESOURCE_PATH
        self._err_msg = []
        self._origin_info = []

    def patch_info(self, param):
        lldp_enbale = param.getLldpEnable()
        if lldp_enbale is None:
            return
        self._origin_info.append(str(
            self._client.request_patch(self._resource_path, {"LldpEnabled": bool(lldp_enbale)}).resource))

    def get_origin_info(self):
        return "\n".join(self._origin_info)


class HttpService(object):
    HTTP_STATE_REG = re.compile(r'HTTP\s+\|\s+(Enabled|Disabled)')
    CONFIRM_KEY_RET = "Y/N]:"

    def __init__(self, context):
        self._context = context
        self._err_msg = []
        self._origin_info = []

    @ssh_util.add_ssh_judge([CONFIRM_KEY_RET])
    def patch_info(self, param):
        ssh = BmcConnection(self._context).get_ssh()
        http_service = param.getHttpService()
        # 与前端对接，LLDP与HTTP为一个配置项，如果LLDP没有值，表示没有进入IBMC配置，则不需要配置Http,
        # 如果lldp有值，http没值，说明没有操作http，附默认值false
        if param.getLldpEnable() is None and http_service is None:
            return
        if http_service is None:
            http_service = "disabled"
        ssh_ret = ssh.execCmd('ipmcset -t service -d state -v http {}'.format(http_service))
        self._origin_info.append(ssh_ret)
        if 'successfully' in ssh_ret:
            return
        if HttpService.CONFIRM_KEY_RET not in ssh_ret and 'successfully' not in ssh_ret:
            raise DeployException('ipmcset param error', origin_info=ssh_ret)
        ssh_ret = ssh.execCmd('Y')
        self._origin_info.append(ssh_ret)
        if 'successfully' not in ssh_ret:
            raise DeployException('change http state failed', origin_info=ssh_ret,
                                  err_msg=entity.create_msg('config.bmc.failed').format('http'))

    def get_origin_info(self):
        return "\n".join(self._origin_info)
