# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.
import time

from Common.base import entity, context_util
from Common.base.entity import ResultFactory, DeployException
from Common.protocol.redfish.entity.resource_client import AccountServiceClient
from Common.util import deal_none_key_util

LOGGER = entity.create_logger(__file__)


def execute(task):
    if not task.getParam().getDeleteUser() and not task.getParam().getPostUser() and not task.getParam().getPatchUser():
        return ResultFactory.create_pass(origin_info=entity.create_msg('not.support'))
    return UserAccountService(task.getJythonContext()).execute(task.getParam())


class UserAccountService(object):
    def __init__(self, context):
        self._context = context
        self._login_info = context_util.get_login_info(context)
        self._client = AccountServiceClient(self._login_info, entity.create_logger(__file__))
        self._err_msg = []
        self.origin_info = []

    @staticmethod
    def _build_patch_param(user):
        param = {"UserName": user.getUsername(), "Password": user.getPassword(), "RoleId": user.getRoleId()}
        deal_none_key_util.deal_none_key(param)
        return param

    def execute(self, param):
        self.delete_info(param)
        self.patch_info(param)
        self.post_info(param)
        if self._err_msg:
            return ResultFactory.create_not_pass("\n".join(self.origin_info), err_msg="\n".join(self._err_msg))
        return ResultFactory.create_pass("\n".join(self.origin_info))

    def patch_info(self, param):
        patch_users = param.getPatchUser()
        if not patch_users:
            LOGGER.info("no user to patch")
            return
        for user in patch_users:
            if not user.getId():
                LOGGER.info("skip patch user：{} because of no user id".format(user.getUsername()))
                continue
            self.do_patch_user(user)

    def do_patch_user(self, user):
        try:
            self.origin_info.append(
                str(self._client.patch_assign_user_info_by_id(user.getId(), self._build_patch_param(user))))
            if user.getUsername() == self._login_info.username:
                LOGGER.info("current password changed in hardware config scene")
                self.update_password(user)
                self.origin_info.insert(0, entity.create_msg("current.user.password.changed"))
        except DeployException as e:
            self._err_msg.append(entity.create_msg("patch.user.failed").format(user.getUsername()))
            self.origin_info.append("patch {}: \n".format(user.getUsername()) + e.origin_info)
        # 从bmc现象来看，不管操作成功还是失败，只要对当前用户进行了修改操作，会话就会被踢掉，所以只要是修改当前用户，就刷新会话
        if user.getUsername() == self._login_info.username:
            self.refresh_session()

    def refresh_session(self):
        self._login_info = context_util.get_login_info(self._context)
        AccountServiceClient.release_session(self._login_info, LOGGER)
        self._client = AccountServiceClient(self._login_info, entity.create_logger(__file__))

    def update_password(self, user):
        node = context_util.get_deploy_node(self._context)
        bmc_user = node.getBmcUser()
        bmc_user.setOriginPassword(user.getPassword())
        bmc_user.setNewPassword(user.getPassword())
        login_user = node.getServer().getLoginUser()
        login_user.setPwd(bmc_user.getEncrtyptOriginPassword())
        login_user.setNewPwd(bmc_user.getEncrtyptOriginPassword())

    def post_info(self, param):
        post_users = param.getPostUser()
        if not post_users:
            LOGGER.info("no user to post")
            return
        for user in post_users:
            self.do_post_user(user)
            # 如果有设置登录策略，需要调用patch接口生效
            # 与前端约定，oem的固定放Public，此处构建参数的时候按照环境实际情况构建为public或者huawei
            oem_param = user.getOem()
            if not oem_param or not oem_param.get("Public"):
                continue
            # 新建用户后，需要等待10秒，iBMC同步用户状态后，再进行首次登录策略配置，否则可能出现412状态码
            time.sleep(10)
            self.patch_first_login_policy(oem_param, user)

    def patch_first_login_policy(self, oem_param, user):
        try:
            self.origin_info.append(str(
                self._client.patch_assign_user_info_by_id(user.getId(), self.build_login_policy_param(
                    oem_param.get("Public")))))
        except DeployException as e:
            self._err_msg.append(entity.create_msg("patch.user.first.login.policy.failed").format(user.getUsername()))
            self.origin_info.append("{} first login policy: \n".format(user.getUsername()) + e.origin_info)

    def do_post_user(self, user):
        try:
            self.origin_info.append(str(self._client.post_user_account(self._build_post_param(user))))
        except DeployException as e:
            self._err_msg.append(entity.create_msg("post.user.failed").format(user.getUsername()))
            self.origin_info.append("post {}: \n".format(user.getUsername()) + e.origin_info)

    def build_login_policy_param(self, login_policy):
        return {"Oem": {self._client.vendor: {"FirstLoginPolicy": login_policy.getFirstLoginPolicy()}}}

    def delete_info(self, param):
        delete_users = param.getDeleteUser()
        if not delete_users:
            LOGGER.info("no user to delete")
            return
        for user in delete_users:
            if self._login_info.username == user.getUsername():
                self._err_msg.append(entity.create_msg("cur.login.user.can.not.delete").format(user.getUsername()))
                continue
            self.do_delete_user(user)

    def do_delete_user(self, user):
        try:
            self.origin_info.append(str(self._client.delete_user_account(user.getId())))
        except DeployException as e:
            self._err_msg.append(entity.create_msg("delete.user.failed").format(user.getUsername()))
            self.origin_info.append("delete {}: \n".format(user.getUsername()) + e.origin_info)

    def _build_post_param(self, user):
        param = self._build_patch_param(user)
        param["Id"] = user.getId()
        return param
