#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.

"""
@time: 2020/12/26
@file: check_bios_firmware_version.py
@function:
"""
from Common.base import entity, context_util
from Common.base.entity import ResultFactory, DeployException
from Common.protocol.redfish import redfish_util

# 支持打开LLDP服务的网卡列表（1823网卡就是SP680网卡）
SUPPORT_NET_CARD = ["(SP380)", "(SP680)"]


def execute(task):
    return NetCardSetService(task.getJythonContext()).open_net_card_lldp_service()


class NetCardSetService(object):
    def __init__(self, java_py_env):
        self._origin_info = []
        # 需要打开LLDP服务的网卡
        self._lldp_net_cards = []
        self._login_info = context_util.get_login_info(java_py_env)
        self._logger = entity.create_logger(__file__)
        self._need_restart = False

    def open_net_card_lldp_service(self):
        try:
            members, response = redfish_util.get_net_cards(self._login_info, self._logger)
            self._origin_info.append(str(response))
            if not members:
                return ResultFactory.create_pass(origin_info=self._origin_info,
                                                 err_msg=entity.create_msg("not.support"))
            self._parse_net_card_data_id(members)
            if not self._lldp_net_cards:
                return ResultFactory.create_pass(origin_info=self._origin_info,
                                                 err_msg=entity.create_msg("not.support"))
            self._find_network_ports_by_net_card_id()
            return ResultFactory.create_pass(origin_info=self._origin_info)
        except DeployException as e:
            self._logger.error(e.message)
            return ResultFactory.create_not_pass(origin_info=e.origin_info, err_msg=e.err_msg)

    def _parse_net_card_data_id(self, member_resource):
        """  解析网卡类型，获得支持打开LLDP服务的网卡  """
        for member in member_resource:
            data_id = member.get("@odata.id", "")
            for card_type in SUPPORT_NET_CARD:
                if card_type in data_id:
                    self._lldp_net_cards.append(data_id)

    def _find_network_ports_by_net_card_id(self):
        """  通过网卡的URl，获取其端口，打开对应的LLDP服务 """
        for data_id in self._lldp_net_cards:
            port_members, resource = redfish_util.get_chassis_resource_url(
                self._login_info, "{}/{}".format(data_id, "NetworkPorts"), self._logger)
            self._origin_info.append(str(resource))
            if not port_members:
                continue
            self._open_lldp_service(port_members)

    def _open_lldp_service(self, port_members):
        """  打开对应端口的LLDP服务  """
        self._need_restart = True
        for port_id in port_members:
            port_id_resource_url = port_id.get("@odata.id")
            response = redfish_util.open_lldp_service_by_port_id_url(self._login_info, port_id_resource_url,
                                                                     self._logger)
            self._logger.info("open port_id:{} with response:{}".format(port_id_resource_url, response.resource))
            self._origin_info.append(str(response.resource))

    def need_restart(self):
        return self._need_restart
