#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/12
@file: config_mgmt_ssh_listen.py
@function:配置管理的ssh监听。
    ssh监听：默认不开启，则无论配了什么管理ip，都可以ssh访问系统。
    因安全加固，必须要限制访问的ip，则需要针对管理ip设置ssh监听，按理所有的
管理ip都需要配上此监听，但因为部署工具使用后会使用软件安装install工具，install工具
已经实现了该功能，而针对HCSO场景的外管软件不感知，则没有做，故需要在部署助手中配置.
    但是如果专门配外管，内管不配，会导致内管无法使用，而软件安装过程中需要用到内管，
故需要在这儿都一起配了。
@modify:
"""
from Common.base import context_util
from Common.protocol import ssh_util
from Common.base.entity import ResultFactory
PY_JAVA_ENV = py_java_env


def execute(task):
    return MgmtSshListenConfig().do_config()


class MgmtSshListenConfig(object):

    def __init__(self):
        self._strategys = context_util.get_os_network_strategy(PY_JAVA_ENV)
        self._ssh_rets = list()
        self._added_ssh_listen = False

    def do_config(self):
        for strategy in self._strategys:
            self._add_ssh_listen(strategy.getIpv4Address())
            self._add_ssh_listen(strategy.getIpv6Address())
        if self._added_ssh_listen:
            self._restart_ssh_service()
        return ResultFactory.create_pass(origin_info=self._ssh_rets)

    def _add_ssh_listen(self, ip_address):
        """
        添加ssh listen
        :param ip_address:  com.huawei.ism.tool.distributeddeploy.logic.
        importfile.entity.osnetwork.OsIpAddress 对象
        """
        if ip_address and ip_address.getPlane().isOmPlane():
            if not self._had_add_ssh_listen(ip_address.getIp()):
                self._ssh_rets.append(ssh_util.exec_ssh_cmd_nocheck(
                    PY_JAVA_ENV,
                    'echo -e "ListenAddress {}" >> /etc/ssh/sshd_config'.format(
                        ip_address.getIp())
                ))
                self._added_ssh_listen = True

    def _had_add_ssh_listen(self, ip):
        """
        已经被加过ssh监听
        :param ip: ip str
        :return: True/False
        """
        cmd = 'cat /etc/ssh/sshd_config | grep -w "^ListenAddress {}"'.format(ip)
        origin_info = ssh_util.exec_ssh_cmd_nocheck(
            PY_JAVA_ENV, cmd
        )
        self._ssh_rets.append(origin_info)
        return ip in origin_info.replace(cmd, "")

    def _restart_ssh_service(self):
        """
        重启ssh服务，因为ip不通等情况都有可能导致重启失败，故不做重启成功失败判断
        """
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_nocheck(
            PY_JAVA_ENV,
            "systemctl restart sshd.service"
        ))
