# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from Business.adaptor import java_adaptor
from Business.adaptor.java_adaptor import get_ipmi_monitor
from Common.base import context_util, entity
from Common.base.entity import ResultFactory, DeployException
from Common.protocol import ssh_util

ToolException = java_adaptor.get_tool_exception()


def execute(task):
    if task.getParam().getMobileCustom() is None and task.getParam().getChangeToPublic() is None:
        return ResultFactory.create_pass(origin_info=entity.create_msg('not.support'))
    context = task.getJythonContext()
    try:
        config_services = [MobileCustomService(context), ConfigToPublicService(context)]
        origin_info = []
        for service in config_services:
            service.patch_info(task.getParam())
            origin_info.append(service.get_origin_info())
        return ResultFactory.create_pass(origin_info="\n".join(origin_info))
    except DeployException as exception:
        return ResultFactory.create_not_pass(origin_info=exception.origin_info, err_msg=exception.err_msg)


def exec_ipmi_cmd(cmd, ssh, context):
    try:
        origin_info = str(ssh.execIpmiCmdWithoutSol(cmd, context_util.get_dev_node(context)))
        origin_info = deal_with_ipmi_login_error(cmd, context, origin_info, ssh)
        if origin_info.startswith("error"):
            raise DeployException("ipmi exec failed", origin_info=origin_info.split("-")[1],
                                  err_code="http.error.common")
        return origin_info
    except ToolException as exception:
        raise DeployException("ipmi exec failed", origin_info=exception.getDes(), err_code="http.error.common")


def deal_with_ipmi_login_error(cmd, context, origin_info, ssh):
    if ssh_util.IPMI_LOGIN_ERROR in origin_info:
        ssh_util.off_and_on_sol(ssh_util.BmcConnection(context).get_ssh())
        origin_info = str(ssh.execIpmiCmdWithoutSol(cmd, context_util.get_dev_node(context)))
    return origin_info


class MobileCustomService(object):
    def __init__(self, context):
        self._param = {
            True: "0xdb 0x07 0x00 0x35 0x63 0x00 0x01 0x00 0x00 0x00 0xff 0xff 0x00 0x00 0x03 0x00 0x79 0x01 0x01",
            False: ""
        }
        self._cmd = "raw 0x30 0x93 {}"
        self._context = context
        self._origin_info = []

    def patch_info(self, param):
        mobile_custom = param.getMobileCustom()
        # 当前没有打开功能
        if not mobile_custom:
            return
        self._origin_info.append(
            exec_ipmi_cmd(self._cmd.format(self._param.get(bool(mobile_custom))),
                          get_ipmi_monitor(), self._context)
        )

    def get_origin_info(self):
        return "\n".join(self._origin_info)


class ConfigToPublicService(object):
    def __init__(self, context):
        self._cmd_param = {
            True: "0xdb 0x07 0x00 0x5a 0x23 0x00 0x06 0x00 0x50 0x75 0x62 0x6C 0x69 0x63",
            False: "0xdb 0x07 0x00 0x5a 0x23 0x00 0x06 0x00 0x48 0x75 0x61 0x77 0x65 0x69"
        }
        self._cmd = "raw 0x30 0x93 {}"
        self._ssh = get_ipmi_monitor()
        self._origin_info = []
        self._context = context

    def patch_info(self, param):
        change_to_public = param.getChangeToPublic()
        if change_to_public is None:
            return
        self._origin_info.append(
            exec_ipmi_cmd(self._cmd.format(self._cmd_param.get(bool(change_to_public))),
                          self._ssh, self._context)
        )

    def get_origin_info(self):
        return "\n".join(self._origin_info)
