#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/20
@file: config_os_network.py
@function:
"""
import re

from com.huawei.ism.tool.distributeddeploy.entity.IpAddress import IpPattern
from com.huawei.ism.tool.distributeddeploy.logic.importfile.entity import \
    TransportProtocol, Plane
from com.huawei.ism.tool.distributeddeploy.logic.importfile.entity.osnetwork \
    import PhysicalConfigStrategy, BondConfigStrategy, \
    BondToVlanConfigStrategy, PhysicalToVlanConfigStrategy

from Common.base import context_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util
from Common.service import os_network_config_creator
from Common.service.config_mlx_nic_mode import ConfigMlxNicMode
from Common.service.physical_card_to_logical import PhysicalCardToLogical
from Common.util import restore_and_compress_ipv6

PY_JAVA_ENV = py_java_env
ROUTE_TABLE_ID_REG = re.compile(r"^\d{1,3} ")
ONCE_WRITE_LENE = 1
# 遇到执行命令卡住的问题，将所有写文件的操作改为单行写入，目前没有遇到单行命令卡主的问题，修改此值可以最快适配该场景，注释说明，避免后续歧义
CMD_LEN_LIMIT = 1
RECORD_DIR_PATH = "/etc/sysconfig/network-record-smartkit"
RECORD_FILE_PATH = "/etc/sysconfig/network-record-smartkit/record_list"


def execute(task):
    return ConfigOsNetwork(task).config()


def restore_initial_configuration():
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'cat {}'.format(RECORD_FILE_PATH))
    # 没有记录文件说明是首次执行，创建目录文件后直接返回
    if 'No such file or directory' in ssh_ret:
        ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'mkdir {}'.format(RECORD_DIR_PATH))
        ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'touch {}'.format(RECORD_FILE_PATH))
        return
    for line in ssh_ret.splitlines():
        if 'add ' in line:
            file_path = line.strip().split()[1]
            ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'rm -f {}'.format(file_path))
            # 控制IP可能漂移到其他网卡配置上，这里做个兜底删除。
            try_to_delete_ctrl_ip_ifcfg_file(file_path)
        if 'modify ' in line:
            modify_file_path = line.strip().split()[1]
            save_file_path = line.strip().split()[2]
            ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'rm -f {};cp -a {} {}'.format(modify_file_path, save_file_path,
                                                                                     modify_file_path))


def try_to_delete_ctrl_ip_ifcfg_file(file_path):
    """
    尝试删除控制IP的网卡配置文件，因存储前后端的控制IP默认配置在IP1上，但是可能出现漂移到IP2，IP3，IP4上，但是工具只配置并记录了IP1，
    所以出现漂移时其他IP下需要尝试删除。控制IP的网卡配置文件默认为：网卡名称:0
    :param file_path: 网卡配置文件
    :return: 无
    """
    if "ifcfg-" in file_path and not file_path.endswith(":0"):
        ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'rm -f {}:0'.format(file_path))


def record_summary_configuration(file_path):
    # 只在第一次操作时记录初始情况
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'cat {}'.format(RECORD_FILE_PATH))
    if check_has_record_summary_configuration(file_path, ssh_ret):
        return
    ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'ls {}'.format(file_path))
    # 新增文件只记录，标记”add“，修改文件记录并复制，标记”modify“
    if 'No such file or directory' in ssh_ret:
        ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'echo -e "add {}" >> {}'.format(file_path, RECORD_FILE_PATH))
    else:
        file_name = file_path.split('/')[-1]
        # 原子化操作，防止因工具连接波动导致复制初始文件成功但是记录失败
        ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'cp -a {} {};echo -e "modify {} {}/{}" >> {}'
                                      .format(file_path, RECORD_DIR_PATH, file_path, RECORD_DIR_PATH, file_name,
                                              RECORD_FILE_PATH))


def check_has_record_summary_configuration(file_path, ssh_ret):
    # 检查该文件是否被记录过，因为文件名可能出现file和file:0的场景，因此必须把路径解析出来判断相等。
    for line in ssh_ret.splitlines():
        if 'add ' in line and line.strip().split()[1] == file_path:
            return True
        if 'modify ' in line and line.strip().split()[1] == file_path:
            return True
    return False


class ConfigOsNetwork(object):
    STATIC_ROUTE_FILE = '/etc/static_route_tool.cfg'
    POLICY_ROUTE_FILE = '/etc/policy_route_tool.cfg'

    def __init__(self, task):
        self._logger = entity.create_logger(__file__)
        self._task = task
        self._strategys = context_util.get_os_network_strategy(PY_JAVA_ENV)
        self.mapping_verion = context_util.get_mapping_version(PY_JAVA_ENV)
        self._port_names = list()
        self._port_bond_names = list()
        self._physical_to_logical = None
        self._ssh_rets = list()
        self._err_msgs = list()
        self._black_table_id_2_names = dict()
        self._table_id = 100
        # DeviceManager读取工具配置的静态路由信息
        self._device_manager_static_route_infos = list()
        # DeviceManager读取工具配置的策略路由信息
        self._device_manager_policy_route_infos = list()
        self._slot_ports_relation_dict = self._get_slot_ports_relation_dict()
        self._config_mlx_nic_mode = ConfigMlxNicMode(PY_JAVA_ENV)

    def config(self):
        try:
            restore_initial_configuration()
            self._init_route_table_info()
            self._physical_to_logical = PhysicalCardToLogical(PY_JAVA_ENV)
            self._ssh_rets.append(self._physical_to_logical.slot_ssh_info)
            self._task.updateProgress(5)
            # 检查网卡模式是否匹配
            check_nic_result = self._check_nic_mode_is_match()
            self._ssh_rets.extend(self._config_mlx_nic_mode.ssh_rets)
            if not check_nic_result:
                return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
            self._task.updateProgress(15)
            for index in range(len(self._strategys)):
                strategy_progress = 50 / len(self._strategys)
                self._task.updateProgress(5 + strategy_progress * (index + 1))
                self._config_network_strategy(self._strategys[index])
            self._write_device_manager_route_info()
            self._task.updateProgress(80)
            context_util.set_port_list(PY_JAVA_ENV, self._port_names)
            context_util.set_port_bond_list(PY_JAVA_ENV, self._port_bond_names)
            if not self._restart_network_service():
                return ResultFactory.create_not_pass(self._ssh_rets,
                                                     self._err_msgs)

            self._task.updateProgress(90)
        except DeployException as exception:
            self._logger.error(exception.message)
            self._ssh_rets.append(exception.origin_info)
            self._err_msgs.append(exception.err_msg)
            self._task.openAutoRetry()
        if len(self._err_msgs) != 0:
            return ResultFactory.create_not_pass(self._ssh_rets,
                                                 self._err_msgs)
        return ResultFactory.create_pass(self._ssh_rets)

    def _config_network_strategy(self, strategy):
        self._logger.info("config strategy: {}".format(strategy))
        if isinstance(strategy, PhysicalConfigStrategy):
            self._config_physical_port_strategy(strategy)
        elif isinstance(strategy, BondConfigStrategy):
            self._config_bond_config_strategy(strategy)
        elif isinstance(strategy, PhysicalToVlanConfigStrategy):
            self._config_physical_to_vlan_config_strategy(strategy)
        elif isinstance(strategy, BondToVlanConfigStrategy):
            self._config_bond_to_vlan_config_strategy(strategy)

    def _config_physical_port_strategy(self, strategy):
        self._logger.info('config physical port start...')
        slot = strategy.getPhysicalPort().getSlot()
        port = strategy.getPhysicalPort().getPort()
        port_name, ssh_ret = self._physical_to_logical.get_logical_name(
            slot, port, self._slot_ports_relation_dict)
        self._ssh_rets.append(ssh_ret)
        self._port_names.append(port_name)
        ipv4 = strategy.getIpv4Address()
        ipv6 = strategy.getIpv6Address()
        self._logger.info("[physical]{}、{}、{}".format(slot, port, port_name))
        if strategy.containsIpv4() and strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_logical_port_with_ipv4_and_ipv6(port_name, ipv4, ipv6)
        elif strategy.containsIpv4():
            config_info = os_network_config_creator. \
                create_logical_port_with_ipv4(port_name, ipv4)
        elif strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_logical_port_with_ipv6(port_name, ipv6)

        self._config_physical_ctrl_ip(strategy.getCtrlIpAddress(), port_name)

        self._write_network_file(port_name, config_info)
        self._config_route(strategy, port_name)

    def _config_physical_ctrl_ip(self, ctrl_ip_address, port_name):
        if not ctrl_ip_address:
            return
        ctrl_ip_port_name = self._create_ctrl_ip_port_name(port_name)
        if ctrl_ip_address.getIpPattern() is IpPattern.IPV4:
            ctrl_ip_info = os_network_config_creator. \
                create_logical_port_with_ipv4(ctrl_ip_port_name,
                                              ctrl_ip_address)
        else:
            ctrl_ip_info = os_network_config_creator. \
                create_logical_port_with_ipv6(ctrl_ip_port_name,
                                              ctrl_ip_address)
        ctrl_ip_info += self._get_ctrl_ip_forbid_default_route_info()
        self._write_network_file(ctrl_ip_port_name, ctrl_ip_info)

    def _create_ctrl_ip_port_name(self, port_name):
        return "{}:0".format(port_name)

    def _get_ctrl_ip_forbid_default_route_info(self):
        """
        控制ip是在一个端口上取的别名配的ip，如self._create_ctrl_ip_port_name()
        而这种情况下如果配有网关，通过DEFROUTE=no是控制不了的别名ip，这样会导致控制
        Ip的网关生成默认路由，故需要加上该信息使得别名的不会生成默认路由
        :return:
        """
        return "\nNO_ALIASROUTING=yes"

    def _config_bond_config_strategy(self, strategy):
        self._logger.info('config bond start...')
        bond_port = strategy.getBondPort()
        bond_name = bond_port.getBondName()
        port_names = self._mapping_logical_port(bond_port.getPhysicalPorts())
        self._write_bond_physicals_config(port_names, bond_name,
                                          strategy.getTransportProtocol())
        ipv4 = strategy.getIpv4Address()
        ipv6 = strategy.getIpv6Address()
        if strategy.containsIpv4() and strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_bond_port_with_ipv4_and_ipv6(bond_port, ipv4, ipv6,
                                                    port_names)
        elif strategy.containsIpv4():
            config_info = os_network_config_creator. \
                create_bond_port_with_ipv4(bond_port, ipv4, port_names)
        elif strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_bond_port_with_ipv6(bond_port, ipv6, port_names)

        self._config_bond_ctrl_ip(bond_port, strategy.getCtrlIpAddress(),
                                  port_names)
        config_info = self._add_storage_ib_scene_config_info(
            config_info, strategy)
        self._write_network_file(bond_name, config_info)
        self._config_route(strategy, bond_name)
        # 记录绑定端口，以便重启后检查端口启动状态
        self._port_bond_names.append(bond_name)

    def _add_storage_ib_scene_config_info(self, config_info, strategy):
        # 存储IB下这个可能是多余的，后续对齐看把他干掉
        if strategy.getTransportProtocol() is TransportProtocol.IB:
            config_info += "\nSTARTMODE=onboot"
        return config_info

    def _config_bond_ctrl_ip(self, bond_port, ctrl_ip_address, port_names):
        if not ctrl_ip_address:
            return
        ctrl_ip_port_name = self._create_ctrl_ip_port_name(bond_port.getBondName())
        if ctrl_ip_address.getIpPattern() is IpPattern.IPV4:
            ctrl_ip_info = os_network_config_creator. \
                create_bond_port_with_ipv4(bond_port,
                                           ctrl_ip_address,
                                           port_names)
        else:
            ctrl_ip_info = os_network_config_creator. \
                create_bond_port_with_ipv6(bond_port,
                                           ctrl_ip_address,
                                           port_names)
        ctrl_ip_info += self._get_ctrl_ip_forbid_default_route_info()
        self._write_network_file(ctrl_ip_port_name, ctrl_ip_info)

    def _config_physical_to_vlan_config_strategy(self, strategy):
        self._logger.info('config physical to vlan start...')
        slot = strategy.getPhysicalPort().getSlot()
        port = strategy.getPhysicalPort().getPort()
        port_name, ssh_ret = self._physical_to_logical.get_logical_name(
            slot, port, self._slot_ports_relation_dict)
        self._ssh_rets.append(ssh_ret)
        self._logger.info("[physical to Vlan]{}、{}、{}".format(
            slot, port, port_name))
        self._port_names.append(port_name)
        config_info = os_network_config_creator.create_logical_port_to_vlan(port_name)
        if self.need_modify_config_info(config_info, strategy):
            config_info += "\nDEFROUTE=no"
        self._write_network_file(port_name, config_info)
        # 配置vlan
        self._write_vlan_config(strategy, port_name)

    def need_modify_config_info(self, config_info, strategy):
        return (self._is_ge_813_version(PY_JAVA_ENV) and strategy.getTransportProtocol() is TransportProtocol.ROCE \
                and self._is_storage_plane_or_equivalent(strategy) and "DEFROUTE" not in config_info)

    def _is_storage_plane_or_equivalent(self, strategy):
        return Plane.isStoragePlane(strategy.getPlane()) or strategy.isEquivalentToStoragePlane()

    def _config_bond_to_vlan_config_strategy(self, strategy):
        self._logger.info('config bond to vlan start...')
        bond_port = strategy.getBondPort()
        bond_name = bond_port.getBondName()
        port_names = self._mapping_logical_port(bond_port.getPhysicalPorts())
        self._write_bond_physicals_config(port_names, bond_name,
                                          strategy.getTransportProtocol())
        config_info = os_network_config_creator. \
            create_bond_port_no_ip(bond_port, port_names)
        config_info = self._add_storage_ib_scene_config_info(
            config_info, strategy)
        self._write_network_file(bond_name, config_info)
        # 配置vlan
        self._write_vlan_config(strategy, bond_name)
        # 记录绑定端口，以便重启后检查端口启动状态
        self._port_bond_names.append(bond_name)

    def _write_bond_physicals_config(self, port_names, bond_name,
                                     transport_protocol):
        for index in range(len(port_names)):
            port_name = port_names[index]
            config_info = os_network_config_creator. \
                create_logical_port_to_bond(port_name, bond_name)
            if transport_protocol is TransportProtocol.IB:
                config_info += "\nTYPE=InfiniBand\nBONDING_SLAVE{}={}".format(
                    index, port_name)
            self._write_network_file(port_name, config_info)

    def _mapping_logical_port(self, physical_ports):
        port_names = list()
        for physical_port in physical_ports:
            slot = physical_port.getSlot()
            port = physical_port.getPort()
            port_name, ssh_ret = self._physical_to_logical.get_logical_name(
                slot, port, self._slot_ports_relation_dict)
            self._ssh_rets.append(ssh_ret)
            self._port_names.append(port_name)
            port_names.append(port_name)
        return port_names

    def _write_vlan_config(self, strategy, port_name):
        ipv4 = strategy.getIpv4Address()
        ipv6 = strategy.getIpv6Address()
        vlan_name = strategy.getVlan().getVlanName()
        vlan_id = strategy.getVlan().getVlanId()
        if strategy.containsIpv4() and strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_vlan_with_ipv4_and_ipv6(vlan_name, ipv4, ipv6,
                                               port_name, vlan_id)
        elif strategy.containsIpv4():
            config_info = os_network_config_creator. \
                create_vlan_with_ipv4(vlan_name, ipv4, port_name, vlan_id)
        elif strategy.containsIpv6():
            config_info = os_network_config_creator. \
                create_vlan_with_ipv6(vlan_name, ipv6, port_name, vlan_id)

        self._config_vlan_ctrl_ip(strategy, strategy.getCtrlIpAddress(), port_name,
                                  vlan_id, vlan_name)
        self._write_network_file(vlan_name, config_info)
        self._config_route(strategy, vlan_name)

    def _config_route(self, strategy, port_name):
        ipv4 = strategy.getIpv4Address()
        ipv6 = strategy.getIpv6Address()
        self._write_route_file(
            port_name, strategy.getRouteCollection().getIpv4RouteInfo(
                port_name, port_name, str(self._table_id)), ipv4)
        self._write_route_file(
            port_name, strategy.getRouteCollection().getIpv6RouteInfo(
                port_name, port_name, str(self._table_id)), ipv6)
        if strategy.getRouteCollection().containsPolicyRoute():
            self._write_policy_rule_file(port_name, strategy)
            # 策略路由，需要生成路由表
            self._obtain_valid_route_table_id(port_name)
        self._device_manager_policy_route_infos.append(
            strategy.getRouteCollection().getDeviceManagerPolicyRouteInfo(
                port_name, port_name, str(self._table_id)
            )
        )
        self._device_manager_static_route_infos.append(
            strategy.getRouteCollection().getDeviceManagerStaticRouteInfo(
                port_name)
        )

    def _write_device_manager_route_info(self):
        """
        配置的路由在某些极端场景下可能会造成路由丢失，系统不会自动恢复，故而工具与
        DeviceManager约定，将工具配置的路由信息写到固定的某个os的文件中，软件读取
        这个文件的内容，定期进行监控，如果有路由丢失的情况，软件协助主动拉起路由
        """
        self.compress_ipv6()
        static_route_info = "\n".join(
            self._device_manager_static_route_infos).strip()
        record_summary_configuration(self.STATIC_ROUTE_FILE)
        write_cmd = 'echo -e "{}" > {}'.format(
            static_route_info, self.STATIC_ROUTE_FILE)
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV, write_cmd))
        self._add_file_permission(self.STATIC_ROUTE_FILE)

        policy_route_info = "\n".join(
            self._device_manager_policy_route_infos).strip()
        record_summary_configuration(self.POLICY_ROUTE_FILE)
        write_cmd = 'echo -e "{}" > {}'.format(
            policy_route_info, self.POLICY_ROUTE_FILE)
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV, write_cmd))
        self._add_file_permission(self.POLICY_ROUTE_FILE)

    def compress_ipv6(self):
        self._logger.info("static route info :{}".format(self._device_manager_static_route_infos))
        compress_ips = []
        for route_info in self._device_manager_static_route_infos:
            if not route_info:
                compress_ips.append(route_info)
                continue
            route_infos = route_info.splitlines()
            compress_ips.append(self.compress_line(route_infos))
        self._device_manager_static_route_infos = compress_ips

    @staticmethod
    def compress_line(route_infos):
        compress_ips = []
        for line in route_infos:
            split_route_info = line.split(" ")
            ipaddr = split_route_info[0]
            # ipv6地址有效校验在导入表格时校验过了，此处简单判断有：就是ipv6地址
            if ":" in ipaddr:
                split_route_info[0] = restore_and_compress_ipv6.compress_ipv6(ipaddr)
            compress_ips.append(" ".join(split_route_info))
        return "\n".join(compress_ips)

    def _config_vlan_ctrl_ip(self, strategy, ctrl_ip_address, port_name, vlan_id,
                             vlan_name):
        if not ctrl_ip_address:
            return
        ctrl_ip_port_name = self._create_ctrl_ip_port_name(vlan_name)
        if ctrl_ip_address.getIpPattern() is IpPattern.IPV4:
            ctrl_ip_info = os_network_config_creator. \
                create_vlan_with_ipv4(ctrl_ip_port_name, ctrl_ip_address,
                                      port_name, vlan_id)
        else:
            ctrl_ip_info = os_network_config_creator. \
                create_vlan_with_ipv6(ctrl_ip_port_name, ctrl_ip_address,
                                      port_name, vlan_id)
        ctrl_ip_info += self._get_ctrl_ip_forbid_default_route_info()
        if self._is_ge_813_version(PY_JAVA_ENV) and self._is_storage_plane_or_equivalent(
                strategy) and TransportProtocol.ROCE == strategy.getTransportProtocol():
            ctrl_ip_info = self._remove_networking_key(ctrl_ip_info)
        self._write_network_file(ctrl_ip_port_name, ctrl_ip_info)

    def _write_config_file(self, config_file_name, config_info, timeout=30):
        file_path = "/etc/sysconfig/network-scripts/{}".format(
            config_file_name)
        record_summary_configuration(file_path)
        write_cmd = 'echo -e "{}" > {}'
        if len(config_info) > CMD_LEN_LIMIT:
            self._write_file_by_step(config_info, file_path, write_cmd, timeout)
        else:
            write_cmd = write_cmd.format(config_info, file_path)
            self._exec_write_config_cmd(write_cmd, timeout)
        if not self.mapping_verion.startswith("8.0.1"):
            self._add_file_permission(file_path)
        self._check_write_correct(file_path, config_info)

    def _write_file_by_step(self, config_info, file_path, write_cmd, timeout=30):
        ssh = ssh_util.get_ssh(PY_JAVA_ENV)
        append_write_cmd = 'echo -e "{}" >> {}'
        config_infos = config_info.splitlines()
        start_index = ONCE_WRITE_LENE
        cmd = write_cmd.format('\n'.join(config_infos[0:ONCE_WRITE_LENE]), file_path)
        self._exec_cmd(ssh, cmd, timeout)
        while start_index < len(config_infos):
            cmd = append_write_cmd.format('\n'.join(config_infos[start_index:start_index + ONCE_WRITE_LENE]), file_path)
            self._exec_cmd(ssh, cmd, timeout)
            start_index += ONCE_WRITE_LENE

    def _exec_cmd(self, ssh, cmd, timeout):
        if timeout:
            self._ssh_rets.append(ssh.execCmdWithTimout(cmd, timeout))
        else:
            self._ssh_rets.append(ssh.execCmd(cmd))

    def _exec_write_config_cmd(self, write_cmd, timeout=None):
        if timeout:
            self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid_with_timeout_and_retry(
                PY_JAVA_ENV, write_cmd, timeout, self._logger))
        else:
            self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
                PY_JAVA_ENV, write_cmd))

    def _check_write_correct(self, file_path, config_info):
        cmd = "cat {}".format(file_path)
        ssh_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
        config_infos = config_info.splitlines()
        for config_line in config_infos:
            if config_line in ssh_ret:
                continue
            self._err_msgs.append(entity.create_msg("network.file.write.error").format(file_path, config_info))
            self._logger.error("write_content:{} file_content:{}  check not pass".format(config_info, ssh_ret))
            return

    def _add_file_permission(self, file_path):
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV,
            "chmod 644 {}".format(file_path)))
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV,
            "chown -h root:root {}".format(file_path)))

    def _write_network_file(self, name, config_info):
        network_file_name = "ifcfg-{}".format(name)
        self._write_config_file(network_file_name, config_info)

    def _write_policy_rule_file(self, port_name, strategy):
        self._write_rule_file(
            port_name, strategy.getRouteCollection().getIpv4RuleInfo(
                port_name), strategy.getIpv4Address())
        self._write_rule_file(
            port_name, strategy.getRouteCollection().getIpv6RuleInfo(
                port_name), strategy.getIpv6Address())

    def _write_route_file(self, port_name, route_info, ip):
        # (1)ip为空：表示该端口不配此类Ip，也就不用路由
        # (2)没有配置路由信息也不需要配
        if not ip or not route_info:
            return
        if ip.getIpPattern() is IpPattern.IPV4:
            route_file_name = "route-{}".format(port_name)
        else:
            route_file_name = "route6-{}".format(port_name)
        '''路由配置给15分钟默认超时，超时后释放连接重试一次'''
        self._write_config_file(route_file_name, route_info, 60 * 15)

    def _write_rule_file(self, port_name, route_info, ip):
        # (1)ip为空：表示该端口不配此类Ip，也就不用路由
        # (2)没有配置路由信息也不需要配
        if not ip or not route_info:
            return
        if ip.getIpPattern() is IpPattern.IPV4:
            route_file_name = "rule-{}".format(port_name)
        else:
            route_file_name = "rule6-{}".format(port_name)
        self._write_config_file(route_file_name, route_info)

    def _append_strategy_route_table(self, port_name):
        record_summary_configuration('/etc/iproute2/rt_tables')
        write_cmd = 'echo -e "{}    {}" >> /etc/iproute2/rt_tables'.format(
            self._table_id, port_name)
        self._ssh_rets.append(ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV, write_cmd))
        self._black_table_id_2_names[self._table_id] = port_name

    def _init_route_table_info(self):
        """
        初始化路由表信息，查看记录已有的路由表，作为黑名单，不能重复了
        """
        ssh_ret = ssh_util.exec_ssh_cmd_disable_invalid(
            PY_JAVA_ENV, "cat /etc/iproute2/rt_tables")
        self._ssh_rets.append(ssh_ret)
        for line in ssh_ret.splitlines():
            match = ROUTE_TABLE_ID_REG.findall(line)
            if match:
                table_infos = re.split(r" +", line)
                if len(table_infos) > 1:
                    self._black_table_id_2_names[int(match[0])] = table_infos[
                        1]

    def _obtain_valid_route_table_id(self, port_name):
        """
        配置路由表
        :param port_name: 做路由表名
        """
        while self._table_id >= 0:
            if self._table_id not in self._black_table_id_2_names:
                self._append_strategy_route_table(port_name)
                return
            elif port_name == self._black_table_id_2_names.get(self._table_id):
                # 对于id相同，表名相同的情况，则直接使用，二次配的情况
                return
            self._table_id -= 1
        raise DeployException("not found invalid table id")

    def _restart_network_service(self):
        restart_cmd = "service network restart"
        # 网络重启在IP配置较多时，默认30秒钟可能无法启动完成，修改超时时间为3分钟
        cli_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, restart_cmd, True, 180)
        self._ssh_rets.append(cli_ret)
        restart_success = "[  OK  ]" in cli_ret
        if not restart_success:
            self._task.openAutoRetry()
            self._err_msgs.append(
                entity.create_msg("restart.network.service.failed"))
        return restart_success

    def _get_slot_ports_relation_dict(self):
        slot_to_ports_dict = dict()
        for strategy in self._strategys:
            slot_to_ports = strategy.getSlotToPortsRelation()
            slots = slot_to_ports.keys()
            for slot in slots:
                if slot not in slot_to_ports_dict:
                    slot_to_ports_dict[slot.lower()] = slot_to_ports[slot]
                else:
                    slot_to_ports_dict[slot.lower()].update(slot_to_ports[slot])
        return slot_to_ports_dict

    def _remove_networking_key(self, ctrl_ip_info):
        return ctrl_ip_info.replace("NETWORKING=yes", "").replace("NETWORKING_IPV6=yes", "")

    def _is_ge_813_version(self, context):
        """
        当前部署版本是否大于等于813版本，采用逐位转数字比较，任意一位数字小于给定的标准，则不满足
        :param context:
        :return:
        """
        # A310为820之后衍生出得
        if "a310" in context_util.get_mapping_version(context):
            return True
        mapping_version = re.findall(r"\d+\.\d+\.\d+", context_util.get_mapping_version(context))
        if not mapping_version:
            return False
        need_config_ver = 813
        ver_num = int(mapping_version[0].replace(".", ""))
        return ver_num >= need_config_ver

    def _check_nic_mode_is_match(self):
        need_reboot = False
        self._logger.info("start to check nic mode is match.")
        for slot_address, network_type in self._get_all_strategy_slot_address_and_network_type().items():
            bus_infos = self._config_mlx_nic_mode.query_slot_bus_info_by_slot_address(slot_address)
            self._logger.info("query bus info, slot {}, bus {}.".format(slot_address, bus_infos))
            if not bus_infos:
                continue
            for bus_info in bus_infos:
                if self._config_mlx_nic_mode.check_slot_nic_is_special_mode(bus_info, network_type):
                    self._logger.info("check bus nic is same, bus {} type {}.".format(bus_info, network_type))
                    continue
                self._logger.info("star to change nic mode, bus {} type {}.".format(bus_info, network_type))
                if not self._config_mlx_nic_mode.change_slot_nic_mode(bus_info, network_type):
                    self._logger.info("change nic mode failed, bus {} type {}.".format(bus_info, network_type))
                    return False
                need_reboot = True
                self._logger.info("change nic mode success, bus {} type {}.".format(bus_info, network_type))
        self._logger.info("check nic mode finish, reboot : {}.".format(need_reboot))
        if need_reboot:
            self._task.updateProgress(10)
            return self._config_mlx_nic_mode.reboot_node_and_wait_for_started()
        return True

    def _get_all_strategy_slot_address_and_network_type(self):
        slot_address_2_network_type = dict()
        for strategy in self._strategys:
            if not Plane.isStoragePlane(strategy.getPlane()):
                self._logger.info("current not storage plane, {}".format(strategy))
                continue
            slot_address_2_network_type.update(self._get_strategy_slot_address_and_network_type(strategy))
        self._logger.info(
            "get all strategy slot address and network type result : {}".format(slot_address_2_network_type))
        return slot_address_2_network_type

    def _get_strategy_slot_address_and_network_type(self, strategy):
        slot_address_2_network_type = dict()
        self._logger.info("get strategy slot address and network type: {}".format(strategy))
        if isinstance(strategy, PhysicalConfigStrategy) or isinstance(strategy, PhysicalToVlanConfigStrategy):
            slot_address = self._physical_to_logical.get_address_by_slot(strategy.getPhysicalPort().getSlot())
            if not slot_address:
                self._logger.info("not find slot address : {}".format(strategy))
                return slot_address_2_network_type
            slot_address_2_network_type[slot_address] = strategy.getTransportProtocol()
        elif isinstance(strategy, BondConfigStrategy) or isinstance(strategy, BondToVlanConfigStrategy):
            for physical_port in strategy.getBondPort().getPhysicalPorts():
                slot_address = self._physical_to_logical.get_address_by_slot(physical_port.getSlot())
                if not slot_address:
                    self._logger.info("not find bond slot address : {}".format(physical_port))
                    continue
                slot_address_2_network_type[slot_address] = strategy.getTransportProtocol()
        self._logger.info("get strategy slot address and network type result : {}".format(slot_address_2_network_type))
        return slot_address_2_network_type
