#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

"""
@time: 2023/05/23
@file: config_os_password_and_timezone.py
@function:
"""

import time
import re

from Business.adaptor import java_adaptor
from Common.base.constant import MsgKey
from Common.base import context_util
from Common.base import entity
from Common.base.entity import ResultFactory
from Common.base.entity import DeployException
from Common.protocol.redfish import redfish_util
from Common.protocol.ssh_util import get_ssh, SOL_TO_OS_CMD, OFF_SOL_CMD, is_cmd_time_out

JException = java_adaptor.get_j_exception()
ConnectionManager = java_adaptor.get_connection_manager()
logger = entity.create_logger(__file__)
login_info = context_util.get_login_info(py_java_env)
UMOUNT_MEDIA = {"VmmControlType": "Disconnect"}
origin_infos = []
err_msgs = []


def execute(task):
    try:
        context = task.getJythonContext()
        ssh = get_ssh(context, switch_root=False)
        ssh_ret = ssh.execCmd("\n")
        origin_infos.append(ssh_ret)
        if ssh_ret.strip().endswith("iBMC:/->"):
            ssh_ret = ssh.execCmdWithTimout(SOL_TO_OS_CMD, 10)
            origin_infos.append(ssh_ret)
            ssh_ret = handle_private_model(ssh, ssh_ret)
            if "successfully" not in ssh_ret:
                err_msgs.append(entity.create_msg("switch.os.failed"))
            else:
                # 执行跳转os命令后，有些设备需要发送回车才会有”login:“回显
                # 有时因命令发送太快，串口丢失信息的问题而导致命令未生效，实测通过睡眠可规避此问题
                time.sleep(2)
                check_os_login(context, ssh)
        elif ssh_ret.strip().endswith("login:"):
            login_os_to_modify_root_password(context, ssh)
        else:
            err_msgs.append(entity.create_msg("nonstandard.ibmc.environment"))
        if not err_msgs:
            return ResultFactory.create_pass(origin_infos, entity.create_msg("progress.finish"))
        return ResultFactory.create_not_pass(origin_infos, err_msgs)
    except JException:
        return ResultFactory.create_not_pass(origin_infos, entity.create_msg(MsgKey.OBTAIN_INFO_FAILED))
    finally:
        umount_virtual_media_if_need()
        context = task.getJythonContext()
        dev_node = context_util.get_dev_node(context)
        platform_id = context_util.get_platform_id(context)
        ConnectionManager.releaseConnection(dev_node, platform_id)


def handle_private_model(ssh, ssh_ret):
    if "Current SOL session is in private mode" in ssh_ret:
        origin_infos.append(ssh.execCmdWithTimout(OFF_SOL_CMD, 10))
        tmp_ssh_ret = ssh.execCmdWithTimout(SOL_TO_OS_CMD, 10)
        origin_infos.append(tmp_ssh_ret)
        return tmp_ssh_ret
    return ssh_ret


def umount_virtual_media_if_need():
    virtual_media = redfish_util.current_virtual_media(login_info, logger)
    if virtual_media.get("Inserted"):
        redfish_util.umount_current_virtual_media(login_info, UMOUNT_MEDIA, logger)


def check_os_login(context, ssh):
    """
    检查是否成功跳转至OS
    :param context: 上下文
    :param ssh: ssh链接
    :return: 检查成功配置操作系统密码；检查失败，任务失败，返回错误信息
    """
    ssh_ret = ssh.execCmd("\n")
    origin_infos.append(ssh_ret)
    if ssh_ret.strip().endswith("login:"):
        login_os_to_modify_root_password(context, ssh, True)
    elif ssh_ret.strip().endswith("]#"):
        login_os_to_modify_root_password(context, ssh, False)
    else:
        err_msgs.append(entity.create_msg("switch.os.failed"))


def get_os_root_account_infos(context):
    """
    获取root用户信息
    :param context:上下文
    :return: root用户信息
    """
    try:
        deploy_node = context_util.get_deploy_node(context)
        root_password = deploy_node.getOsRootUser().getOriginPassword()
        root_new_password = deploy_node.getOsRootUser().getNewPassword()
        os_username = deploy_node.getOsUser().getUserName()
        os_password = deploy_node.getOsUser().getOriginPassword()
        os_new_password = deploy_node.getOsUser().getNewPassword()
        if root_new_password and root_password != root_new_password:
            return {
                "username": "root",
                "root_password": root_password,
                "root_new_password": root_new_password,
                "os_username": os_username,
                "os_passwd": os_password,
                "os_new_passwd": os_new_password
            }
        err_msgs.append(entity.create_msg("modify.password.failed").format("root"))
        return {}
    finally:
        del root_password, root_new_password


def login_os_to_modify_root_password(context, ssh, need_login=True):
    """
    使用root账户登录OS，并视情况修改密码
    :param context: 上下文
    :param ssh: ssh链接
    :param need_login: 是否需要登录OS
    :return: 登录失败，添加错误信息，提前返回
    """
    try:
        root_account_infos = get_os_root_account_infos(context)
        if err_msgs:
            return
        if not need_login:
            modify_os_timezone(context, ssh)
            modify_os_password(root_account_infos, ssh)
            return
        # 登录os
        origin_infos.append(ssh.execCmd("root"))
        ssh_ret = ssh.execCmdNoLog(root_account_infos.get("root_password"))
        origin_infos.append(ssh_ret)
        if ssh_ret.strip().endswith("Current password:"):
            modify_root_password(ssh, root_account_infos, context)
        elif ssh_ret.strip().endswith("]#"):
            modify_os_timezone(context, ssh)
            modify_os_password(root_account_infos, ssh)
        else:
            err_msgs.append(entity.create_msg("login.os.failed").format("root"))
    finally:
        del root_account_infos


def modify_root_password(ssh, root_account_infos, context):
    """
    首次登录，修改root用户密码
    :param ssh: ssh链接
    :param root_account_infos:root用户信息
    :return: 密码输入错误，添加错误信息，提前返回
    """
    origin_infos.append(ssh.execCmdNoLog(root_account_infos.get("root_password")))
    time.sleep(10)
    ssh_ret = ssh.execCmdNoLog(root_account_infos.get("root_new_password"))
    origin_infos.append(ssh_ret)
    if not ssh_ret.strip().endswith("Retype new password:"):
        err_msgs.append(entity.create_msg("modify.password.failed").format("root"))
        return
    ssh_ret = ssh.execCmdNoLog(root_account_infos.get("root_new_password"))
    origin_infos.append(ssh_ret)
    if ssh_ret.strip().endswith("login:"):
        origin_infos.append(ssh.execCmd("root"))
        ssh_ret = ssh.execCmdNoLog(root_account_infos.get("root_new_password"))
    if ssh_ret.strip().endswith("]#"):
        modify_os_timezone(context, ssh)
        modify_os_password(root_account_infos, ssh)
    else:
        err_msgs.append(entity.create_msg("modify.os.password.failed").format("root"))


def modify_os_timezone(context, ssh):
    """
    修改时区
    :param context: 上下文
    :param ssh: ssh链接
    :return: 检查时区是否配置成功
    """
    deploy_node = context_util.get_deploy_node(context)
    timezone_infos = deploy_node.getConfigParam().getTimezone()
    logger.info("timezone_infos: {}".format(timezone_infos))
    try:
        exec_ssh_cmd_with_timeout_and_retry("rm -f /etc/localtime", 10, ssh)
        exec_ssh_cmd_with_timeout_and_retry("ln -s /usr/share/zoneinfo/{} /etc/localtime".format(
            timezone_infos), 10, ssh)
    except DeployException as de:
        logger.info("config timezone failed! exception:{}".format(de.message))
    return check_os_timezone(ssh, timezone_infos)


def exec_ssh_cmd_with_timeout_and_retry(cmd, timeout, ssh):
    """
    执行命令 超时时间+1次重试  该命令会再原始信息中打印原始命令！！！不可支持输入密码的命令
    :param cmd: 命令
    :param timeout 超时时间
    :return: 回显信息
    """
    for i in range(2):
        ssh_ret = ssh.execCmdWithTimout(cmd, timeout)
        if not is_cmd_time_out(ssh_ret):
            origin_infos.append(ssh_ret)
            return
        # 超时情况下增加 原始命令记录；正常情况回显会携带
        origin_infos.append(cmd)
        origin_infos.append(ssh_ret)
        logger.info("cmd execute timeout. Retry times{}. cmd:{} ".format(i, cmd))
        ssh.execCmd("\n")
    raise DeployException("exec cmd:{} failed after retry".format(cmd))


def check_os_timezone(ssh, timezone_infos):
    """
    通过“timedatectl”命令检查是否成功配置时区
    :param ssh: ssh链接
    :param timezone_infos: 时区信息
    :return: 成功配置时区后，任务通过；否则不通过
    """
    ssh_ret = ssh.execCmd("timedatectl")
    origin_infos.append(ssh_ret)
    match = re.findall(r'Time zone:\s+(\S+)\s+\(', ssh_ret)
    if not match or match[0] != timezone_infos:
        err_msgs.append(entity.create_msg("modify.os.timezone.failed"))


def modify_os_password(root_account_infos, ssh):
    """
    修改SSH密码
    :param root_account_infos:root用户信息
    :param ssh: ssh链接
    """
    new_passwd = root_account_infos.get("os_new_passwd")
    if root_account_infos.get("os_passwd") == new_passwd:
        logger.info("The new password is the same as the old password. You do not need to change it.")
        return
    os_username = root_account_infos.get("os_username")
    if os_username == "root":
        logger.info("ssh user is root, not need to modify.")
        return
    # 检查os指定用户是否存在
    time_out = 10 * 60
    start_time = time.time()
    while True:
        if time.time() - start_time > time_out:
            logger.info("Query timeout. The current OS user {} does not exist.".format(os_username))
            err_msgs.append(entity.create_msg("check.os.username.not.exist").format(os_username))
            return
        ssh_ret = ssh.execCmd("id {}".format(os_username))
        if "({})".format(os_username) in ssh_ret:
            break
        time.sleep(30)
    ssh_ret = ssh.execCmd("passwd {}".format(os_username))
    origin_infos.append(ssh_ret)
    if ssh_ret.strip().endswith("New password:"):
        ssh_ret = ssh.execCmdNoLog("{}".format(new_passwd))
        origin_infos.append(ssh_ret)
    if ssh_ret.strip().endswith("Retype new password:"):
        ssh_ret = ssh.execCmdNoLog("{}".format(new_passwd))
        origin_infos.append(ssh_ret)
    # 检查指令结果
    if "successfully" not in ssh_ret.strip():
        err_msgs.append(entity.create_msg("modify.ssh.user.password.failed"))
