# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
from Common.base import entity, context_util
from Common.base.entity import ResultFactory, DeployException
from Common.protocol.redfish.entity.resource_client import SystemsResourceClient, ManagerResourceClient

NOT_SUPPORT = 'not.support'


def execute(task):
    context = task.getJythonContext()
    result_msgs = []
    origin_infos = []
    param = task.getParam().getIpToDeviceInfoMap().get(context_util.get_deploy_ip(context))
    if not param:
        return ResultFactory.create_pass(origin_info=entity.create_msg(NOT_SUPPORT))
    asset_tag_result = patch_asset_tag(result_msgs, origin_infos, param, context)
    location_result = patch_location(result_msgs, origin_infos, param, context)
    if not asset_tag_result or not location_result:
        return ResultFactory.create_not_pass(origin_info=origin_infos, err_msg=result_msgs)
    return ResultFactory.create_pass(origin_info=origin_infos, err_msg=result_msgs)


def patch_location(result_msgs, origin_infos, param, context):
    """
    修改位置信息
    :param result_msgs: 结果信息
    :param origin_infos: 原始信息
    :param param: 参数
    :param context: 上下文
    :return:
    """
    location_result = entity.create_msg("location.result")
    location = param.getLocation()
    if not location:
        result_msgs.append(location_result.format(entity.create_msg(NOT_SUPPORT)))
        return True
    origin_info, error_msg = ProductLocation(context).patch_info(location)
    origin_infos.append(origin_info)
    if error_msg:
        result_msgs.append(location_result.format(error_msg))
        return False
    result_msgs.append(location_result.format(entity.create_msg("config.pass")))
    return True


def patch_asset_tag(result_msgs, origin_infos, param, context):
    """
    修改资产标签
    :param result_msgs: 结果信息
    :param origin_infos: 原始信息
    :param param: 参数
    :param context: 上下文
    :return:
    """
    asset_tag = param.getAssetTag()
    asset_tag_result = entity.create_msg("asset.tag.result")
    if not asset_tag:
        result_msgs.append(asset_tag_result.format(entity.create_msg(NOT_SUPPORT)))
        return True
    origin_info, error_msg = AssetTag(context).patch_info(asset_tag)
    origin_infos.append(origin_info)
    if error_msg:
        result_msgs.append(asset_tag_result.format(error_msg))
        return False
    result_msgs.append(asset_tag_result.format(entity.create_msg("config.pass")))
    return True


class AssetTag(object):
    def __init__(self, context):
        self._login_info = context_util.get_login_info(context)
        self._client = SystemsResourceClient(self._login_info, entity.create_logger(__file__))
        self._err_msg = []

    def patch_info(self, asset_tag):
        try:
            return str(self._client.patch_system_resource_info({"AssetTag": asset_tag})), ""
        except DeployException as exception:
            return exception.origin_info, exception.err_msg

    def get_err_msg(self):
        return self._err_msg


class ProductLocation(object):
    def __init__(self, context):
        self._login_info = context_util.get_login_info(context)
        self._client = ManagerResourceClient(self._login_info, entity.create_logger(__file__))
        self._origin_info = []

    def patch_info(self, location):
        try:
            param = self._build_location_param(location)
            if not param:
                self._origin_info, entity.create_msg("redfish.location.resource.not.found")
            return str(self._client.patch_manager_info(param)), ""
        except DeployException as exception:
            return exception.origin_info, exception.err_msg

    def _build_location_param(self, location):
        resource = self._client.get_assign_resource_info()
        self._origin_info.append(resource)
        oem_dict = resource.get("Oem", {})
        param_dict = {}
        for key, value in oem_dict.items():
            if "DeviceLocation" not in value.keys():
                continue
            device_location = {key: {"DeviceLocation": location}}
            param_dict["Oem"] = device_location
        return param_dict
