#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/08/07
@file: config_system_disk_cache_strategy.py
@function:
"""
import re

from Common.base import context_util
from Common.base import entity
from Common.base.constant import MsgKey
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util
from Common.protocol.redfish import redfish_util

PY_JAVA_ENV = py_java_env


def execute(task):
    logger = entity.create_logger(__file__)
    login_info = context_util.get_login_info(PY_JAVA_ENV)
    try:
        try:
            raid_type, origin_info = ssh_util.get_raid_card_type(PY_JAVA_ENV)
            if "SP686C-M-40i/SP686C-M-16i" == raid_type:
                return config_1880_pd_cache()
            if "SP186-M-8i" == raid_type or not raid_type:
                return ResultFactory.create_pass(origin_info, entity.create_msg("not.support"))
        except DeployException:
            # 1880卡为临时规避措施，如果ssh查询raid失败，则保持原有逻辑，redfish接口配置
            logger.info("try found 1880 raid but notfound")
        original_info = redfish_util.modify_logical_drive_attribute(
            login_info, "DriveCachePolicy", "Disabled", logger)
        return ResultFactory.create_pass(original_info)
    except DeployException as de:
        logger.error("Get http failed: {}".format(de.message))
        return ResultFactory.create_not_pass(
            de.origin_info, entity.create_msg(MsgKey.HTTP_ERROR))


def config_1880_pd_cache():
    origin_infos = []
    controller_id = query_controller_id(origin_infos)
    member_types = query_raid_member_type(origin_infos, controller_id)
    if is_all_close_status(origin_infos, member_types, controller_id):
        return ResultFactory.create_pass(origin_infos)
    change_status(origin_infos, controller_id)
    return ResultFactory.create_pass(origin_infos)


def parse_member(res):
    member_type = {}
    lines = res.splitlines()
    for line in lines:
        if "member" not in line:
            continue
        line_res = line.split()
        member_type.setdefault(line_res[1], []).append(line_res[2])
    return member_type


def is_all_close_status(origin_infos, member_types, controller_id):
    cmd = "/usr/bin/hiraidadm c{}:e{}:s{} show pdcache"
    for enc_id, slots in member_types.items():
        for slot in slots:
            res = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd.format(controller_id, enc_id, slot))
            origin_infos.append(res)
            on_status = re.findall(r"Drive Cache switch = On|Drive Cache switch = on", res)
            if on_status:
                return False
    return True


def query_controller_id(origin_infos):
    cmd = "/usr/bin/hiraidadm show allctrl"
    res = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
    origin_infos.append(res)
    controller_id = re.findall(r"Controller Id\s+\|\s+(\d)", res)
    if controller_id:
        return controller_id[0]
    raise DeployException(message=entity.create_msg("controller.id.query.error"), origin_info="\n".join(origin_infos))


def query_raid_member_type(origin_infos, controller_id):
    cmd = "/usr/bin/hiraidadm c{} show pdlist".format(controller_id)
    res = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
    origin_infos.append(res)
    return parse_member(res)


def close_cache(origin_infos, vd_id, controller_id):
    cmd = "/usr/bin/hiraidadm c{}:vd{} set pdcache sw=off".format(controller_id, vd_id)
    res = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
    origin_infos.append(res)


def change_status(origin_infos, controller_id):
    close_cache(origin_infos, find_vd_id(controller_id, origin_infos), controller_id)


def find_vd_id(controller_id, origin_infos):
    cmd = "/usr/bin/hiraidadm c{} show vdlist".format(controller_id)
    res = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd)
    origin_infos.append(res)
    find_head = False
    for line in res.splitlines():
        if "VDID" in line:
            find_head = True
            continue
        if find_head and re.findall(r"\d", line):
            return line.split()[0]
    raise DeployException(message=entity.create_msg("vdid.not.found"), origin_info="\n".join(origin_infos))
