#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2020/08/05
@file: config_system_password.py
@function:
"""
import copy
import time
from Common.base.constant import MsgKey
from Common.base import context_util
from Common.base import entity
from Common.base.entity import ResultFactory
from Common.base.entity import DeployException
from Common.protocol.ssh_util import get_ssh, SOL_TO_OS_CMD
from Common.factory.deploy_factory import DeployFactory
from Common.protocol.redfish.entity.resource_client import \
    ManagerResourceClient
from Business.adaptor.java_adaptor import get_j_exception

JException = get_j_exception()


def execute(task):
    return ConfigSystemPassword(task).do_config()


class ConfigSystemPassword(object):

    def __init__(self, task):
        self._task = task
        self._logger = entity.create_logger(__file__)
        self._login_info = context_util.get_login_info(task.getJythonContext())
        self._err_msgs = list()
        self._origin_infos = list()
        self._deploy_factory = DeployFactory(task.getJythonContext(), self._login_info, self._logger)

    def do_config(self):
        try:
            # 修改OS密码
            need_modify_infos = self._get_need_modify_os_password_infos()
            need_config_os = len(need_modify_infos) > 0
            if need_config_os:
                self._modify_os_password(need_modify_infos)
                # os密码修改晚后面立即遇重启可能未生效
                time.sleep(3)
        except JException:
            return ResultFactory.create_not_pass(
                self._origin_infos, entity.create_msg(
                    MsgKey.OBTAIN_INFO_FAILED))
        # 修改BMC密码
        need_config_bmc = self._need_modify_bmc_password()
        if need_config_bmc:
            self._modify_bmc_password()
        if len(self._err_msgs) != 0:
            return ResultFactory.create_not_pass(self._origin_infos,
                                                 self._err_msgs)
        return ResultFactory.create_pass(self._origin_infos)

    def _need_modify_bmc_password(self):
        if not self._login_info.had_new_password() or \
                self._login_info.password == self._login_info.new_password:
            return False
        try:
            if not self._deploy_factory.is_hg_storage():
                ManagerResourceClient(copy.copy(self._login_info), self._logger). \
                    get_assign_resource_path()
            return True
        except DeployException as de:
            self._logger.info("login bmc failed: {}".format(de.message))
            return False

    def _modify_bmc_password(self):
        try:
            res = self._deploy_factory.modify_user_password(self._login_info.username, self._login_info.new_password)
            self._origin_infos.append(res)
        except DeployException as de:
            self._logger.info("login bmc failed: {}".format(de.message))
            self._err_msgs.append(entity.create_msg(
                "modify.bmc.password.failed").format(
                self._login_info.username))
            self._origin_infos.append(de.origin_info)

    def _get_need_modify_os_password_infos(self):
        need_modify_infos = list()
        if not self._login_os_to_update_correct_password():
            return need_modify_infos
        deploy_node = context_util.get_deploy_node(self._task.getJythonContext())
        username = deploy_node.getOsUser().getUserName()
        password = deploy_node.getOsUser().getOriginPassword()
        new_password = deploy_node.getOsUser().getNewPassword()
        if new_password and password != new_password:
            self._logger.info("need modify {} password".format(username))
            need_modify_infos.append({"username": username,
                                      "new_password": new_password})

        root_password = deploy_node.getOsRootUser().getOriginPassword()
        root_new_password = deploy_node.getOsRootUser().getNewPassword()
        if root_new_password and root_password != root_new_password:
            self._logger.info("need modify {} password".format("root"))
            need_modify_infos.append({"username": "root",
                                      "new_password": root_new_password})
        return need_modify_infos

    def _login_os_to_update_correct_password(self):
        # 先获取连接，退出OS登录状态
        status, ssh = self._get_ssh()
        if not status or not self._exit_to_login(ssh):
            return False
        # 再次获取连接，重新登录OS，触发用户登录密码尝试，更新设备中的正确密码
        status, ssh = self._get_ssh()
        return status

    def _exit_to_login(self, ssh):
        """
        退出登录状态到login
        :param ssh: ssh连接
        :return: 退出成功否
        """
        # 退10次
        for _ in range(20):
            ssh_ret = ssh.execCmd("exit")
            self._origin_infos.append(ssh_ret)
            if ssh_ret.strip().endswith("iBMC:/->"):
                ssh_ret = ssh.execCmd(SOL_TO_OS_CMD)
            if ssh_ret.strip().endswith("login:"):
                return True
            # 睡眠下，反应可能较慢
            time.sleep(1)
        self._err_msgs.append(entity.create_msg(MsgKey.OBTAIN_INFO_FAILED))
        return False

    def _modify_os_password(self, need_modify_infos):
        status, ssh = self._get_ssh()
        if not status:
            return
        for modify_info in need_modify_infos:
            self._origin_infos.append(ssh.execCmd("passwd {}".format(
                modify_info.get("username"))))
            ssh_ret = ""
            # 修改密码需要连续输入两次新密码，第二次是确认密码
            for password in [modify_info.get("new_password")] * 2:
                if ssh_ret.strip().endswith("~]#"):
                    break
                ssh_ret = ssh.execCmdNoLog(password)
                self._origin_infos.append(ssh_ret)
            self._check_modify_password(ssh_ret, modify_info.get("username"))

    def _check_modify_password(self, ssh_info, username):
        if "all authentication tokens updated successfully" not in ssh_info:
            self._err_msgs.append(entity.create_msg(
                "modify.os.password.failed").format(username))

    def _get_ssh(self):
        try:
            return True, get_ssh(self._task.getJythonContext())
        except DeployException as de:
            self._logger.error("modify OsPassword failed{}".format(de.message))
            self._origin_infos.append(de.origin_info)
            self._err_msgs.append(de.err_msg)
            return False, None
