#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

"""
@time: 2023/08/25
@file: post_configuration_check.py
@function:
"""
import time

from Common.base import entity, context_util
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
LOGGER = entity.create_logger(__file__)


def execute(task):
    check_items = (CheckPortLink, CheckPortBond,)
    ssh_rets = list()
    err_msgs = list()
    res_list = []
    try:
        for item in check_items:
            check_obj = item(task, ssh_rets, err_msgs)
            # 检查物理端口是否为空，为空说明网络配置跳过，直接报不通过
            if isinstance(check_obj, CheckPortLink) and check_obj.is_port_names_empty():
                err_msgs.append(entity.create_msg("check.port.no.port.name.fail"))
                res_list.append(False)
                break
            res_list.append(check_obj.check())
        if not all(res_list):
            return ResultFactory.create_not_pass(ssh_rets, err_msgs)
        return ResultFactory.create_pass(ssh_rets, err_msgs)
    except DeployException as exception:
        LOGGER.error(exception.message)
        return ResultFactory.create_not_pass(exception.origin_info, exception.err_msg)


class CheckPortBase(object):
    def __init__(self, task, ssh_rets, err_msgs, port_names):
        self._task = task
        self._ssh_rets = ssh_rets
        self._err_msgs = err_msgs
        self._port_names = port_names

    def check(self):
        # 检查端口列表是否为空，可能存在没有绑定端口的情况
        if self.is_port_names_empty():
            self._err_msgs.append(self.get_port_names_empty_msg())
            return True
        try:
            return self.is_all_port_status_up()
        except DeployException as exception:
            LOGGER.error(exception.message)
            self._ssh_rets.append(exception.origin_info)
            self._err_msgs.append(exception.err_msg)
            self._task.openAutoRetry()
            return False

    def is_all_port_status_up(self):
        no_port_up_port_names = self.get_all_port_down_port_name()
        if not no_port_up_port_names:
            self._err_msgs.append(self.get_port_check_pass_msg())
            return True
        # 尝试一次，重试三次（每次等待5秒）
        for _ in range(3):
            LOGGER.info("[{}] not link up, wait 5s to retry".format(str(no_port_up_port_names)))
            time.sleep(5)
            no_port_up_port_names = self.get_all_port_down_port_name()
            if not no_port_up_port_names:
                self._err_msgs.append(self.get_port_check_pass_msg())
                return True
        self._err_msgs.append(self.get_port_check_not_pass_msg(no_port_up_port_names))
        self._task.openManualIgnored()
        return False

    def get_all_port_down_port_name(self):
        """
        获取所有的端口未启动状态的端口名称列表
        :return: 端口名称列表
        """
        return []

    def get_port_check_pass_msg(self):
        """
        获取端口检查通过的描述信息，entity.create_msg("xxx")
        :return: 描述信息
        """
        pass

    def get_port_check_not_pass_msg(self, port_names):
        """
        获取端口检查不通过的描述信息：entity.create_msg("xxxx").format(", ".join(port_names))
        :param port_names: 未启动的端口名称列表
        :return: 描述信息
        """
        pass

    def get_port_names_empty_msg(self):
        """
        获取端口名称列表为空的描述信息，entity.create_msg("xxx")
        :return: 描述信息
        """
        pass

    def is_port_names_empty(self):
        """
        检查端口是否为空
        :return: 端口是否为空
        """
        return not bool(self._port_names)


class CheckPortLink(CheckPortBase):
    def __init__(self, task, ssh_rets, err_msgs):
        super(CheckPortLink, self).__init__(task, ssh_rets, err_msgs, context_util.get_port_list(PY_JAVA_ENV))

    def get_port_check_pass_msg(self):
        return entity.create_msg("check.port.link.pass")

    def get_port_names_empty_msg(self):
        return entity.create_msg("check.port.link.empty")

    def get_port_check_not_pass_msg(self, port_names):
        return entity.create_msg("port.not.link.up").format(", ".join(port_names))

    def get_all_port_down_port_name(self):
        no_link_up_port_names = list()
        for port_name in self._port_names:
            status = self._get_port_link_status(port_name)
            if status.lower() != "yes":
                no_link_up_port_names.append(port_name)
        return no_link_up_port_names

    def _get_port_link_status(self, port_name):
        cmd = "ethtool {}".format(port_name)
        cli_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd, True, 180)
        self._ssh_rets.append(cli_ret)
        for line in cli_ret.splitlines():
            if "Link detected:" in line:
                return line.split(":")[1].strip()
        return ""


class CheckPortBond(CheckPortBase):
    """
    检查绑定端口的状态: cat /proc/net/bonding/bond0接口状态,检查 MII Status是否为up状态
    """

    def __init__(self, task, ssh_rets, err_msgs):
        super(CheckPortBond, self).__init__(task, ssh_rets, err_msgs, context_util.get_port_bond_list(PY_JAVA_ENV))

    def get_port_check_pass_msg(self):
        return entity.create_msg("check.port.bond.pass")

    def get_port_names_empty_msg(self):
        return entity.create_msg("check.port.bond.empty")

    def get_port_check_not_pass_msg(self, port_names):
        return entity.create_msg("port.not.bond.up").format(", ".join(port_names))

    def get_all_port_down_port_name(self):
        no_bond_up_port_names = list()
        for port_name in self._port_names:
            status = self._get_port_bond_status(port_name)
            if status.lower() != "up":
                no_bond_up_port_names.append(port_name)
        return no_bond_up_port_names

    def _get_port_bond_status(self, port_name):
        cmd = "cat /proc/net/bonding/{}".format(port_name)
        cli_ret = ssh_util.exec_ssh_cmd(PY_JAVA_ENV, cmd, True, 180)
        self._ssh_rets.append(cli_ret)
        for line in cli_ret.splitlines():
            if "MII Status:" in line:
                return line.split(":")[1].strip()
        return ""
