#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/20
@file: reboot_server.py
@function:
"""
from Common.base.constant import Platform
from Common.base.entity import ResultFactory
from Common.base import entity, context_util
from Common.protocol import ssh_util
from Common.base.entity import DeployException

PY_JAVA_ENV = py_java_env
LOGGER = entity.create_logger(__file__)
OS_REBOOT_CMD = "reboot"
SOL_SERVICE_NAME = "console=ttyS0"
GRUB_FILE = "/etc/default/grub"
GRUB_RESOURCE_FILE = "/boot/efi/EFI/euleros/grub.cfg"
PARAM_LINE_FLAG = "GRUB_CMDLINE_LINUX"


def execute(task):
    ssh_rets = list()
    try:
        add_sol_server_to_boot_param()
        entity.AutoBrushProgressTask(task, 360).start()
        ssh_util.exec_ssh_cmd_without_return(PY_JAVA_ENV, OS_REBOOT_CMD)
        # 循环等待重启
        login_success, reboot_ssh_rets = ssh_util.reboot_had_succeeded(
            PY_JAVA_ENV)
        ssh_rets.extend(reboot_ssh_rets)
        if not login_success:
            return ResultFactory.create_not_pass(ssh_rets, entity.create_msg("reboot.server.failed"))
        return ResultFactory.create_pass(ssh_rets, err_msg=entity.create_msg("progress.finish"))
    except DeployException as exception:
        LOGGER.error(exception.message)
        return ResultFactory.create_not_pass(exception.origin_info,
                                             exception.err_msg)


def add_sol_server_to_boot_param():
    """
    801版本，镜像安装后默认自带串口服务启动，但是后续重启后，串口服务启动参数没有在配置文件中，导致重启后没有自动开启串口服务
    工具在重启之前提前将启动参数添加到配置文件再重启，确保工具重启后能正常识别设备状态
    仅限X86平台
    :return:
    """
    if is_no_need_to_set():
        return
    cmd = "cat {}".format(GRUB_FILE)
    result = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
    for invalid in ssh_util.CMD_INVALID_KEYS:
        if invalid in result:
            # 没有配置文件的场景，忽略配置
            return
    if SOL_SERVICE_NAME in result:
        return
    write_param_to_file()
    resource_config()


def write_param_to_file():
    """
    在文件中替换指定字符串
    sed -i 's/origin/replace/g file'
    sed 替换命令
    -i 参数，表示在文件中替换
    s 作用范围，前可跟具体的行号，例如6s，表示在第六行替换
    origin 需要被替换的字符串
    replace 替换的字符串
    g 替换所有目标
    :return:
    """
    cmd = "sed -i 's/GRUB_CMDLINE_LINUX=\"/GRUB_CMDLINE_LINUX=\"{} /g' {}".format(SOL_SERVICE_NAME, GRUB_FILE)
    ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)


def is_no_need_to_set():
    current_platform = context_util.get_platform_id(PY_JAVA_ENV)
    current_version = context_util.get_mapping_version(PY_JAVA_ENV)
    LOGGER.info("sol server---current platform :{}, current version:{}".format(current_platform, current_version))
    return Platform.X86_64 != current_platform or not current_version.startswith("8.0.1")


def resource_config():
    cmd = "grub2-mkconfig -o {}"
    ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd.format(GRUB_RESOURCE_FILE), timeout=60)
