#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/11
@file: context_util.py
@function:
"""
from Business.adaptor.java_adaptor import get_ip_pattern, get_context
from Common.base.constant import Platform, HAI_YAN_DEV_TYPES, SingleItemStatus
from Common.protocol.redfish.entity.ip import Ipv4Addresses, Ipv6Addresses
from Common.protocol.redfish.entity.login import LoginInfo
from Common.base.entity import DeployException
from com.huawei.ism.tool.distributeddeploy.entity import VersionKeyEnum

IpPattern = get_ip_pattern()

NORMALIZED_DRIVER_KEYS = ["pacific", "atlantic", "x86", "arm"]
VERSION_MAX_LINE_LENGTH = 100


def get_dev_node(context):
    return get_deploy_node(context).createDevNodeInfo()


def get_deploy_node(context):
    return context.get("deployNode")


def get_config_os_name(context):
    return get_deploy_node(context).getNodeName()


def contain_need_check_key(context, version_keys):
    for key in version_keys:
        if get_mapping_attribute(context, key):
            return True
    return False


def get_mapping_strategy(context):
    strategy_map = context.get("mappingStrategy")
    if strategy_map.isEmpty():
        return ""

    return strategy_map.get(get_platform_id(context))


def get_mapping_attribute(context, key):
    platform_strategy = get_mapping_strategy(context)
    if not platform_strategy:
        return ""

    value = platform_strategy.findAttributeValue(key)

    if value == "--":
        return ""
    return value


def get_mapping_attribute_url(context, key):
    platform_strategy = get_mapping_strategy(context)
    if not platform_strategy:
        return ""

    url = platform_strategy.findAttributeUrl(key)
    if url == "--":
        return ""
    return url


def get_mapping_source_file(context):
    return context.get("sourceFile")


def get_mapping_fw_version_with_haiyan(context, key):
    """
    获取配套的固件版本，包含海燕机型的判断
    :param context: 上下文
    :param key: mapping key
    :return: 配套版本
    """
    platform_id = get_platform_id(context)
    dev_type = get_dev_type(context)
    if platform_id == Platform.X86_64 and dev_type in HAI_YAN_DEV_TYPES:
        key = "{}_HaiYan".format(key)
    return get_mapping_attribute(context, key)


def get_mapping_fw_url_with_haiyan(context, key):
    """
    获取配套的固件版本，包含海燕机型的判断
    :param context: 上下文
    :param key: mapping key
    :return: 配套版本
    """
    platform_id = get_platform_id(context)
    dev_type = get_dev_type(context)
    if platform_id == Platform.X86_64 and dev_type in HAI_YAN_DEV_TYPES:
        key = "{}_HaiYan".format(key)
    return get_mapping_attribute_url(context, key)


def get_platform_id(context):
    """
    获取设备平台id，例如x86_64，在strategy.xml中配置
    :param context: 上下文
    :return: 设备平台id
    """
    return context.get("PlatformId", "")


def is_ocean_platform(context):
    """
    判断平台id是否是atlantic，pacific，eastSea,sinan
    :param context:
    :return:
    """
    platform_id = get_platform_id(context)
    return platform_id in (Platform.PACIFIC, Platform.EAST_SEA,
                           Platform.ATLANTIC, Platform.NEW_ATLANTIC, Platform.SI_NAN, Platform.NEW_SI_NAN)


def is_atlantic_ssm_platform(context):
    """
    判断平台ID是否为大西洋直通版（数据集群模块）
    :param context: 上下文
    :return: 判断结果
    """
    return get_platform_id(context) == Platform.SSM_ATLANTIC


def is_east_sea_or_sinan(context):
    """
    判断平台id是否是eastSea,sinan,这两种专有硬件存在1880raid卡/软Raid两种情况，检查不是1880卡的情况默认是软Raid，
    跳过raid卡型号检查/系统盘缓存策略配置
    :param context:
    :return:
    """
    platform_id = get_platform_id(context)
    return platform_id in (Platform.EAST_SEA, Platform.SI_NAN, Platform.NEW_SI_NAN)


def set_platform_id(context, platform_id):
    context["PlatformId"] = platform_id


def get_dev_type(context):
    """
    设备类型，例如5288 V5，在strategy.xml中配置
    :param context: 上下文
    :return: 设备类型
    """
    return context.get("DevType", "")


def set_dev_type(context, platform_id):
    context["DevType"] = platform_id


def set_port_list(context, port_list):
    context["portList"] = port_list


def get_port_list(context):
    return context.get("portList", [])


def set_port_bond_list(context, port_bond_list):
    """
    记录绑定端口的名称列表
    :param context: 上下文
    :param port_bond_list: 绑定端口名称
    :return: None
    """
    context["portBondList"] = port_bond_list


def get_port_bond_list(context):
    """
    获取绑定端口的名称列表
    :param context: 上下文
    :return: 绑定端口列表
    """
    return context.get("portBondList", [])


def get_login_info(context):
    deploy_node = get_deploy_node(context)
    ip = deploy_node.getIpAddress().getIp()
    user_name = deploy_node.getBmcUser().getUserName()
    password = deploy_node.getBmcUser().getOriginPassword()
    new_password = deploy_node.getBmcUser().getNewPassword()
    return LoginInfo(ip, user_name, password, new_password=new_password)


def get_target_bmc_ip_address(context):
    ip_address = get_deploy_node(context).getIpAddress()
    return __get_ip_address(ip_address)


def get_target_bmc_ip_address_ds(context):
    ip_address_ds = get_deploy_node(context).getIpAddressDS()
    if not ip_address_ds:
        return None
    return __get_ip_address(ip_address_ds)


def __get_ip_address(ip_address):
    if ip_address.getIpPattern() is IpPattern.IPV4:
        return Ipv4Addresses(ip_address.getIp(),
                             ip_address.getNetMaskOrPrefix(),
                             ip_address.getGateway())
    elif ip_address.getIpPattern() is IpPattern.IPV6:
        return Ipv6Addresses(ip_address.getIp(),
                             ip_address.getNetMaskOrPrefix(),
                             ip_address.getGateway())
    raise DeployException("un valid ip pattern: {}".format(
        ip_address.getIpPattern().toString()))


def get_os_network_strategy(context):
    return get_deploy_node(context).getOsNetworkStrategys()


def contain_normalized_driver_key(ver_info):
    for driver_key in NORMALIZED_DRIVER_KEYS:
        if driver_key in ver_info:
            return True
    return False


def get_current_mapping_version():
    return get_context().INS.getCurrentMappingVersion()


def get_products(context):
    return context.get("products")


def get_mapping_version(context):
    return context.get("mappingVersion")


def get_scene(context):
    return context.get("scene", "")


def get_deploy_ip(context):
    return get_deploy_node(context).getIp()


def get_iso_image_path(context):
    if get_deploy_node(context).getCpuArchitecture() == "ARM":
        return context.get("armIsoPath", "")
    return context.get("x86IsoPath", "")


def get_version_key_enum():
    return VersionKeyEnum


def get_pass_key():
    return SingleItemStatus.ITEM_PASS


def get_not_pass_key():
    return SingleItemStatus.ITEM_NOT_PASS


def get_node_sn(context):
    return get_deploy_node(context).getSn()


def update_node_sn(context, sn):
    return get_deploy_node(context).setSn(sn)


def get_mapping_version_desc(context):
    return context.get("mappingVersionDesc")
