#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/06/06
@file: base_util.py
@function:
"""
import re
import threading
import time

from Business.adaptor.java_adaptor import get_cbb_logger
from Common.base.resource import MESSAGES_DICT
from constant import ItemStatus, MsgKey
from constant import LOGGER, LANG

Logger = get_cbb_logger()


# 默认构建的日志打印器是没有敏感日志框架处理的。注意：不允许直接打印敏感信息。
def create_logger(py_path, is_ignore_sensitive=True):
    return Logger(LOGGER, py_path, is_ignore_sensitive)


def create_msg(msg_key):
    if msg_key not in MESSAGES_DICT:
        return "--"
    return MESSAGES_DICT.get(msg_key).get(LANG)


def create_source_file_msg(context, error_msg):
    source_file = context.get("sourceFile")
    if not source_file:
        return error_msg
    return u"{}\n\n{}".format(error_msg, create_msg("current.mapping.file").format(source_file))


def build_url_error_msg(url, error_msg, version_match=False):
    if version_match:
        return error_msg
    if not url:
        return error_msg
    return u"{}\n{}\n".format(error_msg, create_msg("match.download.url").format(url))


def build_fw_tool_tips():
    return create_msg("use.fw.tool.to.upgrade") + "\n"


def build_driver_tool_tips():
    return create_msg("use.driver.tool.to.upgrade") + "\n"


def build_os_tool_tips():
    return create_msg("use.os.tool.to.install") + "\n"


def get_digit_str(str_ver):
    match_ret = re.compile("\d+").findall(str_ver)
    if not match_ret:
        return "0"
    return "".join(match_ret)


class DeployException(Exception):
    class ErrCode(object):
        """
        异常的错误码
        NONE: 通常错误
        MAY_INFO_MISS: 工具在通过sol登录OS，执行命令过程中可能会出现命令回显缺失的
        情况，遇到这个情况可以抛异常，并使用该错误码标记，在item脚本中统一捕获异常，
        打开task中的自动重试功能
        if exception.may_info_miss():
        task.openAutoRetry()
        """
        NONE = "0"
        MAY_INFO_MISS = "1"
        BMC_DEFAULT_PASSWORD = "2"

    def __init__(self, message, origin_info="", err_msg="",
                 err_code=ErrCode.NONE):
        self.message = message
        self.origin_info = origin_info
        self.err_msg = err_msg
        self.err_code = err_code

    @property
    def err_msg(self):
        if not self._err_msg:
            self.err_msg = create_msg(MsgKey.OBTAIN_INFO_FAILED)
        return self._err_msg

    @err_msg.setter
    def err_msg(self, err_msg):
        self._err_msg = err_msg

    def may_info_miss(self):
        return self.err_code == DeployException.ErrCode.MAY_INFO_MISS


class ResultFactory(object):

    @staticmethod
    def __join_err_msg(err_msg):
        if isinstance(err_msg, list):
            return Join.join_with_lf(err_msg)
        return err_msg

    @staticmethod
    def __join_origin_info(origin_info):
        if isinstance(origin_info, list):
            return Join.join_with_2lf(origin_info)
        return origin_info

    @staticmethod
    def create_pass(origin_info="", err_msg=""):
        return ItemStatus.PASS, ResultFactory.__join_origin_info(
            origin_info), ResultFactory.__join_err_msg(err_msg)

    @staticmethod
    def create_not_pass(origin_info="", err_msg=""):
        return ItemStatus.NOT_PASS, ResultFactory.__join_origin_info(
            origin_info), ResultFactory.__join_err_msg(err_msg)


class Compare(object):
    @staticmethod
    def compare_digital_version(version1, version2):
        if not version1:
            return -1 if version2 else 0
        elif not version2:
            return 1

        def str_to_int(str_ver):
            try:
                return int(get_digit_str(str_ver))
            except ValueError:
                raise DeployException(
                    "not int version {}".format(str_ver),
                    err_code=DeployException.ErrCode.MAY_INFO_MISS)

        match = re.compile(r'[.-]')
        version1_sections = match.split(version1)
        version2_sections = match.split(version2)
        for index in range(
                max(len(version1_sections), len(version2_sections))):
            if index == len(version1_sections):
                version1_sections.append("0")
            if index == len(version2_sections):
                version2_sections.append("0")
            if str_to_int(version1_sections[index]) > str_to_int(
                    version2_sections[
                        index]):
                return 1
            elif str_to_int(version1_sections[index]) < str_to_int(
                    version2_sections[index]):
                return -1
        return 0


class Join(object):
    @staticmethod
    def __join(info_list, join_str):
        def is_empty_str(string):
            return string and string.strip()

        return join_str.join(list(filter(is_empty_str, info_list)))

    @staticmethod
    def join_with_2lf(info_list):
        return Join.__join(info_list, "\n\n")

    @staticmethod
    def join_with_lf(info_list):
        return Join.__join(info_list, "\n")


class AutoBrushProgressTask(threading.Thread):

    def __init__(self, task, total_time):
        self.task = task
        self.total_time = total_time
        super(AutoBrushProgressTask, self).__init__()

    def run(self):
        progress = 0
        interval_time = int(self.total_time / 20)
        while not self.task.isFinish() and progress < 95:
            progress += 5
            self.task.updateProgress(progress)
            time.sleep(interval_time)


class CheckCommon(object):
    def __init__(self, task):
        self.task = task
        self._logger = create_logger(__file__)
        self._ssh_rets = []
        self._err_msgs = []
        self.not_support_msg = create_msg(MsgKey.NOT_INVOLVE)

    def check_version(self):
        """
        检查版本主体逻辑
        """
        raise NotImplementedError

    def get_match_msg(self):
        """
        获取当前版本的配套资源文件
        """
        raise NotImplementedError

    def obtain_match_versions(self):
        """
        通过Version_Mapping.xml文件中的配套关系初始化当前配套版本
        """
        raise NotImplementedError

    def check(self):
        self.obtain_match_versions()
        match_msg = self.get_match_msg()
        try:
            res = self.check_version()
            self._err_msgs.append(match_msg)
            if not res:
                self._err_msgs.insert(0, build_driver_tool_tips())
                return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
            return ResultFactory.create_pass(self._ssh_rets, self._err_msgs)
        except DeployException as exception:
            self._logger.error(exception.message)
            self._ssh_rets.append(exception.origin_info)
            self._err_msgs.append(match_msg)
            self._err_msgs.append("\n" + exception.err_msg)
            if exception.may_info_miss():
                self.task.openAutoRetry()
            return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
