#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

from Common.base.constant import HG_BOARD_TYPE
from Common.base.entity import DeployException
from Common.protocol.redfish import redfish_util
import re


def get_board_type(cmd, ssh):
    res = str(ssh.execCmdNoLog(cmd))
    if not res:
        return '', res
    res_list = res.splitlines()
    for item in res_list:
        if 'Board Product' in item:
            spl = item.split()
            return spl[-1], res
    return '', res


class DeployFactory(object):
    def __init__(self, context, login_info, logger):
        self._deploy_step = CommonDeployStep(context, login_info, logger)
        self._board_type = ''

    def get_product_unique_id(self):
        return self._deploy_step.get_product_unique_id()

    def modify_user_password(self, username, new_password):
        return self._deploy_step.modify_user_password(username, new_password)

    def modify_bmc_host_name(self, config_hostname_name):
        return self._deploy_step.modify_bmc_host_name(config_hostname_name)

    def is_hg_storage(self):
        return self._board_type == HG_BOARD_TYPE


class CommonDeployStep(object):

    def __init__(self, context, login_info, logger):
        self._context = context
        self._login_info = login_info
        self._logger = logger

    def get_product_unique_id(self):
        return redfish_util.get_product_unique_id(self._login_info, self._logger)

    def modify_user_password(self, username, new_password):
        return redfish_util.modify_user_password(self._login_info, username, new_password, self._logger)

    def modify_bmc_host_name(self, config_hostname_name):
        return redfish_util.modify_bmc_host_name(self._login_info, config_hostname_name, self._logger)


class HgDeployStep(object):
    OLD_PASSWORD_RE = re.compile(r'(?<=-P ).+(?= user)')
    NEW_PASSWORD_RE = re.compile(r'(?<=password \d ).+(?= ?\n?Set)')

    def __init__(self, context, ssh, login_info, logger, board_type, type_cmd_info):
        self._context = context
        self._ssh = ssh
        self._login_info = login_info
        self._logger = logger
        self._board_type = board_type
        self._type_cmd_info = type_cmd_info

    def get_product_unique_id(self):
        return self._board_type.lower(), self._type_cmd_info

    def modify_user_password(self, username, new_password):
        cmd = 'ipmitool -H {} -I lanplus -C 17 -U {} -P {} user list'.format(self._login_info.ip, username,
                                                                             self._login_info.password)
        res = str(self._ssh.execCmdNoLog(cmd))
        user_no = None
        if res:
            res_lines = res.splitlines()
            for line in res_lines:
                if username in line:
                    user_no = line.split()[0]
        if user_no:
            cmd = 'ipmitool -H {} -I lanplus -C 17 -U {} -P {} user set password {} {}' \
                .format(self._login_info.ip,
                        username,
                        self._login_info.password,
                        user_no, new_password)
        else:
            raise DeployException('no user {}'.format(username), origin_info=res, err_msg="")
        pass_res = str(self._ssh.execCmdNoLog(cmd))
        return HgDeployStep.NEW_PASSWORD_RE.sub('***', HgDeployStep.OLD_PASSWORD_RE.sub('***', pass_res))

    def modify_bmc_host_name(self, config_hostname_name):
        name_hex = ''
        for item in config_hostname_name:
            name_hex += hex(ord(item)) + ' '
        cmd = 'ipmitool raw 0x2e 0xc0 0xcc 0xcb 0x00 {}'.format(name_hex.strip())
        return str(self._ssh.execCmd(cmd))
