#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/06
@file: http_error.py
@function:
"""
import json
from Common.base import entity
from Common.base.constant import MsgKey
from Common.base.entity import DeployException
from Common.protocol.redfish.entity.login import LoginInfo

INITIAL_PASSWORD_KEYS = [
    "This password is an initial password or has been changed by another administrator",
    "iBMC.1.0.PasswordChangeRequired"
]


class HttpCode(object):
    SUCCESS = 200
    CREATED_SUCCESS = 201
    TASK_RUNNING = 202
    UNAUTHORIZED = 401
    FORBIDDEN = 403
    PRECONDITION_FAILED = 412
    INTERNAL_SERVER_ERROR = 500
    NOT_IMPLEMENTED = 501


class HttpResponse(object):
    def __init__(self, code, resource, headers):
        self.code = code
        self.resource = resource
        self.headers = headers


def has_extended_info(response_obj):
    """
    redifsh接口返回信息中含有@Message.ExtendedInfo，但是http_code是200的情况下代表配置信息部分写入成功，总体上应该上报配置失败
    :param response_obj: redfish请求返回值，已转换json对象
    :return: 错误信息
    """
    if "@Message.ExtendedInfo" not in response_obj.keys():
        return ""
    return response_obj.get("@Message.ExtendedInfo", [])


def has_initial_password_error(response):
    for key in INITIAL_PASSWORD_KEYS:
        if key in response:
            return True
    return False


def deal_response(func):
    """
    处理转换http响应结果
    """

    def do_func(self, *args, **kwargs):
        logger = args[0] if isinstance(self, LoginInfo) else self._logger
        try:
            response = func(self, *args, **kwargs)
        except DeployException:
            raise
        except Exception as e:
            logger.error("http exception: {}".format(e))
            raise DeployException(str(e))
        error_code = response.status
        if error_code in [HttpCode.SUCCESS, HttpCode.CREATED_SUCCESS, HttpCode.TASK_RUNNING]:
            response_obj = json.loads(response.data.decode('utf8'))
            extend_info = has_extended_info(response_obj)
            if extend_info:
                raise DeployException("redfish config error", origin_info=str(extend_info),
                                      err_msg=entity.create_msg(MsgKey.HTTP_ERROR))
            return HttpResponse(response.status, response_obj, response.headers)
        elif error_code == HttpCode.PRECONDITION_FAILED:
            raise DeployException("http 412 error", origin_info="Precondition failed",
                                  err_msg=entity.create_msg(MsgKey.HTTP_ERROR))
        if has_initial_password_error(response.data.decode('utf8')):
            raise DeployException("default password", err_code=DeployException.ErrCode.BMC_DEFAULT_PASSWORD)
        raise DeployException("http error", origin_info=str(json.loads(
            response.data.decode('utf8'))), err_msg=entity.create_msg(
            MsgKey.HTTP_ERROR))

    return do_func
