#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/05
@file: http_client.py
@function:
"""
import json
import urllib3
import copy

from deal_response import deal_response

CREATED_SESSION_RESOURCE_PATH = "/redfish/v1/SessionService/Sessions"


class HttpClient(object):
    HEADERS_DICT = dict()
    SESSION_DICT = dict()
    # 忽略认证告警
    urllib3.disable_warnings()
    # 沿用原有逻辑，忽略证书校验
    HTTP_POOL = urllib3.PoolManager(cert_reqs="CERT_NONE")

    def __init__(self, login_info, logger):
        self._login_info = login_info
        self._logger = logger
        self._headers = HttpClient.HEADERS_DICT.get(self._login_info.ip)
        if not self._headers:
            self.create_headers()
        self.vendor = self._get_vendor()

    @staticmethod
    def release_session(login_info, logger, timeout=30):
        try:
            if not HttpClient.HEADERS_DICT.get(login_info.ip):
                logger.info("ip: {} has no session".format(login_info.ip))
                return
            header = HttpClient.HEADERS_DICT.pop(login_info.ip)
            if not HttpClient.SESSION_DICT.get(login_info.ip):
                logger.info("ip: {} has no session".format(login_info.ip))
            session_id = HttpClient.SESSION_DICT.pop(login_info.ip)
            url = HttpClient.build_delete_url(login_info, session_id)
            response = HttpClient.HTTP_POOL.request("DELETE", url, headers=header, timeout=timeout)
            logger.info("DELETE Response:\n[{}] {}".format(response.status, response.data))
        except Exception as e:
            logger.error("delete session for :{} failed :{}".format(login_info.ip, e))

    @staticmethod
    def build_delete_url(login_info, session_id):
        return "https://{}:{}{}/{}".format(login_info.ip, login_info.port, CREATED_SESSION_RESOURCE_PATH, session_id)

    @staticmethod
    def has_auth_token(ip):
        """
        判断是否存在token
        @:param ip BMC IP地址
        :return: 是否存在token
        """
        headers = HttpClient.HEADERS_DICT.get(ip)
        return headers and bool(headers.get("X-Auth-Token"))

    @deal_response
    def request_get(self, resource_path, time_out=60):
        """
        get请求
        :param resource_path: 资源路径
        :param time_out: 超时时间
        :return: 结果
        """

        url = self._combine_url(resource_path)
        self._logger.info("GET Request: " + url)
        response = self.HTTP_POOL.request("GET", url,
                                          headers=self._headers,
                                          timeout=time_out)
        self._logger.info("GET Response:\n[{}] {}".format(
            response.status, response.data))
        return response

    @deal_response
    def request_post(self, resource_path, data, time_out=60, log_params=True):
        """
        post请求
        :param log_params: 是否打印请求参数
        :param resource_path: 资源路径
        :param data: 请求参数
        :param time_out: 超时时间
        :return: 结果
        """
        return self._request_post(resource_path, data, time_out, log_params)

    def _request_post(self, resource_path, data, time_out=60, log_params=True):
        """
        post请求
        :param log_params: 是否打印请求参数
        :param resource_path: 资源路径
        :param data: 请求参数
        :param time_out: 超时时间
        :return: 结果
        """
        url = self._combine_url(resource_path)
        data = self._deal_request_data(data)
        self._logger.info("POST Request: " + url)
        if log_params:
            self._logger.info("POST Request Params: " + str(data))
        response = self.HTTP_POOL.request("POST", url,
                                          headers=self._headers,
                                          body=data,
                                          timeout=time_out)
        self._logger.info("POST Response:\n[{}] {}".format(
            response.status, response.data))
        return response

    @deal_response
    def request_delete(self, resource_path, time_out=60):
        """
        delete请求
        :param resource_path: 资源路径
        :param time_out: 超时时间
        :return: 结果
        """
        url = self._combine_url(resource_path)
        self._logger.info("DELETE Request: " + url)
        response = self.HTTP_POOL.request("DELETE", url,
                                          headers=self._headers,
                                          timeout=time_out)
        self._logger.info("DELETE Response:\n[{}] {}".format(
            response.status, response.data))
        return response

    def request_patch(self, resource_path, data, time_out=60,
                      log_params=True):
        """
        patch请求
        :param log_params: 是否打印data日志
        :param resource_path: 子资源路径
        :param data: 请求参数
        :param time_out: 超时时间
        :return: 结果
        """
        return self.request_patch_with_etag(resource_path, data, None, time_out, log_params)

    @deal_response
    def request_patch_with_etag(self, resource_path, data, e_tag_param, time_out=60,
                                log_params=True):
        """
        patch请求
        :param log_params: 是否打印data日志
        :param resource_path: 子资源路径
        :param e_tag_param: patch请求的if-match头
        :param data: 请求参数
        :param time_out: 超时时间
        :return: 结果
        """
        data = self._deal_request_data(data)
        headers = copy.copy(self._headers)
        e_tag = e_tag_param
        if not e_tag:
            e_tag = self._get_resource_etag(resource_path)
        headers["If-Match"] = e_tag
        return self.do_request_patch(data, headers, log_params, resource_path, time_out)

    def do_request_patch(self, data, headers, log_params, resource_path, time_out):
        url = self._combine_url(resource_path)
        self._logger.info("PATCH Request: " + url)
        if log_params:
            self._logger.info("PATCH Request Params: " + str(data))
        response = self.HTTP_POOL.request("PATCH", url,
                                          headers=headers,
                                          body=data,
                                          timeout=time_out)
        self._logger.info("PATCH Response:\n[{}] {}".format(
            response.status, response.data))
        return response

    def get_patch_response_etag(self, resource_path, data, time_out=60, log_params=True):
        """
        获取patch请求所需要的etag标签，只用于特定的场景
        特定场景：开局交付时，bmc处于默认密码的状态，登录需要强制改密码，所以直接发送修改用户密码的patch请求，第一次会失败，但是会返回etag
        第二次再次发送patch请求，将本次的etag带上则可修改密码成功
        :param resource_path: 请求路径
        :param data: 数据
        :param time_out: 超时时间
        :param log_params: 是否记录日志
        :return:
        """

        data = self._deal_request_data(data)
        headers = copy.copy(self._headers)
        return self.do_request_patch(data, headers, log_params, resource_path, time_out).headers.get("Etag")

    def _combine_url(self, resource_path):
        """
        组合拼接ip、端口、资源路径
        :param resource_path: 资源路径
        :return: 组合的URL
        """
        if resource_path.startswith("/"):
            return "https://{}:{}{}".format(
                self._login_info.ip, self._login_info.port, resource_path)
        return "https://{}:{}/{}".format(
            self._login_info.ip, self._login_info.port, resource_path)

    def _get_resource_etag(self, resource_path):
        """
        获取资源etag
        :param resource_path: 资源路径
        :return: 结果
        """
        response = self.request_get(resource_path)
        return response.headers.get("Etag")

    def get_auth_token(self):
        """
        获取token
        :return: token
        """
        response = self.create_session()
        session_id = json.loads(response.data.decode('utf8')).get("Id")
        HttpClient.SESSION_DICT[self._login_info.ip] = session_id
        return response.headers.get("X-Auth-Token")

    def create_session(self):
        data = {"UserName": self._login_info.username,
                "Password": self._login_info.password}
        return self._request_post(
            CREATED_SESSION_RESOURCE_PATH, json.dumps(data), log_params=False)

    def _get_vendor(self):
        response = self.request_get("/redfish/v1/")
        return response.resource.get("Oem").keys()[0]

    def create_headers(self):
        """
        添加token到消息头中
        """
        auth_token = self.get_auth_token()
        if not auth_token:
            self._logger.error("[{}] get auth token failed.".format(self._login_info.ip))
            self._headers = {}
            return
        self._headers = {'content-type': 'application/json',
                         "X-Auth-Token": auth_token}
        HttpClient.HEADERS_DICT[self._login_info.ip] = self._headers

    def _deal_request_data(self, data):
        """
        处理请求参数为str
        :param data: 请求参数
        :return: 请求参数str
        """
        if isinstance(data, str):
            return data
        return json.dumps(data)
