#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/08
@file: resource_client.py
@function:
"""
from Common.protocol.redfish.entity.http import HttpClient

from Common.base.entity import DeployException


class ResourcePath(object):
    MANAGER = "/redfish/v1/Managers/"
    ETHERNET_INTERFACE = MANAGER + "{}/EthernetInterfaces"
    ASSIGN_ETHERNET_INTERFACE = ETHERNET_INTERFACE + "{}"
    SYSTEMS = "/redfish/v1/Systems/"
    STORAGE_PATH_SUFFIX = "/Storages/"
    BIOS = SYSTEMS + "{}/Bios/"
    BIOS_SET = BIOS + "Settings"
    SERVER_REBOOT = SYSTEMS + "{}/Actions/ComputerSystem.Reset"
    USER_COLLECTS = "/redfish/v1/AccountService/Accounts/"
    SYSTEM_OVERVIEW = "/redfish/v1/SystemOverview/"
    CHASSIS = "/redfish/v1/Chassis/"
    CHASSIS_BOARDS = CHASSIS + "{}/Boards/"
    CHASSIS_MAIN_BOARD = CHASSIS + "{}/Boards/chassisMainboard"
    CHASSIS_BY_ID = "/redfish/v1/Chassis/{}"
    NET_WORK_ADAPTER = "/redfish/v1/Chassis/{}/NetworkAdapters"
    BIOS_PASSWORD = "{}/Bios/Actions/Bios.ChangePassword"
    CURRENT_VIRTUAL_MEDIA = MANAGER + "{}/VirtualMedia/CD"
    UMOUNT_CURRENT_VIRTUAL_MEDIA_HUAWEI = CURRENT_VIRTUAL_MEDIA + "/Oem/Huawei/Actions/VirtualMedia.VmmControl"
    UMOUNT_CURRENT_VIRTUAL_MEDIA_PUBLIC = CURRENT_VIRTUAL_MEDIA + "/Oem/Public/Actions/VirtualMedia.VmmControl"
    UPDATE_SERVICE = "/redfish/v1/UpdateService/"
    UPDATE_SERVICE_FIRMWARE_INVENTORY = UPDATE_SERVICE + "FirmwareInventory/{}"


class ManagerResourceClient(HttpClient):
    """
    管理资源
    """

    def __init__(self, login_info, logger):
        super(ManagerResourceClient, self).__init__(login_info, logger)

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 资源路径
        """
        response = self.request_get(ResourcePath.MANAGER)
        path = response.resource.get("Members")[0].get("@odata.id")
        self._logger.info("assign managers resource path: {}".format(path))
        return path

    def get_assign_resource_info(self):
        """
        获取指定资源的信息
        :return: 资源信息
        """
        path = self.get_assign_resource_path()
        return self.request_get(path).resource

    def get_ethernet_interface_gather_resource_info(self):
        """
        获取网口集合资源信息
        :return: 资源信息
        """
        resource_path = self.get_assign_resource_path()
        resource_id = resource_path.split("/")[-1]
        response = self.request_get(
            ResourcePath.ETHERNET_INTERFACE.format(resource_id))
        return response.resource

    def get_assign_ethernet_interfaces_resource_path(self):
        """
        获取指定网口的资源访问路径
        :return: 路径
        """
        info = self.get_ethernet_interface_gather_resource_info()
        path = info.get("Members")[0].get("@odata.id")
        self._logger.info(
            "managers ethernet interface path: {}".format(path))
        return path

    def get_assign_ethernet_interfaces_info(self):
        """
        获取指定网口的资源信息
        :return: 信息
        """
        resource_path = self.get_assign_ethernet_interfaces_resource_path()
        response = self.request_get(resource_path)
        return response.resource

    def current_virtual_media(self):
        """
        查询当前挂载镜像信息
        :return: 挂载镜像信息
        """
        resource_id = self.get_assign_resource_path().split("/")[-1]
        response = self.request_get(
            ResourcePath.CURRENT_VIRTUAL_MEDIA.format(resource_id), 5 * 60)
        return response.resource

    def umount_current_virtual_media(self, param):
        """
        卸载当前挂载镜像
        :return: 卸载结果
        """
        resource_id = self.get_assign_resource_path().split("/")[-1]
        if self.current_virtual_media().get("Oem").get("Public"):
            response = self.request_post(
                ResourcePath.UMOUNT_CURRENT_VIRTUAL_MEDIA_PUBLIC.format(resource_id), param)
        else:
            response = self.request_post(
                ResourcePath.UMOUNT_CURRENT_VIRTUAL_MEDIA_HUAWEI.format(resource_id), param)
        return response.resource

    def patch_assign_ethernet_interfaces_info(self, data):
        """
        修改自定网口的资源信息
        :param data: 修改参数
        :return: 修改结果
        """
        resource_path = self.get_assign_ethernet_interfaces_resource_path()
        return self.request_patch(resource_path, data).resource

    def get_sp_server_info(self):
        resource_path = self.get_assign_resource_path()
        sp_server_path = "{}/SPService".format(resource_path)
        return self.request_get(sp_server_path).resource

    def patch_manager_info(self, param):
        """
        修改管理资源

        :param param: 参数，字典对象
        :return: result
        """
        return self.request_patch(self.get_assign_resource_path(), param).resource


class SystemsResourceClient(HttpClient):
    """
    系统资源
    """

    def __init__(self, login_info, logger):
        super(SystemsResourceClient, self).__init__(login_info, logger)
        self.storages_path = self.get_storages_path()

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 路径
        """
        response = self.request_get(ResourcePath.SYSTEMS)
        path = response.resource.get("Members")[0].get("@odata.id")
        self._logger.info("assign systems resource path: {}".format(path))
        return path

    def get_assign_resource_info(self):
        """
        获取指定资源的信息
        :return: 资源信息
        """
        path = self.get_assign_resource_path()
        return self.request_get(path).resource

    def get_storages_path(self):
        response = self.request_get(ResourcePath.SYSTEMS)
        members_list = response.resource.get("Members")
        return members_list[0].get("@odata.id") + ResourcePath.STORAGE_PATH_SUFFIX

    def get_storage_resource_paths(self):
        response = self.request_get(self.storages_path)
        resources = response.resource.get("Members")
        return [resource.get("@odata.id") for resource in
                resources], response.resource

    def get_storage_resource_infos(self):
        original_infos = list()
        resource_paths, resource = self.get_storage_resource_paths()
        original_infos.append(str(resource))
        resource_info_dict = dict()
        for path in resource_paths:
            storage_id = path.split("/")[-1]
            resource = self.request_get(path).resource
            resource_info_dict[storage_id] = resource
            original_infos.append(str(resource))
        return resource_info_dict, "\n\n".join(original_infos)

    def patch_assign_storage_resource_info(self, storage_id, data):
        path = self.storages_path + storage_id
        response = self.request_patch(path, {
            "StorageControllers": [{"Oem": {self.vendor: data}}]})
        return str(response.resource)

    def get_storage_logical_drive_resource_infos(self):
        original_infos = list()
        resource_paths, resource = self.get_storage_resource_paths()
        original_infos.append(str(resource))
        storage_logical_drive_resources = dict()
        for path in resource_paths:
            resource = self.request_get(path + "/Volumes").resource
            original_infos.append(str(resource))
            for member in resource.get("Members"):
                logical_drive_path = member.get("@odata.id")
                resource = self.request_get(logical_drive_path).resource
                storage_logical_drive_resources[logical_drive_path] = resource
                original_infos.append(str(resource))
        return storage_logical_drive_resources, "\n\n".join(original_infos)

    def patch_storage_logical_drive_attribute(self, logical_drive_path, data):
        params = {"Oem": {self.vendor: data}}
        return str(self.request_patch(logical_drive_path, params, 180).resource)

    def get_bios_info(self):
        """
        获取系统BIOS信息
        :return: BIOS信息
        """
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        bios_resource_path = ResourcePath.BIOS.format(resource_id)
        response = self.request_get(bios_resource_path)
        return response.resource.get("Attributes"), response.resource

    def patch_bios_info(self, data):
        """
        修改系统BIOS信息
        :param data: 修改参数
        :return: 修改结果
        """
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        bios_set_path = ResourcePath.BIOS_SET.format(resource_id)
        response = self.request_patch(
            bios_set_path, {"Attributes": data})
        return response.resource

    def patch_bios_password(self, data):
        """
        修改系统BIOS密码
        :param data: 修改参数
        :return: 修改结果
        """
        path = self.get_assign_resource_path()
        return self.request_post(ResourcePath.BIOS_PASSWORD.format(path), data, log_params=False).resource

    def patch_system_resource_info(self, param):
        """
        修改系统资源的接口
        :param param: 参数
        :return: 接口返回值
        """
        path = self.get_assign_resource_path()
        return self.request_patch(path, param).resource

    def reboot_server(self):
        """
        重启服务器
        """
        path = self.get_assign_resource_path()
        resource_path = ResourcePath.SERVER_REBOOT.format(path.split("/")[-1])
        reboot_data = {"ResetType": "ForcePowerCycle"}
        return self.request_post(resource_path, reboot_data).resource

    def set_power_state(self, power_state):
        """
        设置服务器的电源状态，可上电，下电，强制下电
        :param power_state: On：上电, ForceOff：强制下电, GracefulShutdown：正常下电, ForceRestart：强制重启,
        Nmi：触发不可屏蔽中断, ForcePowerCycle：强制下电再上电
        :return: 接口返回值
        """
        path = self.get_assign_resource_path()
        resource_path = ResourcePath.SERVER_REBOOT.format(path.split("/")[-1])
        reboot_data = {"ResetType": power_state}
        return self.request_post(resource_path, reboot_data).resource


class AccountServiceClient(HttpClient):

    def __init__(self, login_info, logger):
        super(AccountServiceClient, self).__init__(login_info, logger)

    def get_assign_user_resource_path(self, user_name):
        return self.get_assign_user_resource_path_by_type("UserName", user_name)

    def get_assign_user_resource_path_by_type(self, key_type, value):
        response = self.request_get(ResourcePath.USER_COLLECTS)
        for member in response.resource.get("Members"):
            if member.get(key_type) == value:
                path = member.get("@odata.id")
                self._logger.info("assign user resource path: {}".format(path))
                return path
        raise DeployException("get assign user resource path failed")

    def patch_assign_user_info(self, user_name, data):
        resource_path = self.get_assign_user_resource_path(user_name)
        return self.request_patch(resource_path, data, log_params=False). \
            resource

    def post_user_account(self, user):
        return self.request_post(ResourcePath.USER_COLLECTS, user, log_params=False).resource

    def delete_user_account(self, user_id):
        return self.request_delete(ResourcePath.USER_COLLECTS + user_id).resource

    def get_all_users(self):
        return self.request_get(ResourcePath.USER_COLLECTS).resource

    def patch_assign_user_info_by_id(self, user_id, user_info):
        path = self.get_assign_user_resource_path_by_type("Id", user_id)
        return self.request_patch(path, user_info, log_params=False).resource

    def modify_bmc_default_password(self, user_id, user_info):
        resource_path = ResourcePath.USER_COLLECTS + user_id
        etag = self.get_patch_response_etag(resource_path, user_info, log_params=False)
        return self.request_patch_with_etag(resource_path, user_info, etag, log_params=False).resource


class SystemOverviewResourceClient(HttpClient):
    """
    只有Arm服务器只支持
    """

    def __init__(self, login_info, logger):
        super(SystemOverviewResourceClient, self).__init__(login_info, logger)

    def get_system_overview_info(self):
        return self.request_get(ResourcePath.SYSTEM_OVERVIEW).resource

    def get_systems_info(self):
        return self.request_get(ResourcePath.SYSTEMS).resource


class ChassisResourceClient(HttpClient):

    def __init__(self, login_info, logger):
        super(ChassisResourceClient, self).__init__(login_info, logger)

    def get_assign_resource_path(self):
        """
        获取指定资源的访问路径
        :return: 路径
        """
        response = self.request_get(ResourcePath.CHASSIS)
        path = response.resource.get("Members")[0].get("@odata.id")
        self._logger.info("assign chassis resource path: {}".format(path))
        return path

    def get_chassis_Members(self):
        """
        获取所有机箱资源资源
        :return: 路径
        """
        response = self.request_get(ResourcePath.CHASSIS)
        return response.resource.get("Members")

    def get_resource_by_blade(self, blade_id):
        return self.request_get(ResourcePath.CHASSIS_BY_ID.format(
            blade_id)).resource

    def modify_resource_state_by_blade(self, blade_id,
                                       resource_id, state):
        params = {resource_id: state}
        return self.request_patch(ResourcePath.CHASSIS_BY_ID.format(
            blade_id), params)

    def get_boards_resource_info(self):
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        return self.request_get(ResourcePath.CHASSIS_BOARDS.format(
            resource_id)).resource

    def get_main_board_resource_info(self):
        path = self.get_assign_resource_path()
        resource_id = path.split("/")[-1]
        return self.request_get(ResourcePath.CHASSIS_MAIN_BOARD.format(
            resource_id)).resource

    def get_assign_board_infos(self, board_name):
        resource = self.get_boards_resource_info()
        if "Oem" in resource:
            return resource.get("Oem").values()[0].get("Boards").get(
                board_name)

    def get_network_adapter(self, resource_id):
        """
        通过redfish查询当前的网卡信息
        @:param resource_id 系统ID，通用服务器一般为1，专有服务器一般为Blade1
        :return: 网卡信息
        """
        return self.request_get(ResourcePath.NET_WORK_ADAPTER.format(resource_id)).resource

    def modify_resource_by_url(self, resource_url, param):
        """
        修改当前资源路径下的信息
        :param resource_url: 当前资源
        :param param: 需要修改的目标信息，是已经完成组装的正确格式的json数据
        :return: 响应信息
        """
        return self.request_patch(resource_url, param)


class UpdateServiceClient(HttpClient):
    """
    升级服务的客户端
    """

    def __init__(self, login_info, logger):
        super(UpdateServiceClient, self).__init__(login_info, logger)

    def get_firmware_inventory_infos(self, type_name):
        """
        查询指定可升级固件资源信息
        :param type_name: 类型，如ActiveBMC、BackupBMC、AvailableBMC、ActiveUboot、
        BackupUboot、Bios、MainBoardCPLD、MainboardVRD、mainboardFanBoardCPLD等
        :return: 响应信息
        """
        return self.request_get(ResourcePath.UPDATE_SERVICE_FIRMWARE_INVENTORY.format(type_name)).resource
