#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/05
@file: redfish_util.py
@function: redfish接口
log参数要放在最后
"""
import traceback

from com.huawei.ism.tool.distributeddeploy.logic.config import ProductStrategy

from Common.base import entity, context_util
from Common.base.entity import DeployException
from Common.protocol import ssh_util
from Common.protocol.redfish.entity.http import HttpClient
from entity.ip import IpAddresses
from entity.login import LoginInfo
from entity.resource_client import ManagerResourceClient, \
    SystemsResourceClient, AccountServiceClient, SystemOverviewResourceClient, \
    ChassisResourceClient, UpdateServiceClient


def __trans_2_deploy_exception(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except DeployException:
            entity.create_logger(__file__).error(traceback.format_exc())
            raise
        except Exception as exception:
            entity.create_logger(__file__).error(traceback.format_exc())
            raise DeployException(exception.message)

    return wrapper


def session_timeout(origin_info):
    return "There is no valid session established with the implementation" in origin_info


def __try_new_password(func):
    @__trans_2_deploy_exception
    def wrapper(*args, **kwargs):
        login_info = args[0]
        logger = args[-1]
        if not isinstance(login_info, LoginInfo):
            raise DeployException("the first param should be LoginInfo.")
        try:
            ManagerResourceClient(login_info, logger).get_assign_resource_path()
        except DeployException as exception:
            if session_timeout(exception.origin_info) and HttpClient.has_auth_token(login_info.ip):
                # 若session超时，则清除缓存重试，重新创建session
                logger.info("release cache session")
                HttpClient.release_session(login_info, logger)
                ManagerResourceClient(login_info, logger).get_assign_resource_path()
            elif exception.err_code == DeployException.ErrCode.BMC_DEFAULT_PASSWORD:
                raise exception
            else:
                if login_info.had_new_password():
                    login_info.update_password_to_new()
                    return func(*args, **kwargs)
                raise exception
        except Exception:
            if login_info.had_new_password():
                login_info.update_password_to_new()
                return func(*args, **kwargs)
            raise
        return func(*args, **kwargs)

    return wrapper


def __build_ip_param(ip_address, ip_address_ds, client):
    resource = client.request_get(client.get_assign_ethernet_interfaces_resource_path()).resource
    oem = resource.get("Oem", {})
    if not oem:
        return __create_redfish_data(ip_address, ip_address_ds)
    oem_value = oem.get(oem.keys()[0], {})
    if "IPVersion" not in oem_value.keys():
        # 如果当前数据oem下没有ipversion这个key，就保持原有逻辑
        return __create_redfish_data(ip_address, ip_address_ds)
    ip_version = oem_value.get("IPVersion")
    if "IPv4AndIPv6" in ip_version:
        # 如果已经是ipv4和ipv6都支持的场景，就不修改网络协议
        return __create_redfish_data(ip_address, ip_address_ds)
    return __create_redfish_data(ip_address, ip_address_ds, oem.keys()[0], ip_version)


def __create_redfish_data(ip_address, ip_address_ds, oem_key="", ip_version=""):
    data = {}
    if ip_address:
        data.update(ip_address.create_redfish_data(oem_key, ip_version))
    if ip_address_ds:
        data.update(ip_address_ds.create_redfish_data(oem_key, ip_version))
    return data


def __string_default_if_empty(str_text, default=""):
    """
    如果字符串为空，则返回默认值
    :param str_text: 待检查字符串
    :param default: 默认值，默认为空串
    :return: 非None的字符串
    """
    return str_text if str_text else default


@__try_new_password
def set_bmc_ip(login_info, ip_address, ip_address_ds, logger):
    """
    设置BMC IP
    :param login_info: BMC登录信息 LoginInfo
    :param ip_address: 主IP信息
    :param ip_address_ds: 备IP信息(设置双栈时使用,可空)
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("start set bmc ip.")
    if not isinstance(ip_address, IpAddresses) or (ip_address_ds and not isinstance(ip_address_ds, IpAddresses)):
        raise DeployException("the ip addresses data should be IpAddresses.")
    client = ManagerResourceClient(login_info, logger)
    resource = client.patch_assign_ethernet_interfaces_info(__build_ip_param(ip_address, ip_address_ds, client))
    login_info.ip = ip_address.ip
    return str(resource)


@__try_new_password
def get_blade_id(login_info, logger):
    """
     获取blade（节点、机箱）的资源id，兼容有hmm（数据集群模块）的服务器
     太平洋北冰洋东海设备的members返回值会多一个enc，兼容适配
     :param login_info: BMC登录信息 LoginInfo
     :param logger: 日志
     :raise: DeployException 获取失败异常
     """
    logger.info("start get blade or hmm id.")
    client = ChassisResourceClient(login_info, logger)
    members_list = client.get_chassis_Members()
    for members in members_list:
        path = members.get("@odata.id")
        if "HMM" in path or "Blade" in path:
            return str(path.split("/")[-1])
    return str(members_list[0].get("@odata.id").split("/")[-1])


@__try_new_password
def get_indicator_LED_state(login_info, blade_or_hmm_id, logger):
    """
    获取机箱定位指示灯状态
    :param login_info: BMC登录信息 LoginInfo
    :param blade_or_hmm_id: blade（节点、机箱）或者hmm（数据集群模块）资源的id
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("start get indicator LED state.")
    client = ChassisResourceClient(login_info, logger)
    resource = client.get_resource_by_blade(blade_or_hmm_id)
    return str(resource.get("IndicatorLED"))


@__try_new_password
def modify_indicator_LED_state(login_info, blade_or_hmm_id, state, logger):
    """
    修改机箱定位指示灯状态
    :param login_info: BMC登录信息 LoginInfo
    :param blade_or_hmm_id: blade（节点、机箱）或者hmm（数据集群模块）资源的id
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("modify lit indicator LED.")
    client = ChassisResourceClient(login_info, logger)
    resource = client.modify_resource_state_by_blade(blade_or_hmm_id,
                                                     "IndicatorLED",
                                                     state)
    return str(resource.resource)


@__try_new_password
def get_one_bios_item_info(login_info, bios_key, logger):
    """
    获取某一Bios信息
    :param login_info: BMC登录信息 LoginInfo
    :param bios_key: Bios项名
    :param logger: 日志
    :return: Bios项的值
    :raise: DeployException 获取失败异常
    """
    logger.info("start get one item bios info.")
    client = SystemsResourceClient(login_info, logger)
    bios_info, resource = client.get_bios_info()
    if bios_info and bios_key in bios_info:
        value = bios_info.get(bios_key)
        logger.info("[One Bios Info] {} : {}".format(bios_key, value))
        return value, str(resource)
    return None, str(resource)


@__try_new_password
def get_all_bios_info(login_info, logger):
    """
    获取所有Bios信息
    :param login_info: BMC登录信息 LoginInfo
    :param logger: 日志
    :return: bios信息 dict
    :raise: DeployException 获取失败异常
    """
    logger.info("start get all bios info.")
    client = SystemsResourceClient(login_info, logger)
    bios_info, resource = client.get_bios_info()
    logger.info("[All Bios Info] {} ".format(bios_info))
    return bios_info, str(resource)


@__try_new_password
def set_one_bios_info(login_info, bios_key, value, logger):
    """
    设置某一bios项
    :param login_info: BMC登录信息 LoginInfo
    :param bios_key: bios key
    :param value: bios value
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("start set one bios info.")
    client = SystemsResourceClient(login_info, logger)
    return str(client.patch_bios_info({bios_key: value}))


@__try_new_password
def set_some_bios_info(login_info, bios_info_dict, logger):
    """
    设置某些bios项
    :param login_info: BMC登录信息 LoginInfo
    :param bios_info_dict: 设置bios信息 dict
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("start set some bios info.")
    client = SystemsResourceClient(login_info, logger)
    return str(client.patch_bios_info(bios_info_dict))


@__try_new_password
def get_product_unique_id(login_info, logger):
    """
    获取服务器product unique id
    :param login_info: BMC登录信息 LoginInfo
    :param logger: 日志
    :return: productUniqueId
    :raise: DeployException 获取失败异常
    """
    logger.info("start get ProductUniqueId.")
    client = ManagerResourceClient(login_info, logger)
    info = client.get_assign_resource_info()
    try:
        unique_id = info.get("Oem").values()[0].get("ProductUniqueID")
        logger.info("[ProductUniqueID] {}".format(unique_id))
        return str(unique_id).lower(), str(info)
    except AttributeError as error:
        raise DeployException("Get unique id error: {}".format(error))


@__try_new_password
def reboot_server(login_info, logger):
    """
    重启服务器
    :param login_info: BMC登录信息 LoginInfo
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("start reboot server.")
    client = SystemsResourceClient(login_info, logger)
    return str(client.reboot_server())


@__try_new_password
def set_power_state(login_info, power_state, logger):
    """
    重启服务器
    :param login_info: BMC登录信息 LoginInfo
    :param power_state: power_state: On：上电, ForceOff：强制下电, GracefulShutdown：正常下电, ForceRestart：强制重启,
        Nmi：触发不可屏蔽中断, ForcePowerCycle：强制下电再上电
    :param logger: 日志
    :raise: DeployException 获取失败异常
    """
    logger.info("set server power state : {}.".format(power_state))
    client = SystemsResourceClient(login_info, logger)
    return str(client.set_power_state(power_state))


@__try_new_password
def modify_user_password(login_info, user_name, new_password, logger):
    logger.info("modify user {} password".format(user_name))
    client = AccountServiceClient(login_info, logger)
    modify_data = {"Password": new_password}
    return str(client.patch_assign_user_info(user_name, modify_data))


@__try_new_password
def modify_bmc_host_name(login_info, host_name, logger):
    logger.info("modify host_name: {}".format(host_name))
    client = ManagerResourceClient(login_info, logger)
    modify_data = {"HostName": host_name}
    return str(client.patch_assign_ethernet_interfaces_info(modify_data))


@__try_new_password
def modify_storage_attribute(login_info, attribute_key, target_value, logger):
    logger.info("start modify storage attribute: {}/{}".format(
        attribute_key, target_value))
    client = SystemsResourceClient(login_info, logger)
    original_infos = list()
    resource_info_dict, original_info = client.get_storage_resource_infos()
    original_infos.append(original_info)
    for storage_id, resource in resource_info_dict.items():
        for storage_ctrl_info in resource.get("StorageControllers"):
            huawei_info = storage_ctrl_info.get("Oem").values()[0]
            if attribute_key in huawei_info and huawei_info.get(attribute_key) is not None and \
                    huawei_info.get(attribute_key) != target_value:
                logger.info("should modify storage {}".format(storage_id))
                original_info = client.patch_assign_storage_resource_info(
                    storage_id, {attribute_key: target_value})
                original_infos.append(original_info)

    return "\n\n".join(original_infos)


@__try_new_password
def get_storage_raid_model_type_attributes(login_info, logger):
    logger.info("start to get storage raid model and type attribute")
    client = SystemsResourceClient(login_info, logger)
    resource_info_dict, original_info = client.get_storage_resource_infos()
    raid_model_types = list()
    for storage_id, resource in resource_info_dict.items():
        for storage_ctrl_info in resource.get("StorageControllers"):
            huawei_info = storage_ctrl_info.get("Oem").values()[0]
            raid_model_types.append({
                "ID": storage_id,
                "Model": __string_default_if_empty(storage_ctrl_info.get("Model")),
                "Type": __string_default_if_empty(huawei_info.get("Type"))
            })
    return raid_model_types, original_info


@__try_new_password
def modify_logical_drive_attribute(login_info, attribute_key, target_value,
                                   logger):
    logger.info("start modify logical drive attribute: {}/{}".format(
        attribute_key, target_value))
    client = SystemsResourceClient(login_info, logger)
    storage_logical_drive_resources, original_info = client. \
        get_storage_logical_drive_resource_infos()
    original_infos = [original_info]
    for path, resource in storage_logical_drive_resources.items():
        if need_patch_target_value(resource, attribute_key, target_value):
            logger.info("should modify logical drive: {}".format(path))
            original_info = client.patch_storage_logical_drive_attribute(path, {attribute_key: target_value})
            original_infos.append(original_info)
    return "\n\n".join(original_infos)


def need_patch_target_value(resource, attribute_key, target_value):
    if resource.get(attribute_key) is not None and resource.get(attribute_key) != target_value:
        return True
    oem_map = resource.get("Oem", {})
    if not oem_map:
        return False
    for key, item in oem_map.items():
        if isinstance(item, dict) and item.get(attribute_key) is not None \
                and resource.get(attribute_key) != target_value:
            return True
    return False


@__try_new_password
def get_system_overview_managers_info(login_info, attribute_key, logger):
    """
    获取系统汇总管理属性信息
    :param login_info: 设备登录信息
    :param attribute_key: 属性key
    :param logger: 日志
    :return:
    """
    client = SystemOverviewResourceClient(login_info, logger)
    resource_info = client.get_system_overview_info()
    attribute = resource_info.get("Managers")[0].get(attribute_key)
    return attribute if attribute else "", str(resource_info)


@__try_new_password
def get_managers_info(login_info, logger):
    return ManagerResourceClient(login_info, logger).get_assign_resource_info()


@__try_new_password
def get_systems_info(login_info, logger):
    return SystemsResourceClient(login_info, logger).get_assign_resource_info()


@__try_new_password
def get_system_overview_systems_info(login_info, attribute_key, logger):
    client = SystemOverviewResourceClient(login_info, logger)
    resource_info = client.get_system_overview_info()
    attribute = resource_info.get("Systems")[0].get(attribute_key)
    return attribute if attribute else "", str(resource_info)


@__try_new_password
def get_ibmc_main_board_num(login_info, logger):
    client = ManagerResourceClient(login_info, logger)
    info = client.get_assign_resource_info()
    manager_type_2_main_bord_num = {"iBMC": "1710", "iBMC3": "1711"}
    manager_type = info.get("Model")
    if manager_type in manager_type_2_main_bord_num:
        return manager_type_2_main_bord_num.get(manager_type), str(info)
    raise DeployException("unidentifiable manager type", origin_info=str(info))


@__try_new_password
def set_bios_boot_cd_target(login_info, logger):
    # 部分ibmc版本不支持修改bios启动配置OverrideMode，需要兼容
    client = SystemsResourceClient(login_info, logger)
    boot_sequence = {
        "BootSourceOverrideEnabled": "Once",
        "BootSourceOverrideTarget": "Cd",
    }
    if client.get_assign_resource_info().get("Boot").get("BootSourceOverrideMode") != "UEFI":
        boot_sequence["BootSourceOverrideMode"] = "UEFI"
    return str(client.patch_system_resource_info({"Boot": boot_sequence}))


@__try_new_password
def current_virtual_media(login_info, logger):
    client = ManagerResourceClient(login_info, logger)
    return client.current_virtual_media()


@__try_new_password
def umount_current_virtual_media(login_info, param, logger):
    client = ManagerResourceClient(login_info, logger)
    return client.umount_current_virtual_media(param)


@__try_new_password
def get_main_board_info(login_info, attribute_key, logger):
    client = ChassisResourceClient(login_info, logger)
    resource = client.get_main_board_resource_info()
    if attribute_key in resource:
        attribute = resource.get(attribute_key)
        return attribute if attribute else "", str(resource)
    raise DeployException("not found main bord attribute {}".format(
        attribute_key), origin_info=str(resource))


@__try_new_password
def get_firmware_inventory_version(login_info, type_name, logger):
    client = UpdateServiceClient(login_info, logger)
    resource = client.get_firmware_inventory_infos(type_name)
    if resource and resource.get("Version"):
        return resource.get("Version"), str(resource)
    raise DeployException("not found firmware inventory {} version.".format(
        type_name), origin_info=str(resource))


@__try_new_password
def get_disk_back_board_info(login_info, attribute_key, logger):
    """
    获取背板属性信息
    :param login_info: 设备登录信息
    :param attribute_key: 属性key
    :param logger: 日志
    :return: {"Description"： 属性， ...}
    """
    client = ChassisResourceClient(login_info, logger)
    disk_back_board_infos = dict()
    origin_infos = list()
    infos = client.get_assign_board_infos("DiskBackplane")
    if not infos:
        return disk_back_board_infos, "\n\n".join(origin_infos)
    origin_infos.append(str(infos))
    for link in infos.get("Links"):
        link_uri = link.get("@odata.id")
        resource = client.request_get(link_uri).resource
        origin_infos.append(str(resource))
        description = resource.get("Description")
        attribute = resource.get(attribute_key)
        disk_back_board_infos[description] = attribute if attribute else ""
    return disk_back_board_infos, "\n\n".join(origin_infos)


@__try_new_password
def get_raid_card_board_info(login_info, attribute_key, logger):
    """
    获取raid卡板属性信息
    :param login_info: 设备登录信息
    :param attribute_key: 属性key
    :param logger: 日志
    :return: {"Description"： 属性， ...}
    """
    client = ChassisResourceClient(login_info, logger)
    raid_card_board_infos = dict()
    origin_infos = list()
    infos = client.get_assign_board_infos("RAIDCard")
    if not infos:
        return raid_card_board_infos, "\n\n".join(origin_infos)
    origin_infos.append(str(infos))
    for link in infos.get("Links"):
        link_uri = link.get("@odata.id")
        resource = client.request_get(link_uri).resource
        origin_infos.append(str(resource))
        description = resource.get("Description")
        attribute = resource.get(attribute_key)
        raid_card_board_infos[description] = attribute if attribute else ""
    return raid_card_board_infos, "\n\n".join(origin_infos)


@__try_new_password
def get_io_board_info(login_info, attribute_key, logger):
    """
    获取io板属性信息
    :param login_info: 设备登录信息
    :param attribute_key: 属性key
    :param logger: 日志
    :return: 属性 list()
    """
    client = ChassisResourceClient(login_info, logger)
    io_board_infos = list()
    origin_infos = list()
    infos = client.get_assign_board_infos("IOBoard")
    if not infos:
        return io_board_infos, "\n\n".join(origin_infos)
    origin_infos.append(str(infos))
    for link in infos.get("Links"):
        link_uri = link.get("@odata.id")
        resource = client.request_get(link_uri).resource
        origin_infos.append(str(resource))
        info = resource.get(attribute_key)
        io_board_infos.append(info if info else "")
    return io_board_infos, "\n\n".join(origin_infos)


@__try_new_password
def get_sp_server_info(login_info, logger):
    return ManagerResourceClient(login_info, logger).get_sp_server_info()


@__try_new_password
def get_net_cards(login_info, logger):
    """
    获取网卡信息
    :param login_info: 登录信息
    :param logger:日志
    :return:
    """
    path = SystemsResourceClient(login_info, logger).get_assign_resource_path()
    resource_id = path.split("/")[-1]
    resource = ChassisResourceClient(login_info, logger).get_network_adapter(resource_id)
    return resource.get("Members"), resource


@__try_new_password
def open_lldp_service_by_port_id_url(login_info, port_id_url, logger):
    """
    通过网卡端口的资源路径，打开此端口的lldp服务
    :param login_info: 登录信息
    :param port_id_url: 端口对应的redfish资源路径
    :param logger: 日志信息
    :return:
    """
    logger.info("open lldp service with port:{}".format(port_id_url))
    client = ChassisResourceClient(login_info, logger)
    response = client.request_get(port_id_url).resource
    param = _build_open_lldp_param(logger, response)
    if not param:
        raise DeployException("no lldp service param",
                              response, entity.create_msg("no.param.lldp"))
    try:
        return client.modify_resource_by_url(port_id_url, param)
    except DeployException as exception:
        raise DeployException("modify lldp service param fail",
                              exception.origin_info, entity.create_msg("modify.lldp.param.fail"))


@__try_new_password
def get_chassis_resource_url(login_info, resource_url, logger):
    """
    通过url获取该路径下的资源
    :param login_info: 登录信息
    :param resource_url: 资源路径
    :param logger: 日志
    :return:
    """
    resource = ChassisResourceClient(login_info, logger).request_get(resource_url).resource
    return resource.get("Members"), resource


def _build_open_lldp_param(logger, response):
    logger.info("build param with resource:{}".format(response))
    resource_key = "Oem"
    if resource_key not in response:
        return {}
    param = dict()
    oem_resource = response.get(resource_key)
    for key, values in oem_resource.items():
        if "LldpService" not in values:
            continue
        param[resource_key] = {key: {
            "LldpService": {
                "LldpEnabled": True
            }
        }}
        break
    logger.info("lldp param:{}".format(param))
    return param


def is_support(login_info, logger):
    product_unique_id, _ = get_product_unique_id(login_info, logger)
    return ProductStrategy.INS.isSupport(product_unique_id)


def reboot(need_reboot, context, login_info, logger):
    if not need_reboot:
        return
    try:
        reboot_server(login_info, logger)
    except DeployException as exception:
        exception.err_msg = entity.create_msg("reboot.server.failed")
        raise
    login_success, reboot_ssh_rets = ssh_util.reboot_had_succeeded(context)
    if not login_success:
        err_msg = entity.create_msg("reboot.server.failed")
        raise DeployException("reboot os failed",
                              origin_info=entity.Join.join_with_2lf(
                                  reboot_ssh_rets), err_msg=err_msg)


@__try_new_password
def get_node_sn(login_info, logger):
    client = SystemOverviewResourceClient(login_info, logger)
    over_view = client.get_systems_info()
    system_over_views = over_view.get("Members", [])
    for member in system_over_views:
        path = member.get("@odata.id", "")
        if not path:
            continue
        system_info = client.request_get(path).resource
        serial_number = system_info.get("SerialNumber")
        if serial_number:
            return serial_number
    return ""


@__trans_2_deploy_exception
def modify_bmc_default_password(context, logger, ip=""):
    """
    检查和尝试是否需要修改默认密码，如果该账号处于首次登录强制修改密码时，则尝试修改Administrator账号的密码
    :param context: 上下文
    :param logger: 日志打印器
    :param ip: 设备IP
    :return: 是否修改了默认密码
    """
    login_info = context_util.get_login_info(context)
    if ip:
        login_info.ip = ip
    client = AccountServiceClient(login_info, logger)
    try:
        client.get_all_users()
    except DeployException as exception:
        if exception.err_code != DeployException.ErrCode.BMC_DEFAULT_PASSWORD and login_info.had_new_password():
            # 此处是修改默认密码，若查询时的报错非默认密码不可登录的错误，则忽略，让正常执行流程的时候报出具体错误
            return False
        logger.info("start modify bmc default password")
        param = {"UserName": login_info.username, "Password": login_info.new_password}
        # 获取不到当前修改用户的ID，默认他需要执行超级管理员的id，如果id与用户名不匹配，会修改失败
        client.modify_bmc_default_password("2", param)
        # 原有连接已经被远程注销，此处释放缓存
        HttpClient.release_session(login_info, logger)
        return True
    return False
