#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/06/20
@file: cli_util.py
@function:
"""
import re
import time

import Business.adaptor.java_adaptor as java_adaptor
from Common.base import context_util
from Common.base import entity
from Common.base.constant import Platform
from Common.base.entity import DeployException

ToolConfig = java_adaptor.get_tool_config()
SshJudge = java_adaptor.get_ssh_judge()

ConnectionManager = java_adaptor.get_connection_manager()
IPMIConnection = java_adaptor.get_ipmi_connection()
ToolConstants = java_adaptor.get_tool_constants()
ToolException = java_adaptor.get_tool_exception()

JException = java_adaptor.get_j_exception()
SshConnection = java_adaptor.get_ssh_connection()

CMD_INVALID_KEYS = ("command not found", "No such file or directory",
                    "COMMAND NOT SUPPORTED")
SOL_TO_OS_CMD = "ipmcset -t sol -d activate -v 1 1"
OFF_SOL_CMD = "ipmcset -t sol -d deactivate -v 1"
SOL_TO_OS_STATE_DISABLED = "ipmcset -t service -d state -v RMCP+ disabled"
SOL_TO_OS_STATE_ENABLED = "ipmcset -t service -d state -v RMCP+ enabled"
SOL_CMD_TIME_OUT = 3

IPMI_LOGIN_ERROR = "Error: Unable to establish IPMI v2 / RMCP+ session"
global_logger = entity.create_logger(__file__)


def exec_ssh_cmd_nocheck(context, cmd, has_log=True, switch_root=True, timeout=30):
    """
     所有命令更改为30秒超时，部署场景评估没有需要执行超过30秒的命令，避免异常场景工具卡很久
    :param context: 上下文
    :param cmd: 命令
    :param has_log: 是否记录日志
    :param switch_root: 是否切换到root
    :param timeout: 超时时间
    :return: 执行结果
    """
    try:
        if has_log:
            ssh_ret = get_ssh(context, switch_root=switch_root).execCmdWithTimout(cmd, timeout)
        else:
            ssh_ret = get_ssh(context, switch_root=switch_root).execCmdNoLogTimout(cmd, timeout)
    except JException as exception:
        if has_log:
            global_logger.error("=========ssh_util exec cmd :{} error ".format(cmd))
        else:
            global_logger.error("=========ssh_util exec no log cmd error ")
        raise DeployException("exec cmd failed")
    if is_cmd_time_out(ssh_ret):
        raise DeployException("exec cmd failed")
    # 屏蔽带颜色的特殊字符
    return str(SshConnection.eliminatesResultColorCode(ssh_ret))


def exec_ssh_cmd_nocheck_with_timeout_and_retry_once(context, cmd, timeout, logger, has_log=True, switch_root=True):
    try:
        if has_log:
            ssh_ret = get_ssh(context, switch_root=switch_root).execCmdWithTimout(cmd, timeout)
        else:
            ssh_ret = get_ssh(context, switch_root=switch_root).execCmdNoLogTimout(
                cmd, timeout)
    except JException:
        raise DeployException("exec cmd failed")
    if is_cmd_time_out(ssh_ret):
        logger.info("cmd 15 minutes timeout. Retry. cmd:{} ".format(cmd))
        '''配置路由失败15分钟超时失败，释放连接后重试一次'''
        ConnectionManager.releaseConnection(context_util.get_dev_node(context), context_util.get_platform_id(context))
        '''释放连接后睡五秒再重新连接'''
        time.sleep(5)
        ssh_ret = exec_ssh_cmd_nocheck(context, cmd, has_log)
    return ssh_ret


def exec_ssh_cmd_without_return(context, cmd, switch_root=True):
    try:
        get_ssh(context,
                switch_root=switch_root).execCmdWithOutReturnAutoEnter(cmd)
    except JException:
        raise DeployException("exec cmd failed")


def exec_ssh_cmd(context, cmd, has_log=True, timeout=30):
    ssh_ret = exec_ssh_cmd_nocheck(context, cmd, has_log, True, timeout)
    if is_none_cmd_info(ssh_ret):
        raise DeployException("execute cmd: {} failed".format(cmd), ssh_ret,
                              err_code=DeployException.ErrCode.MAY_INFO_MISS)
    if is_invalid_cmd(ssh_ret):
        raise DeployException("invalid cmd : {}".format(cmd), ssh_ret,
                              err_msg=entity.create_msg("cmd.not.invalid"))
    return ssh_ret


def exec_ssh_cmd_disable_invalid(context, cmd, has_log=True):
    ssh_ret = exec_ssh_cmd_nocheck(context, cmd, has_log)
    if is_invalid_cmd(ssh_ret):
        raise DeployException("invalid cmd : {}".format(cmd), ssh_ret,
                              err_msg=entity.create_msg("cmd.not.invalid"))
    return ssh_ret


def exec_ssh_cmd_disable_invalid_with_timeout_and_retry(context, cmd, timeout, logger, has_log=True):
    ssh_ret = exec_ssh_cmd_nocheck_with_timeout_and_retry_once(context, cmd, timeout, logger, has_log)
    if is_invalid_cmd(ssh_ret):
        raise DeployException("invalid cmd : {}".format(cmd), ssh_ret,
                              err_msg=entity.create_msg("cmd.not.invalid"))
    return ssh_ret


def exec_ssh_cmd_disable_none(context, cmd, has_log=True):
    ssh_ret = exec_ssh_cmd_nocheck(context, cmd, has_log)
    if is_none_cmd_info(ssh_ret):
        raise DeployException("none cmd info: {}".format(cmd), ssh_ret,
                              err_code=DeployException.ErrCode.MAY_INFO_MISS)
    return ssh_ret


def is_none_cmd_info(ssh_ret):
    lines = ssh_ret.splitlines()
    no_info_line_num = 2
    if len(lines) < no_info_line_num:
        return True
    # 回显可能异常，结束符在正常回显之后，而未另起一行
    if len(lines) == no_info_line_num and lines[1].startswith("["):
        return True
    return False


def is_invalid_cmd(ssh_ret):
    lines = ssh_ret.splitlines()
    for invalid_key in CMD_INVALID_KEYS:
        if invalid_key in lines[-2]:
            return True
    return False


def is_cmd_time_out(ssh_ret):
    return ToolConstants.TIME_OUT == ssh_ret


def _parse_raid_card_type(ssh_ret):
    """
    解析raid卡型号数字部分
    :param ssh_ret: 命令回显
    :return: 只含数字的型号
    """
    sas_match = re.findall(r"\sSAS39xx\s", ssh_ret)
    if sas_match:
        return '3908'
    raid_1880_16i = re.findall(r"Device 3758", ssh_ret)
    if raid_1880_16i:
        # 1880RAID卡对外名称变更
        return 'SP686C-M-40i/SP686C-M-16i'
    raid_1880_8i = re.findall(r"Device 3858", ssh_ret)
    if raid_1880_8i:
        return 'SP186-M-8i'
    sas_res = re.findall(r"\sSAS-?\d*\s?\d+\s", ssh_ret)
    if not sas_res:
        return ''
    raid = re.sub(r'^[^\d]+', '', sas_res[0].strip()).split(' ')
    if len(raid) < 1:
        return ''
    return raid[0] if len(raid) <= 1 else raid[1]


def _get_raid_card_type_with_storcli_tool(context):
    """
    尝试通过storcli工具来获取RAID卡的版本类型，
    主要解决811版本上，lspci获取3908RAID卡信息不符合预期的问题。
    :param context: 上下文环境
    :return: 版本类型和原始执行指令结果
    """
    storcli_cmd = '/opt/MegaRAID/storcli/storcli64 /c0 show | grep -i "Product Name"'
    ssh_ret = exec_ssh_cmd_nocheck(context, storcli_cmd)
    if is_invalid_cmd(ssh_ret) or is_none_cmd_info(ssh_ret):
        return "", ssh_ret
    if "9560-8i" in ssh_ret:
        return "3908", ssh_ret
    return "", ssh_ret


def get_raid_card_type(context):
    """
    获取raid卡型号
    :return: 型号数字部分
    """
    origin_info = ''
    ssh_ret = exec_ssh_cmd_nocheck(context, "lspci | grep -i LSI")
    origin_info += ssh_ret
    if is_invalid_cmd(ssh_ret) or is_none_cmd_info(ssh_ret):
        # [root@Subnet1-Node4 preinstall]# lspci |grep -i RAID
        # 78:01.0 RAID bus controller: Huawei Technologies	Ltd. HiSilicon RDE Engine (rev 21)
        # 存储节点开启RDE功能后，RDE引擎被误识别为RAID卡，需要排除
        ssh_ret = exec_ssh_cmd_nocheck(context, "lspci | grep -i RAID | grep -v RDE")
        origin_info += ssh_ret
    if is_invalid_cmd(ssh_ret) or is_none_cmd_info(ssh_ret):
        return '', origin_info
    raid_card_type = _parse_raid_card_type(ssh_ret)
    # 如果通过lspci无法判断RAID卡版本，则尝试通过storcli命令来判断
    if not raid_card_type:
        raid_card_type, storcli_ret = _get_raid_card_type_with_storcli_tool(context)
        ssh_ret += storcli_ret
    if raid_card_type:
        return raid_card_type, ssh_ret
    if context_util.is_east_sea_or_sinan(context):
        return '', ssh_ret
    raise DeployException(
        "match raid card type failed",
        ssh_ret,
        err_msg=entity.create_msg("raid.card.type.not.support"),
        err_code=DeployException.ErrCode.MAY_INFO_MISS)


def reboot_had_succeeded(context, wait_login_rounds=360):
    ssh_rets = list()

    def is_stay_bmc(ssh_ret):
        return ssh_ret.strip().endswith("iBMC:/->")

    def is_stay_write_login_info():
        # 连续三次都是登录回显，表示重启成功，避免误判
        for _ in range(3):
            try:
                ssh_ret = exec_cmd()
            except DeployException:
                return False
            if not os_ready(ssh_ret):
                return False
        return True

    def os_ready(ssh_ret):
        return ssh_ret.endswith(" login:") or ssh_ret.endswith("Password:")

    def exec_cmd():
        ssh = get_ssh(context, switch_root=False)
        ssh_ret = ssh.execCmdWithTimout("\n", 5, None, False)
        ssh_rets.append(ssh_ret)
        if is_stay_bmc(ssh_ret):
            ssh_ret = ssh.execCmdWithTimout(SOL_TO_OS_CMD, 5, None, False)
            ssh_ret = handle_private_model(ssh, ssh_ret)
            ssh_rets.append(ssh_ret)
        return str(ssh_ret).strip()

    # 循环等待重启
    for i in range(wait_login_rounds):
        time.sleep(5)
        global_logger.info("restart loop query {} times".format(i))
        if is_stay_write_login_info():
            return True, ssh_rets
    return False, ssh_rets


def get_ssh_try_new_password(dev_node, platform_id, context):
    try:
        return ConnectionManager.getSshConnection(
            dev_node, SshJudge.INS, platform_id)
    except ToolException:
        ConnectionManager.releaseConnection(dev_node, platform_id)
        bmc_new_password = context_util.get_deploy_node(context). \
            getBmcUser().getNewPassword()
        dev_node.getLoginUser().setPassword(bmc_new_password)
        ssh_con = ConnectionManager.getSshConnection(
            dev_node, SshJudge.INS, platform_id)
        # 新密码登录成功，修改原密码为新密码
        context_util.get_deploy_node(context).getBmcUser(). \
            setOriginPassword(bmc_new_password)
        return ssh_con


def get_ssh(context, switch_root=True):
    dev_node = context_util.get_dev_node(context)
    platform_id = context_util.get_platform_id(context)

    def create_new_shh():
        ConnectionManager.releaseConnection(dev_node, platform_id)
        new_ssh = get_ssh_try_new_password(dev_node, platform_id, context)
        switch_to_os_root_obj = SwitchToOsRoot(context, new_ssh)
        if not switch_root or switch_to_os_root_obj.switch():
            return switch_to_os_root_obj._ssh
        raise DeployException("create new ssh failed")

    try:
        try:
            ssh = get_ssh_try_new_password(dev_node, platform_id, context)
            switch_to_os_root = SwitchToOsRoot(context, ssh)
            if not switch_root or switch_to_os_root.switch():
                return switch_to_os_root._ssh
            return create_new_shh()
        except ToolException:
            return create_new_shh()
    except (Exception, ToolException):
        # 异常可能是某些命令未执行完，需要释放掉，防止影响下次执行
        ConnectionManager.releaseConnection(dev_node, platform_id)
        raise


def _need_retry(ssh_ret):
    return "successfully" not in ssh_ret.strip().lower()


def off_and_on_sol(ssh):
    # 关闭SOL
    ssh_ret = ssh.execCmdWithTimout(SOL_TO_OS_STATE_DISABLED,
                                    SOL_CMD_TIME_OUT).strip()
    if _need_retry(ssh_ret):
        time.sleep(5)
        ssh.execCmdWithTimout(SOL_TO_OS_STATE_DISABLED,
                              SOL_CMD_TIME_OUT).strip()
    time.sleep(10)
    # 开启SOL
    ssh_ret = ssh.execCmdWithTimout(SOL_TO_OS_STATE_ENABLED,
                                    SOL_CMD_TIME_OUT).strip()
    if _need_retry(ssh_ret):
        time.sleep(5)
        ssh.execCmdWithTimout(SOL_TO_OS_STATE_ENABLED,
                              SOL_CMD_TIME_OUT).strip()
    time.sleep(60)


def handle_private_model(ssh, ssh_ret):
    if "Current SOL session is in private mode" in ssh_ret:
        ssh.execCmdWithTimout(OFF_SOL_CMD, 3)
        return ssh.execCmdWithTimout(SOL_TO_OS_CMD, 3)
    return ssh_ret


class SwitchToOsRoot(object):
    def __init__(self, context, ssh):
        self.context = context
        self._ssh = ssh
        deploy_node = context_util.get_deploy_node(context)
        self._platform_id = context_util.get_platform_id(context)
        self._deploy_node = deploy_node
        self._root_username = deploy_node.getOsRootUser().getUserName()
        self._root_password = deploy_node.getOsRootUser().getOriginPassword()
        self._root_new_password = deploy_node.getOsRootUser().getNewPassword()
        self._username = deploy_node.getOsUser().getUserName()
        self._password = deploy_node.getOsUser().getOriginPassword()
        self._new_password = deploy_node.getOsUser().getNewPassword()
        self._os_logined_ends = ToolConfig.getResource("os.login.success.end").split(",")

    def _modify_root_user_password(self):
        check_new_password(self._root_new_password, self._root_username)
        self._ssh.execCmdNoLog(self._root_password)
        self._ssh.execCmdNoLog(self._root_new_password)
        ssh_info = self._ssh.execCmdNoLog(self._root_new_password)
        # 如果未登录成功，再尝试登录一次，v2r9登录修改密码后，需要重新登录
        # 情况1，退出到登录界面
        if self._need_write_user_info(ssh_info):
            if self._need_su_to_root():
                self._ssh.execCmd(self._username)
                self._ssh.execCmdNoLog(self._password)
                self._ssh.execCmd("su - root")
                ssh_info = self._ssh.execCmdNoLog(self._root_new_password)
            else:
                self._ssh.execCmd(self._root_username)
                ssh_info = self._ssh.execCmdNoLog(self._root_new_password)
        # 情况2，退出到普通用户界面
        elif self._logined_general_user(ssh_info):
            self._ssh.execCmd("su - root")
            ssh_info = self._ssh.execCmdNoLog(self._root_new_password)

        if not self._logined_root(ssh_info):
            time.sleep(60)
            raise DeployException(
                "modify root password failed",
                origin_info=ssh_info,
                err_msg=entity.create_msg("modify.password.failed").format(
                    self._root_username)
            )
        self._root_password = self._root_new_password
        self._deploy_node.getOsRootUser().setOriginPassword(
            self._root_new_password)
        return ssh_info

    def _modify_general_user_password(self):
        check_new_password(self._new_password, self._username)
        self._ssh.execCmdNoLog(self._password)
        self._ssh.execCmdNoLog(self._new_password)
        ssh_info = self._ssh.execCmdNoLog(self._new_password)
        if self._need_write_user_info(ssh_info):
            # 如果未登录成功，再尝试登录一次，v2r9登录修改密码后，需要重新登录
            self._ssh.execCmd(self._username)
            ssh_info = self._ssh.execCmdNoLog(self._new_password)
        if not self._logined_general_user(ssh_info):
            time.sleep(60)
            raise DeployException(
                "modify general password failed",
                origin_info=ssh_info,
                err_msg=entity.create_msg("modify.password.failed").format(
                    self._username)
            )
        self._password = self._new_password
        self._deploy_node.getOsUser().setOriginPassword(self._new_password)
        return ssh_info

    def _had_login_success(self, ssh_info):
        ssh_info = SshConnection.eliminatesResultColorCode(ssh_info)
        for end_key in self._os_logined_ends:
            if ssh_info.strip().endswith(end_key):
                return True
        return False

    def _logined_root(self, ssh_info):
        return "[{}@".format(self._root_username) in ssh_info or self._root_username in self.exec_who_am_i()

    def _logined_general_user(self, ssh_info):
        return "[{}@".format(self._username) in ssh_info

    def _need_write_user_info(self, ssh_info):
        return ssh_info.strip().endswith("login:")

    def _is_sol_fist_login(self, ssh_info):
        return ssh_info.strip().endswith("exit.]")

    def _need_su_to_root(self):
        return self._username and (self._password or self._new_password)

    def _is_stay_bmc(self, ssh_ret):
        return ssh_ret.strip().endswith("iBMC:/->")

    def _need_modify_password(self, ssh_ret, username):
        global_logger.info("need modify default password for user :{}".format(username))
        return "you are required to change your password immediately" in \
            ssh_ret.lower()

    def _su_to_root(self):
        self._ssh.execCmdWithTimout("su - root", 10)
        ssh_ret = self._ssh.execCmdNoLog(self._root_password)
        # 还是在普通用户下，说明没切成功
        if self._logined_general_user(ssh_ret):
            self._ssh.execCmdWithTimout("su - root", 10)
            if self._logined_root(self._ssh.execCmdNoLog(
                    self._root_new_password)):
                self._deploy_node.getOsRootUser().setOriginPassword(
                    self._root_new_password)
                return True
            return False
        elif self._need_modify_password(ssh_ret, "root"):
            ssh_ret = self._modify_root_user_password()
        return self._logined_root(ssh_ret)

    def switch(self):
        if isinstance(self._ssh, IPMIConnection):
            is_range, result = self.sol_to_os()
            return result
        # 切换不会显示os回显，重试并跳过BMC修改密码
        for i in range(5):
            global_logger.info("sol retry time :{}".format(i))
            # 串口卡住问题，只等待30s超时
            ssh_ret = self._ssh.execCmdWithTimout(SOL_TO_OS_CMD, 30).strip()
            ssh_ret = handle_private_model(self._ssh, ssh_ret)
            # 存在过一个问题，切sol后，发送用户名后一直收不到回显，估计是发送命令太快，串口丢失信息的问题
            # 实测通过睡眠能避开当前问题
            time.sleep(2)
            is_need_login, result = self.is_need_login(ssh_ret)
            if not is_need_login:
                return result
            # 针对SOL可能会卡住问题，先输入用户名30秒超时尝试
            # 如果回文是超时标志，进入修复逻辑
            # 如果回文不是超时标志，通过is_need_continue_login方法判断是否需要登录
            is_range, result = self.sol_to_os()
            if not is_range:
                return result
        return False

    def sol_to_os(self):
        if self._need_su_to_root():
            ssh_ret = self._ssh.execCmdWithTimout(self._username,
                                                  3).strip()
            if ssh_ret != ToolConstants.TIME_OUT:
                is_continue_login, result = self.is_need_continue_login(
                    ssh_ret)
                if is_continue_login:
                    return False, self._su_root_to_login_root()
                else:
                    return False, result
            if isinstance(self._ssh, IPMIConnection):
                return False, False
            self.re_login_bmc()
            off_and_on_sol(self._ssh)
            return True, False
        else:
            ssh_ret = self._ssh.execCmdWithTimout(self._root_username,
                                                  3).strip()
            if ssh_ret != ToolConstants.TIME_OUT:
                is_continue_login, result = self.is_need_continue_login(
                    ssh_ret)
                if is_continue_login:
                    return False, self._direct_login_root()
                else:
                    return False, result
            if isinstance(self._ssh, IPMIConnection):
                return False, False
            self.re_login_bmc()
            off_and_on_sol(self._ssh)
        return True, False

    def is_need_login(self, ssh_ret):
        if self._is_sol_fist_login(ssh_ret):
            return True, True
        # 是否SOL切过去就是已登录状态
        if self._had_login_success(ssh_ret):
            result = self._logined_root(ssh_ret) or self._su_to_root()
            return False, result
        if not self._need_write_user_info(
                ssh_ret) and not self._is_sol_fist_login(ssh_ret):
            return False, False
        return True, True

    def is_need_continue_login(self, ssh_ret):
        # 是否SOL切过去就是已登录状态
        if self._had_login_success(ssh_ret):
            result = self._logined_root(ssh_ret) or self._su_to_root()
            return False, result
        return True, False

    def re_login_bmc(self):
        dev_node = context_util.get_dev_node(self.context)
        platform_id = context_util.get_platform_id(self.context)
        ConnectionManager.releaseConnection(dev_node, platform_id)
        self._ssh = get_ssh_try_new_password(dev_node, platform_id,
                                             self.context)

    def _su_root_to_login_root(self):
        ssh_ret = self._ssh.execCmdNoLog(self._password)
        if self._need_write_user_info(ssh_ret):
            self._ssh.execCmdWithTimout(self._username, 10)
            if not self._had_login_success(self._ssh.execCmdNoLog(
                    self._new_password)):
                return False
            self._deploy_node.getOsUser().setOriginPassword(
                self._new_password)
        elif self._need_modify_password(ssh_ret, self._username):
            self._modify_general_user_password()
        self.check_security_sandbox_state()
        return self._su_to_root()

    def check_security_sandbox_state(self):
        ret = self._ssh.execCmdWithTimout("\n", 10).strip()
        global_logger.info("check security sandbox state result:" + ret)
        if ret.endswith("root>") or ret.endswith("cliportal>"):
            self._ssh.execCmdWithTimout("exit", 10)

    def _direct_login_root(self):
        ssh_ret = self._ssh.execCmdNoLog(self._root_password)
        if self._need_write_user_info(ssh_ret):
            self._ssh.execCmd(self._root_username)
            if self._logined_root(self._ssh.execCmdNoLog(
                    self._root_new_password)):
                self._deploy_node.getOsRootUser().setOriginPassword(
                    self._root_new_password)
                return True
            return False
        elif self._need_modify_password(ssh_ret, self._root_username):
            ssh_ret = self._modify_root_user_password()
        self.check_security_sandbox_state()
        return self._logined_root(ssh_ret)

    def exec_who_am_i(self):
        return self._ssh.execCmdWithTimout('whoami', 10)


def check_new_password(new_password, username):
    # 需要改新密码，但是没有填新密码，抛出异常
    if not new_password:
        raise DeployException(
            "modify general password failed",
            origin_info="",
            err_msg=entity.create_msg(
                "need.modify.password.and.new.password.is.none").format(
                username)
        )


def get_ssh_try_new_password_by_type(dev_node, platform_id, deploy_node):
    try:
        return ConnectionManager.getSshConnection(dev_node, SshJudge.INS, platform_id)
    except ToolException:
        ConnectionManager.releaseConnection(dev_node, platform_id)
        bmc_new_password = deploy_node.getBmcUser().getNewPassword()
        dev_node.getLoginUser().setPassword(bmc_new_password)
        ssh_con = ConnectionManager.getSshConnection(dev_node, SshJudge.INS, platform_id)
        # 新密码登录成功，修改原密码为新密码
        deploy_node.getBmcUser().setOriginPassword(bmc_new_password)
        return ssh_con


def add_ssh_judge(ssh_judge):
    def do_func(func):
        def wrapper(*args, **kwargs):
            try:
                if isinstance(ssh_judge, list):
                    SshJudge.INS.setEndStr(ssh_judge)
                else:
                    SshJudge.INS.setEndStr([ssh_judge])
                return func(*args, **kwargs)
            finally:
                SshJudge.INS.removeJudge()

        return wrapper

    return do_func


class BmcConnection(object):
    def __init__(self, context, platform_id=""):
        self._context = context
        self._platform_id = platform_id
        self._ssh = self._init_ssh()

    def get_ssh(self):
        return self._ssh

    def _init_ssh(self):
        return get_ssh_try_new_password_by_type(context_util.get_dev_node(self._context), self._platform_id,
                                                context_util.get_deploy_node(self._context))


class OsConnection(object):
    def __init__(self, context):
        self._jump_handler = []
        self._context = context
        self._ssh = self._init_ssh()
        self._jump_to_os_by_sol()

    def get_ssh(self):
        return self._ssh

    def _init_ssh(self):
        if self._is_tai_shan_v1():
            return BmcConnection(self._context, Platform.TAI_SHAN_V1)
        return BmcConnection(self._context).get_ssh()

    def _jump_to_os_by_sol(self):
        if self._is_tai_shan_v1():
            return
        deploy_node = context_util.get_deploy_node(self._context)
        SolOsUtil(self._ssh, deploy_node.getOsUser(), deploy_node.getOsRootUser()).sol_to_os()

    def _is_tai_shan_v1(self):
        return Platform.TAI_SHAN_V1 == context_util.get_platform_id(self._context)


class SolOsUtil(object):
    def __init__(self, ssh, os_user, os_root_user):
        self._handlers = []
        self._ssh = ssh
        self._os_user = os_user
        self._os_root_user = os_root_user
        self._init_handler()

    def sol_to_os(self):
        self._handle_res("iBMC:/->")

    def _handle_res(self, ssh_ret):
        ret = ssh_ret
        while True:
            executor = self._find_handler(ret)
            if self._is_sol_success(executor):
                break
            ret = executor.execute()

    def _is_sol_success(self, executor):
        return isinstance(executor, RootCmd)

    def _init_handler(self):
        for sub_class in CmdExecutor.__subclasses__():
            self._handlers.append(sub_class(self._ssh, self._context))

    def _find_handler(self, ssh_ret):
        for handler in self._handlers:
            if not handler.match_end_str(ssh_ret):
                continue
            return handler
        raise DeployException('executor not found', origin_info=ssh_ret, err_msg='model.not.support')


def _need_modify_password(ssh_ret):
    return "you are required to change your password immediately" in ssh_ret.lower()


def login_success(ssh_ret):
    return 'Login incorrect' not in ssh_ret and "Permission denied" not in ssh_ret


class CmdExecutor(object):
    def __init__(self, ssh, os_user, os_root_user):
        self._ssh = ssh
        self._os_user = os_user
        self._os_root_user = os_root_user

    def match_end_str(self, ssh_ret):
        for handle_ret in self.get_handle_ssh_ret():
            if ssh_ret.strip().endswith(handle_ret):
                return True
        return False

    def execute(self):
        raise NotImplementedError

    def get_handle_ssh_ret(self):
        raise NotImplementedError


class SolExecutor(CmdExecutor):
    HANDLE_RET = ['iBMC:/->']
    SSH_JUDGE = ['login:', 'exit.]']

    def __init__(self, ssh, context):
        super(SolExecutor, self).__init__(ssh, context)
        self._context = context

        self._ssh = ssh
        self._cmd = SOL_TO_OS_CMD

    def get_handle_ssh_ret(self):
        return SolExecutor.HANDLE_RET

    def execute(self):
        ssh_ret = self._ssh.execCmdWithTimout(self._cmd, 30).strip()
        SolExecutor.handle_sol_error(ssh_ret)
        self._set_cache_user()
        return ssh_ret

    @staticmethod
    def handle_sol_error(ssh_ret):
        if "Current SOL session is in private mode" in ssh_ret:
            raise DeployException(
                "sol in private mode",
                origin_info=ssh_ret,
                err_msg=entity.create_msg("sol.private.mode"))

    def _set_cache_user(self):
        deploy_node = context_util.get_deploy_node(self._context)
        os_user = deploy_node.getOsUser()
        if os_user and os_user.getUserName():
            LoginCache.cache_no_root_user(self._context)
            return
        root_user = deploy_node.getOsRootUser()
        if not root_user:
            raise DeployException('no user can login')
        LoginCache.cache_root_user(self._context)


class UserExecutor(CmdExecutor):
    HANDLE_RET = ['login:', 'exit.]']

    def __init__(self, ssh, context):
        super(UserExecutor, self).__init__(ssh, context)
        self._context = context
        self._ssh = ssh

    def execute(self):
        login_username = LoginCache.get_cache_user(self._context).getUserName()
        ssh_ret = self._ssh.execCmdWithTimout(login_username, 30).strip()
        if is_cmd_time_out(ssh_ret):
            raise DeployException('input username timeout ,maybe sol closed',
                                  err_msg=entity.create_msg('obtain.information.failed'))
        return ssh_ret

    def get_handle_ssh_ret(self):
        return UserExecutor.HANDLE_RET


class PasswordExecutor(CmdExecutor):
    HANDLE_RET = ['Password:']

    def __init__(self, ssh, context):
        super(PasswordExecutor, self).__init__(ssh, context)
        self._context = context
        self._ssh = ssh

    def execute(self):
        login_password = LoginCache.get_cache_user(self._context).getOriginPassword()
        ssh_ret = self._ssh.execCmdNoLog(login_password).strip()
        self._handler_password_error(ssh_ret)
        ssh_ret = self._handle_need_modify_password(ssh_ret)
        return ssh_ret

    def get_handle_ssh_ret(self):
        return PasswordExecutor.HANDLE_RET

    def _handler_password_error(self, ssh_ret):
        if login_success(ssh_ret):
            return
        current_user = LoginCache.get_cache_user(self._context)
        if current_user.getNewPassword() == current_user.getOriginPassword():
            raise DeployException("login failed with login incorrect", origin_info=ssh_ret,
                                  err_msg=entity.create_msg("obtain.information.failed").format(
                                      current_user.getUserName()))
        current_user.setOriginPassword(current_user.getNewPassword())

    def _handle_need_modify_password(self, ssh_ret):
        if not _need_modify_password(ssh_ret):
            return ssh_ret
        user = LoginCache.get_cache_user(self._context)
        check_new_password(user.getNewPassword(), user.getUserName())
        return self._ssh.execCmdNoLog(user.getOriginPassword()).strip()


class NewPasswordExecutor(CmdExecutor):
    HANDLE_RET = ['New password:']

    def __init__(self, ssh, context):
        super(NewPasswordExecutor, self).__init__(ssh, context)
        self._context = context
        self._ssh = ssh

    def execute(self):
        new_password = LoginCache.get_cache_user(self._context).getNewPassword()
        ssh_ret = self._ssh.execCmdNoLog(new_password).strip()
        self._check_new_password_valid(ssh_ret)
        return ssh_ret

    def get_handle_ssh_ret(self):
        return NewPasswordExecutor.HANDLE_RET

    def _check_new_password_valid(self, ssh_ret):
        if "BAD PASSWORD:" in ssh_ret:
            raise DeployException("modify general password failed", origin_info=ssh_ret,
                                  err_msg=entity.create_msg("need.modify.password.and.new.password.is.none")
                                  .format(LoginCache.get_cache_user(self._context).getUserName()))


class SuToRootExecutor(CmdExecutor):
    HANDLE_RET = ['~]#', '~]$']

    def __init__(self, ssh, context):
        super(SuToRootExecutor, self).__init__(ssh, context)
        self._ssh = ssh
        self._context = context

    def match_end_str(self, ssh_ret):
        return super(SuToRootExecutor, self).match_end_str(ssh_ret) and \
            'root' not in self._ssh.execCmdWithTimout('whoami', 30).strip()

    def execute(self):
        ssh_ret = self._ssh.execCmdWithTimout('su - root', 30).strip()
        LoginCache.cache_root_user(self._context)
        return ssh_ret

    def get_handle_ssh_ret(self):
        return SuToRootExecutor.HANDLE_RET


class RootCmd(CmdExecutor):
    HANDLE_RET = ['~]#', '~]$']

    def __init__(self, ssh, context):
        super(RootCmd, self).__init__(ssh, context)
        self._ssh = ssh
        self._context = context

    def execute(self):
        """
        进入OS root模式后，不再需要执行命令了,该类只是为了判断状态，作为登录成功的出口
        """
        pass

    def match_end_str(self, ssh_ret):
        return super(RootCmd, self).match_end_str(ssh_ret) and \
            'root' in self._ssh.execCmdWithTimout('whoami', 30).strip()

    def get_handle_ssh_ret(self):
        return RootCmd.HANDLE_RET
