#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.
"""
@time: 2020/11/16
@file: acpi_spcr_service.py
@function:
"""
from Common.base import context_util
from Common.protocol import ssh_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.protocol.redfish import redfish_util
from Common.base.constant import Platform

SPCR_NEED_PLATFORM_IDS = [Platform.TAI_SHAN_V1, Platform.TAI_SHAN_V2]
SPCR_BIOS_KEY = {
    Platform.TAI_SHAN_V1: "ACPISpcrTableEnable",
    Platform.TAI_SHAN_V2: "EnableSpcr"
}
SPCR_OPEN_BIOS_VALUE = {
    Platform.TAI_SHAN_V1: "Enable",
    Platform.TAI_SHAN_V2: "Enabled"
}
SPCR_CLOSE_BIOS_VALUE = {
    Platform.TAI_SHAN_V1: "Disable",
    Platform.TAI_SHAN_V2: "Disabled"
}


class AcpiSpcrService(object):
    TURN_ON_DEV_IPS = list()

    def __init__(self, context):
        self._context = context
        self._login_info = context_util.get_login_info(context)
        self._logger = entity.create_logger(__file__)

    def open_spcr(self, need_reboot=True):
        """
        开启spcr
        :param need_reboot: 是否需要重启
        :return: 是否执行了开启操作
        """
        if self.need_open_spcr():
            self._logger.info("need open spcr")
            self._open_spcr()
            redfish_util.reboot(need_reboot, self._context, self._login_info, self._logger)
            return True
        return False

    def close_spcr(self, need_reboot=True):
        """
        关闭spcr
        :param need_reboot: 是否需要重启
        :return: 是否执行了关闭操作
        """
        if self._need_close_spcr():
            self._logger.info("need close spcr")
            self._close_spcr()
            redfish_util.reboot(need_reboot, self._context, self._login_info, self._logger)
            return True
        return False

    def need_open_spcr(self):
        return self._need_spcr_dev() and self._is_spcr_closed()

    def _get_spcr_bios_value(self):
        value, resource = redfish_util.get_one_bios_item_info(
            self._login_info, self._get_bios_key(), self._logger)
        self._logger.info("spcr value: {}".format(value))
        return value

    def _is_spcr_closed(self):
        value = self._get_spcr_bios_value()
        return value and value == self._get_spcr_close_value()

    def _is_spcr_opened(self):
        value = self._get_spcr_bios_value()
        return value and value == self._get_spcr_open_value()

    def _need_spcr_dev(self):
        return context_util.get_platform_id(
            self._context) in SPCR_NEED_PLATFORM_IDS

    def _open_spcr(self):
        redfish_util.set_one_bios_info(self._login_info, self._get_bios_key(),
                                       self._get_spcr_open_value(), self._logger)
        AcpiSpcrService.TURN_ON_DEV_IPS.append(context_util.get_deploy_node(
            self._context).getIp())

    def _need_close_spcr(self):
        node_ip = context_util.get_deploy_node(self._context).getIp()
        had_turn_on_spcr = node_ip in AcpiSpcrService.TURN_ON_DEV_IPS
        return had_turn_on_spcr and self._is_spcr_opened()

    def _close_spcr(self):
        redfish_util.set_one_bios_info(self._login_info, self._get_bios_key(),
                                       self._get_spcr_close_value(), self._logger)
        AcpiSpcrService.TURN_ON_DEV_IPS.remove(context_util.get_deploy_node(
            self._context).getIp())

    def _get_bios_key(self):
        return SPCR_BIOS_KEY.get(context_util.get_platform_id(
            self._context))

    def _get_spcr_close_value(self):
        return SPCR_CLOSE_BIOS_VALUE.get(context_util.get_platform_id(
            self._context))

    def _get_spcr_open_value(self):
        return SPCR_OPEN_BIOS_VALUE.get(context_util.get_platform_id(
            self._context))
