#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

"""
@time: 2024/03/14
@file: config_mlx_nic_mode.py
@function:
"""
import time

from com.huawei.ism.tool.distributeddeploy.logic.importfile.entity import TransportProtocol

import Business.adaptor.java_adaptor as java_adaptor
from Common.base import entity, context_util
from Common.base.entity import DeployException
from Common.protocol import ssh_util

SshConnection = java_adaptor.get_ssh_connection()


def execute_cmd_by_client(ssh_client, cmd):
    ssh_ret = ssh_client.execCmdWithTimout(cmd, 30)
    if ssh_util.is_cmd_time_out(ssh_ret):
        raise DeployException("exec cmd failed")
    # 屏蔽带颜色的特殊字符
    return str(SshConnection.eliminatesResultColorCode(ssh_ret))


class ConfigMlxNicMode(object):

    def __init__(self, context):
        self._context = context
        self._logger = entity.create_logger(__file__)
        self._node_ip = context_util.get_deploy_ip(self._context)
        self.ssh_rets = list()

    def query_slot_bus_info_by_slot_address(self, slot_address):
        """
        查询指定slot地址对应的网卡槽位bus地址
        :param slot_address: slot总线地址
        :return: 网卡槽位bus地址
        """
        cmd = "ls -l /sys/class/net/** | grep {0}".format(slot_address)
        self._logger.info("query slot bus info, slot address : {}.".format(slot_address))
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(self._context, cmd)
        self.ssh_rets.append(ssh_ret)
        bus_address_infos = list()
        for line in ssh_ret.splitlines():
            if 'ls -l' not in line and slot_address in line:
                bus_address_infos.append(line.strip().split("/")[-3])
        return bus_address_infos

    def check_slot_nic_is_special_mode(self, bus_info, special_mode):
        """
        检查当前的网卡对应的网络模式是否为指定的模式，如果当前网卡无法查询或查询失败，则默认模式相同
        :param bus_info: 网卡bus地址
        :param special_mode: 指定的网卡模式，TransportProtocol的常量，TransportProtocol.ROCE 和 TransportProtocol.IB
        :return: 是否相同
        """
        # 查询P1端口的模式
        result, current_mode = self._query_slot_nic_current_mode(bus_info, 'LINK_TYPE_P1')
        if result and current_mode is not special_mode:
            return False
        # 查询P2端口的模式
        result, current_mode = self._query_slot_nic_current_mode(bus_info, 'LINK_TYPE_P2')
        if result and current_mode is not special_mode:
            return False
        return True

    def change_slot_nic_mode(self, bus_info, mode):
        """
        修改网卡的模式
        :param bus_info: 网卡bus地址
        :param mode: TransportProtocol的常量，TransportProtocol.ROCE 和 TransportProtocol.IB
        :return:
        """
        set_mode = '2' if mode is TransportProtocol.ROCE else '1'
        return self._change_slot_special_nic_mode(bus_info, 'LINK_TYPE_P1',
                                                  set_mode) and self._change_slot_special_nic_mode(bus_info,
                                                                                                   'LINK_TYPE_P2',
                                                                                                   set_mode)

    def reboot_node_and_wait_for_started(self):
        """
        重启节点，等待系统启动
        :return: 是否重启成功
        """
        self._logger.info("start to reboot node({})..".format(self._node_ip))
        # 启动重启节点
        cmd = "reboot"
        ssh_util.exec_ssh_cmd_without_return(self._context, cmd)
        self.ssh_rets.append(cmd)
        self._logger.info("wait for reboot node({})..".format(self._node_ip))
        # 等待5分钟后，开始检查设备是否重启成功
        time.sleep(5 * 60)
        # 循环等待重启
        login_success, reboot_ssh_rets = ssh_util.reboot_had_succeeded(self._context)
        self.ssh_rets.extend(reboot_ssh_rets)
        self._logger.info("finish to reboot node({})..".format(self._node_ip))
        return login_success

    def _query_slot_nic_current_mode(self, bus_info, link_type):
        """
        查询当前的bus号对应的网卡模式
        :param bus_info: 网卡bus地址
        :param link_type: 类型，LINK_TYPE_P1 或 LINK_TYPE_P2
        :return: 查询结果和当前的网卡类型
        """
        cmd = "mlxconfig -d {0} query {1}".format(bus_info, link_type)
        self._logger.info("query slot mode cmd : {}".format(cmd))
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(self._context, cmd)
        self.ssh_rets.append(ssh_ret)
        if '-E-' in ssh_ret or 'command not found' in ssh_ret:
            return False, ''
        nic_mode = ''
        for line in ssh_ret.splitlines():
            if 'mlxconfig' not in line and link_type in line:
                nic_mode = line.strip().split()[1]
            if 'ETH' in nic_mode:
                return True, TransportProtocol.ROCE
            if 'IB' in nic_mode:
                return True, TransportProtocol.IB
        return False, ''

    def _change_slot_special_nic_mode(self, bus_info, link_type, set_mode):
        cmd = "mlxconfig -d {0} set {1}={2}".format(bus_info, link_type, set_mode)
        self._logger.info("change nic mode, cmd : {0}".format(cmd))
        ssh_client = ssh_util.get_ssh(self._context)
        ssh_ret = execute_cmd_by_client(ssh_client, cmd)
        self.ssh_rets.append(ssh_ret)
        if 'Apply new Configuration? (y/n) [n]' in ssh_ret:
            self._logger.info("input y to make sure to change nic mode.")
            ssh_ret = execute_cmd_by_client(ssh_client, 'y')
            self.ssh_rets.append(ssh_ret)
        return 'Please reboot machine to load new configurations.' in ssh_ret
