#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2020/06/20
@file: os_network_util.py
@function:
"""
import copy
from Common.base.entity import DeployException
from Common.service.bond_opts_builder import BondOptsBuilder

BOND_CONSTANT_OPTS_BUILDER = {
    "BOND1": BondOptsBuilder().mode("1").miimon(),
    "BOND2": BondOptsBuilder().mode("2").miimon().xmit_hash_policy(),
    "BOND4": BondOptsBuilder().mode(
        "4").miimon().lacp_rate().xmit_hash_policy()
}


class OsNetworkConfigTemplate(object):

    LOGICAL_PORT_TO_BOND = {
        "DEVICE": None,  # 文件名
        "BOOTPROTO": "none",
        "ONBOOT": "yes",
        "MASTER": None,
        "SLAVE": "yes",
        "USERCTL": "no",
        "NM_CONTROLLED": "no"
    }

    LOGICAL_PORT_TO_VLAN = {
        "DEVICE": None,  # 文件名
        "BOOTPROTO": "none",
        "ONBOOT": "yes",
        "STARTMODE": "onboot",
        "NM_CONTROLLED": "no"
    }

    LOGICAL_PORT_WITH_IPV4 = {
        "DEVICE": None,  # 文件名
        "BOOTPROTO": "static",
        "ONBOOT": "yes",
        "STARTMODE": "onboot",
        "IPADDR": None,
        "NETMASK": None,
        "IPV6_AUTOCONF": "no",
        "NM_CONTROLLED": "no"
    }

    LOGICAL_PORT_WITH_IPV6 = {
        "DEVICE": None,  # 文件名
        "BOOTPROTO": "static",
        "ONBOOT": "yes",
        "STARTMODE": "onboot",
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "NM_CONTROLLED": "no"
    }

    LOGICAL_PORT_WITH_IPV4_AND_IPV6 = {
        "DEVICE": None,  # 文件名
        "BOOTPROTO": "static",
        "ONBOOT": "yes",
        "STARTMODE": "onboot",
        "IPADDR": None,
        "NETMASK": None,
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "NM_CONTROLLED": "no"
    }

    BOND_PORT_WITH_IPV4 = {
        "DEVICE": None,  # 文件名
        "IPADDR": None,
        "NETMASK": None,
        "ONBOOT": "yes",
        "BOOTPROTO": "static",
        "USERCTL": "no",
        "TYPE": "Bonding",
        "MTU": None,
        "IPV6_AUTOCONF": "no",
        "NM_CONTROLLED": "no",
        "BONDING_OPTS": None,
        "BONDING_SLAVE0": None,
        "BONDING_SLAVE1": None
    }

    BOND_PORT_WITH_IPV6 = {
        "DEVICE": None,  # 文件名
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "ONBOOT": "yes",
        "BOOTPROTO": "static",
        "USERCTL": "no",
        "TYPE": "Bonding",
        "MTU": None,
        "NM_CONTROLLED": "no",
        "BONDING_OPTS": None,
        "BONDING_SLAVE0": None,
        "BONDING_SLAVE1": None
    }

    BOND_PORT_WITH_IPV4_AND_IPV6 = {
        "DEVICE": None,  # 文件名
        "IPADDR": None,
        "NETMASK": None,
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "ONBOOT": "yes",
        "BOOTPROTO": "static",
        "USERCTL": "no",
        "TYPE": "Bonding",
        "MTU": None,
        "NM_CONTROLLED": "no",
        "BONDING_OPTS": None,
        "BONDING_SLAVE0": None,
        "BONDING_SLAVE1": None
    }

    BOND_PORT_NO_IP = {
        "DEVICE": None,  # 文件名
        "ONBOOT": "yes",
        "BOOTPROTO": "none",
        "USERCTL": "no",
        "TYPE": "Bonding",
        "MTU": None,
        "NM_CONTROLLED": "no",
        "BONDING_OPTS": None,
        "BONDING_SLAVE0": None,
        "BONDING_SLAVE1": None
    }

    VLAN_WITH_IPV4 = {
        "DEVICE": None,  # 文件名
        "IPADDR": None,
        "NETMASK": None,
        "BOOTPROTO": "static",
        "PHYSDEV": None,  # * 与设置的绑定端口的名称保持一致。 *
        "ONBOOT": "yes",
        "USERCTL": "no",
        "VLAN": "yes",
        "VID": None,
        "IPV6_AUTOCONF": "no",
        "NM_CONTROLLED": "no"
    }

    VLAN_WITH_IPV6 = {
        "DEVICE": None,  # 文件名
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "BOOTPROTO": "static",
        "PHYSDEV": None,  # 与设置的绑定端口的名称保持一致。
        "ONBOOT": "yes",
        "USERCTL": "no",
        "VLAN": "yes",
        "VID": None,
        "NM_CONTROLLED": "no"
    }

    VLAN_WITH_IPV4_AND_IPV6 = {
        "DEVICE": None,  # 文件名
        "IPADDR": None,
        "NETMASK": None,
        "IPV6INIT": "yes",
        "IPV6ADDR": None,  # IPv6 address/prefix
        "IPV6_AUTOCONF": "no",
        "BOOTPROTO": "static",
        "PHYSDEV": None,  # * 与设置的绑定端口的名称保持一致。 *
        "ONBOOT": "yes",
        "USERCTL": "no",
        "VLAN": "yes",
        "VID": None,
        "NM_CONTROLLED": "no"
    }


def _trans_dict_config_to_file_echo_info(config_dict):
    file_echo_info_list = list()
    for key, value in config_dict.items():
        file_echo_info_list.append("{}={}".format(key, value))
    return "\n".join(file_echo_info_list)


def _add_ipv4_to_template(template, ipv4):
    template["IPADDR"] = ipv4.getIp()
    template["NETMASK"] = ipv4.getNetMaskOrPrefix()
    gateway = ipv4.getGateway()
    if gateway:
        template["NETWORKING"] = "yes"
        template["GATEWAY"] = gateway
    template["DEFROUTE"] = "yes" if ipv4.isOpenDefaultRoute() else "no"


def _add_ipv6_to_template(template, ipv6):
    template["IPV6ADDR"] = "{}/{}".format(
        ipv6.getIp(), ipv6.getNetMaskOrPrefix())
    gateway = ipv6.getGateway()
    if gateway:
        template["NETWORKING"] = "yes"
        template["NETWORKING_IPV6"] = "yes"
        template["IPV6_DEFAULTGW"] = gateway
    template["IPV6_DEFROUTE"] = "yes" if ipv6.isOpenDefaultRoute() else "no"


def _add_bond_info_to_template(template, bond_port, port_names):
    bond_pattern = bond_port.getBondPattern().toString()
    _verify_valid_bond_pattern(bond_pattern)
    template["MTU"] = bond_port.getMtu()
    opts_builder = copy.deepcopy(BOND_CONSTANT_OPTS_BUILDER.get(bond_pattern))
    if bond_port.isArp():
        opts_builder.arp_broadcast()
    template["BONDING_OPTS"] = opts_builder.get_build_info()
    template["BONDING_SLAVE0"] = port_names[0]
    template["BONDING_SLAVE1"] = port_names[1]


def _add_vlan_info_to_template(template, port_name, vlan_id):
    template["PHYSDEV"] = port_name
    template["VID"] = vlan_id


def _verify_valid_bond_pattern(bond_pattern):
    if bond_pattern not in BOND_CONSTANT_OPTS_BUILDER:
        raise DeployException("no support bond model: {}".format(bond_pattern))


def create_logical_port_to_bond(port_name, master):
    template = copy.copy(OsNetworkConfigTemplate.LOGICAL_PORT_TO_BOND)
    template["DEVICE"] = port_name
    template["MASTER"] = master
    return _trans_dict_config_to_file_echo_info(template)


def create_logical_port_to_vlan(port_name):
    template = copy.copy(OsNetworkConfigTemplate.LOGICAL_PORT_TO_VLAN)
    template["DEVICE"] = port_name
    return _trans_dict_config_to_file_echo_info(template)


def create_logical_port_with_ipv4(port_name, ipv4):
    template = copy.copy(OsNetworkConfigTemplate.LOGICAL_PORT_WITH_IPV4)
    template["DEVICE"] = port_name
    _add_ipv4_to_template(template, ipv4)
    return _trans_dict_config_to_file_echo_info(template)


def create_logical_port_with_ipv6(port_name, ipv6):
    template = copy.copy(OsNetworkConfigTemplate.LOGICAL_PORT_WITH_IPV6)
    template["DEVICE"] = port_name
    _add_ipv6_to_template(template, ipv6)
    return _trans_dict_config_to_file_echo_info(template)


def create_logical_port_with_ipv4_and_ipv6(port_name, ipv4, ipv6):
    template = copy.copy(OsNetworkConfigTemplate.
                         LOGICAL_PORT_WITH_IPV4_AND_IPV6)
    template["DEVICE"] = port_name
    _add_ipv4_to_template(template, ipv4)
    _add_ipv6_to_template(template, ipv6)
    return _trans_dict_config_to_file_echo_info(template)


def create_bond_port_no_ip(bond_port, port_names):
    template = copy.copy(OsNetworkConfigTemplate.BOND_PORT_NO_IP)
    template["DEVICE"] = bond_port.getBondName()
    _add_bond_info_to_template(template, bond_port, port_names)
    return _trans_dict_config_to_file_echo_info(template)


def create_bond_port_with_ipv4(bond_port, ipv4, port_names):
    template = copy.copy(OsNetworkConfigTemplate.BOND_PORT_WITH_IPV4)
    template["DEVICE"] = bond_port.getBondName()
    _add_ipv4_to_template(template, ipv4)
    _add_bond_info_to_template(template, bond_port, port_names)
    return _trans_dict_config_to_file_echo_info(template)


def create_bond_port_with_ipv6(bond_port, ipv6, port_names):
    template = copy.copy(OsNetworkConfigTemplate.BOND_PORT_WITH_IPV6)
    template["DEVICE"] = bond_port.getBondName()
    _add_ipv6_to_template(template, ipv6)
    _add_bond_info_to_template(template, bond_port, port_names)
    return _trans_dict_config_to_file_echo_info(template)


def create_bond_port_with_ipv4_and_ipv6(bond_port, ipv4, ipv6, port_names):
    template = copy.copy(OsNetworkConfigTemplate.BOND_PORT_WITH_IPV4_AND_IPV6)
    template["DEVICE"] = bond_port.getBondName()
    _add_ipv4_to_template(template, ipv4)
    _add_ipv6_to_template(template, ipv6)
    _add_bond_info_to_template(template, bond_port, port_names)
    return _trans_dict_config_to_file_echo_info(template)


def create_vlan_with_ipv4(vlan_name, ipv4, port_name, vlan_id):
    template = copy.copy(OsNetworkConfigTemplate.VLAN_WITH_IPV4)
    template["DEVICE"] = vlan_name
    _add_ipv4_to_template(template, ipv4)
    _add_vlan_info_to_template(template, port_name, vlan_id)
    return _trans_dict_config_to_file_echo_info(template)


def create_vlan_with_ipv6(vlan_name, ipv6, port_name, vlan_id):
    template = copy.copy(OsNetworkConfigTemplate.VLAN_WITH_IPV6)
    template["DEVICE"] = vlan_name
    _add_ipv6_to_template(template, ipv6)
    _add_vlan_info_to_template(template, port_name, vlan_id)
    return _trans_dict_config_to_file_echo_info(template)


def create_vlan_with_ipv4_and_ipv6(vlan_name, ipv4, ipv6, port_name, vlan_id):
    template = copy.copy(OsNetworkConfigTemplate.VLAN_WITH_IPV4_AND_IPV6)
    template["DEVICE"] = vlan_name
    _add_ipv4_to_template(template, ipv4)
    _add_ipv6_to_template(template, ipv6)
    _add_vlan_info_to_template(template, port_name, vlan_id)
    return _trans_dict_config_to_file_echo_info(template)
