#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/06/23
@file: physics_netcard_to_logical.py
@function:
"""
import re
from Common.protocol import ssh_util
from Common.base import context_util
from Common.base import entity
from Common.base.entity import DeployException
from Common.base.constant import Platform
from com.huawei.ism.tool.distributeddeploy.common import ToolConfig

SLOT_NAME_KEY = "Designation: "
SLOT_ADDRESS_KEY = "Bus Address: "
QUERY_NETWORK_CARD_INFO_CMD = "dmidecode -t slot"
QUERY_SLOT_INFO_CMD = "ls -l /sys/class/net/** | grep {}"
# 管理的两个槽位地址固定
OM_BUSS_RESOURCE = {Platform.TAI_SHAN_V1: "taishanv1.nic.bdf",
                    Platform.TAI_SHAN_V2: "taishanv2.nic.bdf",
                    Platform.X86_64: "x86_64.nic.bdf",
                    Platform.ATLANTIC: "atlantic.nic.bdf",
                    Platform.NEW_ATLANTIC: "new_atlantic.nic.bdf",
                    Platform.PACIFIC: "pacific.nic.bdf",
                    Platform.X86_V6: "v6.nic.bdf"}
OM_BUSS_RESOURCE_COMPATIBLE = {Platform.X86_V6: "v6.nic.bdf.compatible"}
OCP_BUSS_RESOURCE_KEY = "haiyan.ocp.bdf"


class PhysicalCardToLogical(object):

    def __init__(self, context):
        self._context = context
        self._logger = entity.create_logger(__file__)
        self.slot_ssh_info = ""
        self._method_map = {
            Platform.SI_NAN: self.init_si_nan_key,
            Platform.EAST_SEA: self.init_east_sea_bdf_key,
            Platform.NEW_SI_NAN: self.init_si_nan_key
        }
        self._si_config_key = self.init_bdf_key()
        self._slot_2_address = self._parse_fixed_slot_2_address()
        self._slot_2_address_compatible = self._parse_fixed_slot_2_address_compatible()
        self._init_slot_to_address()

    def _init_slot_to_address(self):
        # 专有硬件设备命令查出来不准确，全部取固定bdf号
        if self._is_proprietary_hardware_dev() or self.is_support_no_slot():
            return
        ssh_ret = ssh_util.exec_ssh_cmd_disable_invalid(
            self._context, QUERY_NETWORK_CARD_INFO_CMD)
        self.slot_ssh_info = ssh_ret
        find_address = False
        slot_name = ""
        for line in ssh_ret.splitlines():
            if not find_address and self._is_slot_name_line(line):
                slot_name = self._get_slot_name_form_name_line(line).lower()
                find_address = True
                continue
            if find_address and self._is_slot_address_line(line):
                slot_address = self._get_slot_address_form_address_line(line)
                find_address = False
                if self._is_valid_slot_address(slot_address) and slot_name \
                        not in self._slot_2_address:
                    self._logger.info("valid address: {}".format(slot_address))
                    self._slot_2_address[slot_name] = slot_address
        self._logger.info("slots:{}".format(self._slot_2_address))

    def _parse_fixed_slot_2_address(self):
        nic_bdf = dict()
        dev_platform = context_util.get_platform_id(self._context)
        for platform, resource in OM_BUSS_RESOURCE.items():
            if platform != dev_platform:
                continue
            bdfs = ToolConfig.getResource(resource).split(",")
            for index, bdf in enumerate(bdfs):
                nic_bdf["nic{}".format(index + 1)] = bdf
            break
        ocp_bdfs = ToolConfig.getResource(OCP_BUSS_RESOURCE_KEY).split(",")
        for index, ocp_bdf in enumerate(ocp_bdfs):
            nic_bdf["ocp{}".format(index + 1)] = ocp_bdf
        # 专有硬件设备命令查出来不准确，全部取固定bdf号
        if self._is_proprietary_hardware_dev():
            slot_bdfs = ToolConfig.getResource("{}.slot.bdf".format(
                dev_platform)).split(",")
            for index, slot_bdf in enumerate(slot_bdfs):
                nic_bdf["slot{}".format(index)] = slot_bdf
        if self.is_support_no_slot():
            self.init_support_no_slot_hardware_slot(nic_bdf)
        self._logger.info("fixed bdf: {}".format(nic_bdf))
        return nic_bdf

    def init_support_no_slot_hardware_slot(self, nic_bdf_dict):
        """
        支持不填slot的机型，并不是不需要slot的机型，所以仍然需要初始化slot
        :param nic_bdf_dict: 总线缓存容器
        :return:
        """
        platform_id = context_util.get_platform_id(self._context)
        slot_key = Platform.get_slot_pdf_key(platform_id, self._si_config_key)
        self._logger.info("get bdf list for {},slot key {}".format(platform_id, slot_key))
        slot_bdf_list = ToolConfig.getResource(slot_key).split(",")
        self._logger.info("get bdf list for bdf list{}".format(slot_bdf_list))
        for slot_address in slot_bdf_list:
            slot = slot_address.split("-")[0]
            address = slot_address.split("-")[1]
            nic_bdf_dict[slot] = address

    def _parse_fixed_slot_2_address_compatible(self):
        """
        解析总线兼容地址
        :return: 端口-地址配对列表
        """
        nic_bdf = dict()
        dev_platform = context_util.get_platform_id(self._context)
        for platform, resource in OM_BUSS_RESOURCE_COMPATIBLE.items():
            if platform != dev_platform:
                continue
            bdfs = ToolConfig.getResource(resource).split(",")
            for index, bdf in enumerate(bdfs):
                nic_bdf["nic{}".format(index + 1)] = bdf
            break
        return nic_bdf

    def _is_proprietary_hardware_dev(self):
        """
        是否是专有硬件设备
        :return: True/False
        """
        dev_platform = context_util.get_platform_id(self._context)
        return dev_platform in [Platform.PACIFIC, Platform.ATLANTIC, Platform.NEW_ATLANTIC]

    def _is_valid_slot_address(self, slot_address):
        # 全是0的地址无效
        return not re.match(r"^[:0.]+$", slot_address)

    def _is_slot_name_line(self, line):
        return SLOT_NAME_KEY in line

    def _get_slot_name_form_name_line(self, line):
        return line.strip().split(SLOT_NAME_KEY)[-1]

    def _is_slot_address_line(self, line):
        return SLOT_ADDRESS_KEY in line

    def _get_slot_address_form_address_line(self, line):
        return line.strip().split(SLOT_ADDRESS_KEY)[-1]

    def get_address_by_slot(self, slot):
        return self._slot_2_address.get(slot.lower())

    def get_logical_name(self, slot, port, slot_2_ports_dict):
        if self.is_support_no_slot() and not slot:
            return self._get_port_name_py_platform(port)
        if slot.lower() not in self._slot_2_address:
            raise DeployException(
                "not exist slot {}".format(slot),
                err_msg=entity.create_msg("not.exist.slot").format(slot),
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        slot_address = self._slot_2_address.get(slot.lower())
        ports, ssh_ret = self._get_slot_ports(slot_address)
        self._logger.info("query_for slot {} address :{},ports :{}".format(slot, slot_address, ports))
        # V6存在两种总线，查不到端口，使用兼容地址
        if len(ports) < 1 and self.is_v6_platform() and slot.lower() in self._slot_2_address_compatible:
            slot_address = self._slot_2_address_compatible.get(slot.lower())
            ports, ssh_ret = self._get_slot_ports(slot_address)
            self._logger.info("use compatible_nic : {}, addr:  {}, ports : {}".format(slot, slot_address, ports))
        port_num = self._get_port_num(port)
        if not self.is_port_valid(port_num, ports, slot_2_ports_dict.get(slot.lower())):
            self.raise_query_port_error_exception(port, slot, ssh_ret)
        return ports[int(port_num) - 1], ssh_ret

    def raise_query_port_error_exception(self, port, slot, ssh_ret):
        raise DeployException(
            "not exist slot {}".format(port),
            origin_info=ssh_ret,
            err_msg=entity.create_msg("not.exist.port").format(slot, port),
            err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def is_v6_platform(self):
        return context_util.get_platform_id(self._context) == Platform.X86_V6

    def is_port_valid(self, port_num, ports, config_total_ports):
        # LLD模板配置的对应slot的port数量应该与根据该slot查询出的真实port数量一致或者更少
        return len(config_total_ports) <= len(ports) and self._is_int_str(port_num) and len(ports) >= int(port_num) >= 1

    def _remove_slot_address_func_num(self, slot_address):
        addrs = slot_address.split(".")
        if len(addrs) == 2:
            return addrs[0] + "."
        self._logger.error("abnormal slot address: {}".format(slot_address))
        return slot_address

    def _get_logical_name(self, line):
        return line.split("/")[-1]

    def _get_port_num(self, port):
        return port.lower().split("port")[-1]

    def _is_int_str(self, port_num):
        try:
            int(port_num)
            return True
        except ValueError:
            return False

    def _get_slot_ports(self, slot_address):
        if ";" in slot_address:
            return self.get_port_loop_address(slot_address)
        return self.query_by_address(slot_address)

    def query_by_address(self, slot_address):
        # 去掉后面的function num，cx网卡会过滤不全
        cmd = QUERY_SLOT_INFO_CMD.format(self._remove_slot_address_func_num(
            slot_address))
        return self.query_for_ports(cmd)

    def query_for_ports(self, cmd):
        ssh_ret = ssh_util.exec_ssh_cmd_disable_invalid(self._context, cmd)
        ports = list()
        self._parse_port(ports, ssh_ret)
        return ports, ssh_ret

    def _parse_port(self, ports, ssh_ret):
        for line in ssh_ret.splitlines():
            self._add_port(line, ports)

    def _add_port(self, line, ports):
        # 排查原始命令的行
        if "ls -l" not in line and "/sys/class/net/" in line:
            self._check_net_config_valid(line)
            ports.append(self._get_logical_name(line))

    def _check_net_config_valid(self, line):
        # 网口软连接文件名必须和真实文件名保持一致，否则视为网口配置错误
        net_config = line.split('->')
        if len(net_config) < 2 or self._get_logical_name(net_config[0].strip()) != self._get_logical_name(
                net_config[1].strip()):
            raise DeployException('check net config failed',
                                  err_msg=entity.create_msg('os.net.config.error'),
                                  origin_info=line)

    def is_support_no_slot(self):
        return context_util.get_platform_id(self._context) in (Platform.EAST_SEA, Platform.SI_NAN, Platform.NEW_SI_NAN)

    def _get_port_name_py_platform(self, port):
        return self.get_nic_port(port, Platform.get_nic_pdf_key(context_util.get_platform_id(self._context)))

    def get_nic_port(self, port, resource_key):
        port_num = int(self._get_port_num(port))
        pdfs = ToolConfig.getResource(resource_key).split(",")
        self._logger.info("{}:{}".format(resource_key, pdfs))
        # 东海和司南硬件板载口，总共P0-P9 ,前6个总线地址为第一个，后四个总线地址为第二个
        if int(port_num) <= 5:
            ports, ret = self._get_slot_ports(pdfs[0])
            if port_num >= len(ports):
                self.raise_query_port_error_exception(port, "", ret)
            return ports[port_num], ret
        ports, ret = self._get_slot_ports(pdfs[1])
        # 东海硬件Port6-Port9是新的总线，查出来的端口名还是以0开始命名，所以配置的端口名序号需要做-6处理
        if port_num - 6 >= len(ports):
            self.raise_query_port_error_exception(port, "", ret)
        return ports[port_num - 6], ret

    def init_bdf_key(self):
        method = self._method_map.get(context_util.get_platform_id(self._context))
        if not method:
            return ""
        return method()

    def init_si_nan_key(self):
        """
        司南硬件在不同型号上，存在硬件配置不同的情况
        在介质框控制板软硬件接口文档中，分为配置1到配置9
        目前在存储使用的有配置4、5、6、7、9，通过以下方式可以在当前节点上查询到是对应的配置几
        命令及回显如下：
        [root@Subnet1-Node1 ~]# bsptool -c readcpldreg 0x10b
        cpldreg addr 0x10b value 0x4
        0x4表示是配置4，其余类推
        :return: 返回当前司南配置；非司南机型返回空串，不会使用到
        """
        try:
            ssh_ret = ssh_util.exec_ssh_cmd(self._context, "bsptool -c readcpldreg 0x10b")
        except DeployException as e:
            e.err_msg = entity.create_msg("can.not.find.net.info")
            raise e
        self.slot_ssh_info += ssh_ret
        lines = ssh_ret.splitlines()
        if len(lines) < 3:
            return ""
        # 取第二行回显的最后一个数据
        return "." + lines[1].split()[-1]

    def init_east_sea_bdf_key(self):
        self._logger.info("init east sea")
        deploy_type = context_util.get_deploy_node(self._context).getDevType()
        if '2 Node' in deploy_type:
            return ".2node"
        return ".1node"

    def get_port_loop_address(self, slot_address):
        """
        东海机型新增槽位，同样的机型存在多种bdf号的情况，
        当前识别同时只会存在一种，通过遍历多种可能的槽位查找端口，只要找到就返回
        若规则由变更，需重新适配
        :param slot_address: 配置的槽位地址，;分割多个地址
        :return: 端口, ssh命令回显
        """
        ssh_rets = list()
        for address in slot_address.split(";"):
            ports, ssh_ret = self.query_by_address(address)
            ssh_rets.append(ssh_ret)
            if ports:
                return ports, "\n".join(ssh_rets)
        return [], "\n".join(ssh_rets)
