/*
 * Decompiled with CFR 0.152.
 */
package io.nats.client.impl;

import io.nats.client.AuthenticationException;
import io.nats.client.Connection;
import io.nats.client.ConnectionListener;
import io.nats.client.Consumer;
import io.nats.client.Dispatcher;
import io.nats.client.JetStream;
import io.nats.client.JetStreamManagement;
import io.nats.client.JetStreamOptions;
import io.nats.client.KeyValue;
import io.nats.client.KeyValueManagement;
import io.nats.client.KeyValueOptions;
import io.nats.client.Message;
import io.nats.client.MessageHandler;
import io.nats.client.NUID;
import io.nats.client.Options;
import io.nats.client.ReconnectDelayHandler;
import io.nats.client.Statistics;
import io.nats.client.Subscription;
import io.nats.client.api.ServerInfo;
import io.nats.client.impl.DataPort;
import io.nats.client.impl.Headers;
import io.nats.client.impl.MessageQueue;
import io.nats.client.impl.NatsConnectionReader;
import io.nats.client.impl.NatsConnectionWriter;
import io.nats.client.impl.NatsConsumer;
import io.nats.client.impl.NatsDispatcher;
import io.nats.client.impl.NatsJetStream;
import io.nats.client.impl.NatsJetStreamManagement;
import io.nats.client.impl.NatsKeyValue;
import io.nats.client.impl.NatsKeyValueManagement;
import io.nats.client.impl.NatsMessage;
import io.nats.client.impl.NatsStatistics;
import io.nats.client.impl.NatsSubscription;
import io.nats.client.impl.NatsSubscriptionFactory;
import io.nats.client.support.ByteArrayBuilder;
import io.nats.client.support.NatsConstants;
import io.nats.client.support.NatsRequestCompletableFuture;
import io.nats.client.support.Validator;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class NatsConnection
implements Connection {
    private final Options options;
    private final NatsStatistics statistics;
    private boolean connecting;
    private boolean disconnecting;
    private boolean closing;
    private Exception exceptionDuringConnectChange;
    private Connection.Status status;
    private final ReentrantLock statusLock;
    private final Condition statusChanged;
    private CompletableFuture<DataPort> dataPortFuture;
    private DataPort dataPort;
    private String currentServer;
    private CompletableFuture<Boolean> reconnectWaiter;
    private final HashMap<String, String> serverAuthErrors;
    private final NatsConnectionReader reader;
    private final NatsConnectionWriter writer;
    private final AtomicReference<ServerInfo> serverInfo;
    private final Map<String, NatsSubscription> subscribers;
    private final Map<String, NatsDispatcher> dispatchers;
    private final Map<String, NatsRequestCompletableFuture> responsesAwaiting;
    private final Map<String, NatsRequestCompletableFuture> responsesRespondedTo;
    private final ConcurrentLinkedDeque<CompletableFuture<Boolean>> pongQueue;
    private final String mainInbox;
    private final AtomicReference<NatsDispatcher> inboxDispatcher;
    private Timer timer;
    private final AtomicBoolean needPing;
    private final AtomicLong nextSid;
    private final NUID nuid;
    private final AtomicReference<String> connectError;
    private final AtomicReference<String> lastError;
    private final AtomicReference<CompletableFuture<Boolean>> draining;
    private final AtomicBoolean blockPublishForDrain;
    private final ExecutorService callbackRunner;
    private final ExecutorService executor;
    private final ExecutorService connectExecutor;
    private final boolean advancedTracking;

    NatsConnection(Options options) {
        boolean trace = options.isTraceConnection();
        this.timeTrace(trace, "creating connection object");
        this.options = options;
        this.advancedTracking = options.isTrackAdvancedStats();
        this.statistics = new NatsStatistics(this.advancedTracking);
        this.statusLock = new ReentrantLock();
        this.statusChanged = this.statusLock.newCondition();
        this.status = Connection.Status.DISCONNECTED;
        this.reconnectWaiter = new CompletableFuture();
        this.reconnectWaiter.complete(Boolean.TRUE);
        this.dispatchers = new ConcurrentHashMap<String, NatsDispatcher>();
        this.subscribers = new ConcurrentHashMap<String, NatsSubscription>();
        this.responsesAwaiting = new ConcurrentHashMap<String, NatsRequestCompletableFuture>();
        this.responsesRespondedTo = new ConcurrentHashMap<String, NatsRequestCompletableFuture>();
        this.serverAuthErrors = new HashMap();
        this.nextSid = new AtomicLong(1L);
        this.timeTrace(trace, "creating NUID");
        this.nuid = new NUID();
        this.mainInbox = this.createInbox() + ".*";
        this.lastError = new AtomicReference();
        this.connectError = new AtomicReference();
        this.serverInfo = new AtomicReference();
        this.inboxDispatcher = new AtomicReference();
        this.pongQueue = new ConcurrentLinkedDeque();
        this.draining = new AtomicReference();
        this.blockPublishForDrain = new AtomicBoolean();
        this.timeTrace(trace, "creating executors");
        this.callbackRunner = Executors.newSingleThreadExecutor();
        this.executor = options.getExecutor();
        this.connectExecutor = Executors.newSingleThreadExecutor();
        this.timeTrace(trace, "creating reader and writer");
        this.reader = new NatsConnectionReader(this);
        this.writer = new NatsConnectionWriter(this);
        this.needPing = new AtomicBoolean(true);
        this.timeTrace(trace, "connection object created");
    }

    /*
     * Enabled aggressive block sorting
     */
    void connect(boolean reconnectOnConnect) throws InterruptedException, IOException {
        if (this.options.getServers().size() == 0) {
            throw new IllegalArgumentException("No servers provided in options");
        }
        boolean trace = this.options.isTraceConnection();
        long start = System.nanoTime();
        this.lastError.set("");
        this.timeTrace(trace, "starting connect loop");
        List<String> serversToTry = this.getServersToTry();
        for (String serverURI : serversToTry) {
            if (this.isClosed()) break;
            this.connectError.set("");
            this.timeTrace(trace, "setting status to connecting");
            this.updateStatus(Connection.Status.CONNECTING);
            this.timeTrace(trace, "trying to connect to %s", serverURI);
            this.tryToConnect(serverURI, System.nanoTime());
            if (this.isConnected()) break;
            this.timeTrace(trace, "setting status to disconnected");
            this.updateStatus(Connection.Status.DISCONNECTED);
            String err = this.connectError.get();
            if (!this.isAuthenticationError(err)) continue;
            this.serverAuthErrors.put(serverURI, err);
        }
        if (!this.isConnected() && !this.isClosed()) {
            String msg;
            if (reconnectOnConnect) {
                this.timeTrace(trace, "trying to reconnect on connect");
                this.reconnect();
                return;
            }
            this.timeTrace(trace, "connection failed, closing to cleanup");
            this.close();
            String err = this.connectError.get();
            if (this.isAuthenticationError(err)) {
                msg = "Authentication error connecting to NATS server: " + err;
                throw new AuthenticationException(msg);
            }
            msg = "Unable to connect to NATS servers: " + String.join((CharSequence)", ", serversToTry);
            throw new IOException(msg);
        }
        if (!trace) return;
        long end = System.nanoTime();
        double seconds = (double)(end - start) / 1.0E9;
        this.timeTrace(trace, "connect complete in %.3f seconds", seconds);
    }

    void reconnect() throws InterruptedException {
        long maxTries = this.options.getMaxReconnect();
        long tries = 0L;
        if (this.isClosed()) {
            return;
        }
        if (maxTries == 0L) {
            this.close();
            return;
        }
        this.writer.setReconnectMode(true);
        boolean doubleAuthError = false;
        while (!(this.isConnected() || this.isClosed() || this.isClosing())) {
            if (tries > 0L) {
                this.waitForReconnectTimeout(tries);
            }
            List<String> serversToTry = this.getServersToTry();
            for (String server : serversToTry) {
                if (this.isClosed()) break;
                this.connectError.set("");
                if (this.isDisconnectingOrClosed() || this.isClosing()) break;
                this.updateStatus(Connection.Status.RECONNECTING);
                this.tryToConnect(server, System.nanoTime());
                if (maxTries > 0L && ++tries >= maxTries) break;
                if (this.isConnected()) {
                    this.statistics.incrementReconnects();
                    break;
                }
                String err = this.connectError.get();
                if (!this.isAuthenticationError(err)) continue;
                if (err.equals(this.serverAuthErrors.get(server))) {
                    doubleAuthError = true;
                    break;
                }
                this.serverAuthErrors.put(server, err);
            }
            if (!doubleAuthError && (maxTries <= 0L || tries < maxTries)) continue;
            break;
        }
        if (!this.isConnected()) {
            this.close();
            return;
        }
        this.subscribers.forEach((sid, sub) -> {
            if (sub.getDispatcher() == null && !sub.isDraining()) {
                this.sendSubscriptionMessage(sub.getSID(), sub.getSubject(), sub.getQueueName(), true);
            }
        });
        this.dispatchers.forEach((nuid, d) -> {
            if (!d.isDraining()) {
                d.resendSubscriptions();
            }
        });
        try {
            this.flush(this.options.getConnectionTimeout());
        }
        catch (Exception exp) {
            this.processException(exp);
        }
        this.writer.setReconnectMode(false);
        this.processConnectionEvent(ConnectionListener.Events.RESUBSCRIBED);
    }

    void timeTrace(boolean trace, String format, Object ... args) {
        if (trace) {
            this._trace(String.format(format, args));
        }
    }

    void timeTrace(boolean trace, String message) {
        if (trace) {
            this._trace(message);
        }
    }

    private void _trace(String message) {
        String timeStr = DateTimeFormatter.ISO_TIME.format(LocalDateTime.now());
        System.out.println("[" + timeStr + "] connect trace: " + message);
    }

    long timeCheck(boolean trace, long endNanos, String message) throws TimeoutException {
        long now = System.nanoTime();
        long remaining = endNanos - now;
        if (trace) {
            double seconds = (double)remaining / 1.0E9;
            this._trace(message + String.format(", %.3f (s) remaining", seconds));
        }
        if (remaining < 0L) {
            throw new TimeoutException("connection timed out");
        }
        return remaining;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void tryToConnect(String serverURI, long now) {
        this.currentServer = null;
        try {
            Duration connectTimeout = this.options.getConnectionTimeout();
            boolean trace = this.options.isTraceConnection();
            long end = now + connectTimeout.toNanos();
            this.timeCheck(trace, end, "starting connection attempt");
            this.statusLock.lock();
            try {
                if (this.connecting) {
                    return;
                }
                this.connecting = true;
                this.statusChanged.signalAll();
            }
            finally {
                this.statusLock.unlock();
            }
            this.dataPortFuture = new CompletableFuture();
            long timeoutNanos = this.timeCheck(trace, end, "waiting for reader");
            this.reader.stop().get(timeoutNanos, TimeUnit.NANOSECONDS);
            timeoutNanos = this.timeCheck(trace, end, "waiting for writer");
            this.writer.stop().get(timeoutNanos, TimeUnit.NANOSECONDS);
            this.timeCheck(trace, end, "cleaning pong queue");
            this.cleanUpPongQueue();
            timeoutNanos = this.timeCheck(trace, end, "connecting data port");
            DataPort newDataPort = this.options.buildDataPort();
            newDataPort.connect(serverURI, this, timeoutNanos);
            this.dataPort = newDataPort;
            this.dataPortFuture.complete(this.dataPort);
            Callable<Object> connectTask = () -> {
                this.readInitialInfo();
                this.checkVersionRequirements();
                long start = System.nanoTime();
                this.upgradeToSecureIfNeeded();
                if (trace && this.options.isTLSRequired()) {
                    this.timeTrace(true, "TLS upgrade took: %.3f (s)", (double)(System.nanoTime() - start) / 1.0E9);
                }
                return null;
            };
            timeoutNanos = this.timeCheck(trace, end, "reading info, version and upgrading to secure if necessary");
            Future<Object> future = this.connectExecutor.submit(connectTask);
            try {
                future.get(timeoutNanos, TimeUnit.NANOSECONDS);
            }
            finally {
                future.cancel(true);
            }
            this.timeCheck(trace, end, "starting reader");
            this.reader.start(this.dataPortFuture);
            this.timeCheck(trace, end, "starting writer");
            this.writer.start(this.dataPortFuture);
            this.timeCheck(trace, end, "sending connect message");
            this.sendConnect(serverURI);
            timeoutNanos = this.timeCheck(trace, end, "sending initial ping");
            CompletableFuture<Boolean> pongFuture = this.sendPing();
            if (pongFuture != null) {
                pongFuture.get(timeoutNanos, TimeUnit.NANOSECONDS);
            }
            if (this.timer == null) {
                long cleanMillis;
                this.timeCheck(trace, end, "starting ping and cleanup timers");
                this.timer = new Timer("Nats Connection Timer");
                long pingMillis = this.options.getPingInterval().toMillis();
                if (pingMillis > 0L) {
                    this.timer.schedule(new TimerTask(){

                        @Override
                        public void run() {
                            if (NatsConnection.this.isConnected()) {
                                NatsConnection.this.softPing();
                            }
                        }
                    }, pingMillis, pingMillis);
                }
                if ((cleanMillis = this.options.getRequestCleanupInterval().toMillis()) > 0L) {
                    this.timer.schedule(new TimerTask(){

                        @Override
                        public void run() {
                            NatsConnection.this.cleanResponses(false);
                        }
                    }, cleanMillis, cleanMillis);
                }
            }
            this.timeCheck(trace, end, "updating status to connected");
            this.statusLock.lock();
            try {
                this.connecting = false;
                if (this.exceptionDuringConnectChange != null) {
                    throw this.exceptionDuringConnectChange;
                }
                this.currentServer = serverURI;
                this.serverAuthErrors.remove(serverURI);
                this.updateStatus(Connection.Status.CONNECTED);
            }
            finally {
                this.statusLock.unlock();
            }
            this.timeTrace(trace, "status updated");
        }
        catch (RuntimeException exp) {
            this.processException(exp);
            throw exp;
        }
        catch (Exception exp) {
            this.processException(exp);
            try {
                this.closeSocket(false);
            }
            catch (InterruptedException e) {
                this.processException(e);
            }
        }
        finally {
            this.statusLock.lock();
            try {
                this.connecting = false;
                this.statusChanged.signalAll();
            }
            finally {
                this.statusLock.unlock();
            }
        }
    }

    void checkVersionRequirements() throws IOException {
        Options opts = this.getOptions();
        ServerInfo info = this.getInfo();
        if (opts.isNoEcho() && info.getProtocolVersion() < 1) {
            throw new IOException("Server does not support no echo.");
        }
    }

    void upgradeToSecureIfNeeded() throws IOException {
        Options opts = this.getOptions();
        ServerInfo info = this.getInfo();
        if (opts.isTLSRequired() && !info.isTLSRequired()) {
            throw new IOException("SSL connection wanted by client.");
        }
        if (!opts.isTLSRequired() && info.isTLSRequired()) {
            throw new IOException("SSL required by server.");
        }
        if (opts.isTLSRequired()) {
            this.dataPort.upgradeToSecure();
        }
    }

    void handleCommunicationIssue(Exception io) {
        this.statusLock.lock();
        try {
            if (this.connecting || this.disconnecting || this.status == Connection.Status.CLOSED || this.isDraining()) {
                this.exceptionDuringConnectChange = io;
                return;
            }
        }
        finally {
            this.statusLock.unlock();
        }
        this.processException(io);
        this.executor.submit(() -> {
            try {
                this.closeSocket(true);
            }
            catch (InterruptedException e) {
                this.processException(e);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void closeSocket(boolean tryReconnectIfConnected) throws InterruptedException {
        boolean wasConnected;
        this.statusLock.lock();
        try {
            if (this.isDisconnectingOrClosed()) {
                this.waitForDisconnectOrClose(this.options.getConnectionTimeout());
                return;
            }
            this.disconnecting = true;
            this.exceptionDuringConnectChange = null;
            wasConnected = this.status == Connection.Status.CONNECTED;
            this.statusChanged.signalAll();
        }
        finally {
            this.statusLock.unlock();
        }
        this.closeSocketImpl();
        this.statusLock.lock();
        try {
            this.updateStatus(Connection.Status.DISCONNECTED);
            this.exceptionDuringConnectChange = null;
            this.disconnecting = false;
            this.statusChanged.signalAll();
        }
        finally {
            this.statusLock.unlock();
        }
        if (this.isClosing()) {
            this.close();
        } else if (wasConnected && tryReconnectIfConnected) {
            this.reconnect();
        }
    }

    @Override
    public void close() throws InterruptedException {
        this.close(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close(boolean checkDrainStatus) throws InterruptedException {
        this.statusLock.lock();
        try {
            if (checkDrainStatus && this.isDraining()) {
                this.waitForDisconnectOrClose(this.options.getConnectionTimeout());
                return;
            }
            this.closing = true;
            if (this.isDisconnectingOrClosed()) {
                this.waitForDisconnectOrClose(this.options.getConnectionTimeout());
                return;
            }
            this.disconnecting = true;
            this.exceptionDuringConnectChange = null;
            this.statusChanged.signalAll();
        }
        finally {
            this.statusLock.unlock();
        }
        if (this.reconnectWaiter != null) {
            this.reconnectWaiter.cancel(true);
        }
        this.closeSocketImpl();
        this.dispatchers.forEach((nuid, d) -> d.stop(false));
        this.subscribers.forEach((sid, sub) -> sub.invalidate());
        this.dispatchers.clear();
        this.subscribers.clear();
        if (this.timer != null) {
            this.timer.cancel();
            this.timer = null;
        }
        this.cleanResponses(true);
        this.cleanUpPongQueue();
        this.statusLock.lock();
        try {
            this.updateStatus(Connection.Status.CLOSED);
        }
        finally {
            this.statusLock.unlock();
        }
        this.callbackRunner.shutdown();
        try {
            this.callbackRunner.awaitTermination(this.options.getConnectionTimeout().toNanos(), TimeUnit.NANOSECONDS);
        }
        finally {
            this.callbackRunner.shutdownNow();
        }
        this.connectExecutor.shutdownNow();
        this.statusLock.lock();
        try {
            this.disconnecting = false;
            this.statusChanged.signalAll();
        }
        finally {
            this.statusLock.unlock();
        }
    }

    void closeSocketImpl() {
        this.currentServer = null;
        Future<Boolean> readStop = this.reader.stop();
        Future<Boolean> writeStop = this.writer.stop();
        try {
            readStop.get(1L, TimeUnit.SECONDS);
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            writeStop.get(1L, TimeUnit.SECONDS);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.dataPortFuture.cancel(true);
        try {
            if (this.dataPort != null) {
                this.dataPort.close();
            }
        }
        catch (IOException ex) {
            this.processException(ex);
        }
        this.cleanUpPongQueue();
        try {
            this.reader.stop().get(10L, TimeUnit.SECONDS);
        }
        catch (Exception ex) {
            this.processException(ex);
        }
        try {
            this.writer.stop().get(10L, TimeUnit.SECONDS);
        }
        catch (Exception ex) {
            this.processException(ex);
        }
    }

    void cleanUpPongQueue() {
        Future b;
        while ((b = (Future)this.pongQueue.poll()) != null) {
            try {
                b.cancel(true);
            }
            catch (CancellationException e) {
                if (b.isDone() || b.isCancelled()) continue;
                this.processException(e);
            }
        }
    }

    @Override
    public void publish(String subject, byte[] body) {
        this.publishInternal(subject, null, null, body, this.options.supportUTF8Subjects());
    }

    @Override
    public void publish(String subject, String replyTo, byte[] body) {
        this.publishInternal(subject, replyTo, null, body, this.options.supportUTF8Subjects());
    }

    @Override
    public void publish(Message message) {
        Validator.validateNotNull(message, "Message");
        this.publishInternal(message.getSubject(), message.getReplyTo(), message.getHeaders(), message.getData(), message.isUtf8mode());
    }

    void publishInternal(String subject, String replyTo, Headers headers, byte[] data, boolean utf8mode) {
        this.checkIfNeedsHeaderSupport(headers);
        this.checkPayloadSize(data);
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (this.blockPublishForDrain.get()) {
            throw new IllegalStateException("Connection is Draining");
        }
        NatsMessage nm = new NatsMessage(subject, replyTo, new Headers(headers), data, utf8mode);
        Connection.Status stat = this.status;
        if (!(stat != Connection.Status.RECONNECTING && stat != Connection.Status.DISCONNECTED || this.writer.canQueueDuringReconnect(nm))) {
            throw new IllegalStateException("Unable to queue any more messages during reconnect, max buffer is " + this.options.getReconnectBufferSize());
        }
        this.queueOutgoing(nm);
    }

    private void checkIfNeedsHeaderSupport(Headers headers) {
        if (headers != null && !headers.isEmpty() && !this.serverInfo.get().isHeadersSupported()) {
            throw new IllegalArgumentException("Headers are not supported by the server, version: " + this.serverInfo.get().getVersion());
        }
    }

    private void checkPayloadSize(byte[] body) {
        if (this.options.clientSideLimitChecks() && body != null && (long)body.length > this.getMaxPayload() && this.getMaxPayload() > 0L) {
            throw new IllegalArgumentException("Message payload size exceed server configuration " + body.length + " vs " + this.getMaxPayload());
        }
    }

    @Override
    public Subscription subscribe(String subject) {
        if (subject == null || subject.length() == 0) {
            throw new IllegalArgumentException("Subject is required in subscribe");
        }
        Pattern pattern = Pattern.compile("\\s");
        Matcher matcher = pattern.matcher(subject);
        if (matcher.find()) {
            throw new IllegalArgumentException("Subject cannot contain whitespace");
        }
        return this.createSubscription(subject, null, null, null);
    }

    @Override
    public Subscription subscribe(String subject, String queueName) {
        if (subject == null || subject.length() == 0) {
            throw new IllegalArgumentException("Subject is required in subscribe");
        }
        Pattern pattern = Pattern.compile("\\s");
        Matcher smatcher = pattern.matcher(subject);
        if (smatcher.find()) {
            throw new IllegalArgumentException("Subject cannot contain whitespace");
        }
        if (queueName == null || queueName.length() == 0) {
            throw new IllegalArgumentException("QueueName is required in subscribe");
        }
        Matcher qmatcher = pattern.matcher(queueName);
        if (qmatcher.find()) {
            throw new IllegalArgumentException("Queue names cannot contain whitespace");
        }
        return this.createSubscription(subject, queueName, null, null);
    }

    void invalidate(NatsSubscription sub) {
        this.remove(sub);
        sub.invalidate();
    }

    void remove(NatsSubscription sub) {
        String sid = sub.getSID();
        this.subscribers.remove(sid);
        if (sub.getNatsDispatcher() != null) {
            sub.getNatsDispatcher().remove(sub);
        }
    }

    void unsubscribe(NatsSubscription sub, int after) {
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (after <= 0) {
            this.invalidate(sub);
        } else {
            sub.setUnsubLimit(after);
            if (sub.reachedUnsubLimit()) {
                sub.invalidate();
            }
        }
        if (!this.isConnected()) {
            return;
        }
        this.sendUnsub(sub, after);
    }

    void sendUnsub(NatsSubscription sub, int after) {
        ByteArrayBuilder bab = new ByteArrayBuilder().append(NatsConstants.UNSUB_SP_BYTES).append(sub.getSID());
        if (after > 0) {
            bab.append((byte)32).append(after);
        }
        this.queueInternalOutgoing(new NatsMessage.ProtocolMessage(bab));
    }

    NatsSubscription createSubscription(String subject, String queueName, NatsDispatcher dispatcher, NatsSubscriptionFactory factory) {
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (this.isDraining() && (dispatcher == null || dispatcher != this.inboxDispatcher.get())) {
            throw new IllegalStateException("Connection is Draining");
        }
        String sid = this.getNextSid();
        NatsSubscription sub = factory == null ? new NatsSubscription(sid, subject, queueName, this, dispatcher) : factory.createNatsSubscription(sid, subject, queueName, this, dispatcher);
        this.subscribers.put(sid, sub);
        this.sendSubscriptionMessage(sid, subject, queueName, false);
        return sub;
    }

    String getNextSid() {
        return Long.toString(this.nextSid.getAndIncrement());
    }

    String reSubscribe(NatsSubscription sub, String subject, String queueName) {
        String sid = this.getNextSid();
        this.sendSubscriptionMessage(sid, subject, queueName, false);
        this.subscribers.put(sid, sub);
        return sid;
    }

    void sendSubscriptionMessage(String sid, String subject, String queueName, boolean treatAsInternal) {
        if (!this.isConnected()) {
            return;
        }
        ByteArrayBuilder bab = new ByteArrayBuilder(StandardCharsets.UTF_8).append(NatsConstants.SUB_SP_BYTES).append(subject);
        if (queueName != null) {
            bab.append((byte)32).append(queueName);
        }
        bab.append((byte)32).append(sid);
        NatsMessage.ProtocolMessage subMsg = new NatsMessage.ProtocolMessage(bab);
        if (treatAsInternal) {
            this.queueInternalOutgoing(subMsg);
        } else {
            this.queueOutgoing(subMsg);
        }
    }

    @Override
    public String createInbox() {
        return this.options.getInboxPrefix() + this.nuid.next();
    }

    int getRespInboxLength() {
        return this.options.getInboxPrefix().length() + 22 + 1;
    }

    String createResponseInbox(String inbox) {
        return inbox.substring(0, this.getRespInboxLength()) + this.nuid.next();
    }

    String getResponseToken(String responseInbox) {
        int len = this.getRespInboxLength();
        if (responseInbox.length() <= len) {
            return responseInbox;
        }
        return responseInbox.substring(len);
    }

    void cleanResponses(boolean closing) {
        ArrayList toRemove = new ArrayList();
        this.responsesAwaiting.forEach((key, future) -> {
            boolean remove = false;
            if (future.hasExceededTimeout()) {
                remove = true;
                future.cancelTimedOut();
            } else if (closing) {
                remove = true;
                future.cancelClosing();
            } else if (future.isDone()) {
                remove = true;
            }
            if (remove) {
                toRemove.add(key);
                this.statistics.decrementOutstandingRequests();
            }
        });
        for (String token : toRemove) {
            this.responsesAwaiting.remove(token);
        }
        if (this.advancedTracking) {
            toRemove.clear();
            this.responsesRespondedTo.forEach((key, future) -> {
                if (future.hasExceededTimeout()) {
                    toRemove.add(key);
                }
            });
            for (String token : toRemove) {
                this.responsesRespondedTo.remove(token);
            }
        }
    }

    @Override
    public Message request(String subject, byte[] body, Duration timeout) throws InterruptedException {
        return this.requestInternal(subject, null, body, this.options.supportUTF8Subjects(), timeout, true);
    }

    @Override
    public Message request(Message message, Duration timeout) throws InterruptedException {
        Validator.validateNotNull(message, "Message");
        return this.requestInternal(message.getSubject(), message.getHeaders(), message.getData(), message.isUtf8mode(), timeout, true);
    }

    Message requestInternal(String subject, Headers headers, byte[] data, boolean utf8mode, Duration timeout, boolean cancelOn503) throws InterruptedException {
        CompletableFuture<Message> incoming = this.requestFutureInternal(subject, headers, data, utf8mode, timeout, cancelOn503);
        try {
            return incoming.get(timeout.toNanos(), TimeUnit.NANOSECONDS);
        }
        catch (CancellationException | ExecutionException | TimeoutException e) {
            return null;
        }
    }

    @Override
    public CompletableFuture<Message> request(String subject, byte[] body) {
        return this.requestFutureInternal(subject, null, body, this.options.supportUTF8Subjects(), null, true);
    }

    @Override
    public CompletableFuture<Message> requestWithTimeout(String subject, byte[] body, Duration timeout) {
        return this.requestFutureInternal(subject, null, body, this.options.supportUTF8Subjects(), timeout, true);
    }

    @Override
    public CompletableFuture<Message> requestWithTimeout(Message message, Duration timeout) {
        Validator.validateNotNull(message, "Message");
        return this.requestFutureInternal(message.getSubject(), message.getHeaders(), message.getData(), message.isUtf8mode(), timeout, true);
    }

    @Override
    public CompletableFuture<Message> request(Message message) {
        Validator.validateNotNull(message, "Message");
        return this.requestFutureInternal(message.getSubject(), message.getHeaders(), message.getData(), message.isUtf8mode(), null, true);
    }

    CompletableFuture<Message> requestFutureInternal(String subject, Headers headers, byte[] data, boolean utf8mode, Duration futureTimeout, boolean cancelOn503) {
        boolean oldStyle;
        NatsDispatcher d;
        this.checkPayloadSize(data);
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (this.isDraining()) {
            throw new IllegalStateException("Connection is Draining");
        }
        if (this.inboxDispatcher.get() == null && this.inboxDispatcher.compareAndSet(null, d = new NatsDispatcher(this, this::deliverReply))) {
            String id = this.nuid.next();
            this.dispatchers.put(id, d);
            d.start(id);
            d.subscribe(this.mainInbox);
        }
        String responseInbox = (oldStyle = this.options.isOldRequestStyle()) ? this.createInbox() : this.createResponseInbox(this.mainInbox);
        String responseToken = this.getResponseToken(responseInbox);
        NatsRequestCompletableFuture future = new NatsRequestCompletableFuture(cancelOn503, futureTimeout == null ? this.options.getRequestCleanupInterval() : futureTimeout);
        if (!oldStyle) {
            this.responsesAwaiting.put(responseToken, future);
        }
        this.statistics.incrementOutstandingRequests();
        if (oldStyle) {
            NatsDispatcher dispatcher = this.inboxDispatcher.get();
            NatsSubscription sub = dispatcher.subscribeReturningSubscription(responseInbox);
            dispatcher.unsubscribe(responseInbox, 1);
            future.whenComplete((msg, exception) -> {
                if (exception instanceof CancellationException) {
                    dispatcher.unsubscribe(responseInbox);
                }
            });
            this.responsesAwaiting.put(sub.getSID(), future);
        }
        this.publishInternal(subject, responseInbox, headers, data, utf8mode);
        this.writer.flushBuffer();
        this.statistics.incrementRequestsSent();
        return future;
    }

    void deliverReply(Message msg) {
        boolean oldStyle = this.options.isOldRequestStyle();
        String subject = msg.getSubject();
        String token = this.getResponseToken(subject);
        String key = oldStyle ? msg.getSID() : token;
        NatsRequestCompletableFuture f = this.responsesAwaiting.remove(key);
        if (f != null) {
            if (this.advancedTracking) {
                this.responsesRespondedTo.put(key, f);
            }
            this.statistics.decrementOutstandingRequests();
            if (msg.isStatusMessage() && msg.getStatus().getCode() == 503 && f.isCancelOn503()) {
                f.cancel(true);
            } else {
                f.complete(msg);
            }
            this.statistics.incrementRepliesReceived();
        } else if (!oldStyle && !subject.startsWith(this.mainInbox)) {
            if (this.advancedTracking && this.responsesRespondedTo.get(key) != null) {
                this.statistics.incrementDuplicateRepliesReceived();
            } else {
                this.statistics.incrementOrphanRepliesReceived();
            }
        }
    }

    @Override
    public Dispatcher createDispatcher() {
        return this.createDispatcher(null);
    }

    @Override
    public Dispatcher createDispatcher(MessageHandler handler) {
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (this.isDraining()) {
            throw new IllegalStateException("Connection is Draining");
        }
        NatsDispatcher dispatcher = new NatsDispatcher(this, handler);
        String id = this.nuid.next();
        this.dispatchers.put(id, dispatcher);
        dispatcher.start(id);
        return dispatcher;
    }

    @Override
    public void closeDispatcher(Dispatcher d) {
        if (this.isClosed()) {
            throw new IllegalStateException("Connection is Closed");
        }
        if (!(d instanceof NatsDispatcher)) {
            throw new IllegalArgumentException("Connection can only manage its own dispatchers");
        }
        NatsDispatcher nd = (NatsDispatcher)d;
        if (nd.isDraining()) {
            return;
        }
        if (!this.dispatchers.containsKey(nd.getId())) {
            throw new IllegalArgumentException("Dispatcher is already closed.");
        }
        this.cleanupDispatcher(nd);
    }

    void cleanupDispatcher(NatsDispatcher nd) {
        nd.stop(true);
        this.dispatchers.remove(nd.getId());
    }

    @Override
    public void flush(Duration timeout) throws TimeoutException, InterruptedException {
        Instant start = Instant.now();
        this.waitForConnectOrClose(timeout);
        if (this.isClosed()) {
            throw new TimeoutException("Attempted to flush while closed");
        }
        if (timeout == null) {
            timeout = Duration.ZERO;
        }
        Instant now = Instant.now();
        Duration waitTime = Duration.between(start, now);
        if (!timeout.equals(Duration.ZERO) && waitTime.compareTo(timeout) >= 0) {
            throw new TimeoutException("Timeout out waiting for connection before flush.");
        }
        try {
            CompletableFuture<Boolean> waitForIt = this.sendPing();
            if (waitForIt == null) {
                return;
            }
            long nanos = timeout.toNanos();
            if (nanos > 0L) {
                if ((nanos -= waitTime.toNanos()) <= 0L) {
                    nanos = 1L;
                }
                waitForIt.get(nanos, TimeUnit.NANOSECONDS);
            } else {
                waitForIt.get();
            }
            this.statistics.incrementFlushCounter();
        }
        catch (CancellationException | ExecutionException e) {
            throw new TimeoutException(e.toString());
        }
    }

    void sendConnect(String serverURI) throws IOException {
        try {
            ServerInfo info = this.serverInfo.get();
            CharBuffer connectOptions = this.options.buildProtocolConnectOptionsString(serverURI, info.isAuthRequired(), info.getNonce());
            ByteArrayBuilder bab = new ByteArrayBuilder(NatsConstants.OP_CONNECT_SP_LEN + connectOptions.limit(), 64, StandardCharsets.UTF_8).append(NatsConstants.CONNECT_SP_BYTES).append(connectOptions);
            this.queueInternalOutgoing(new NatsMessage.ProtocolMessage(bab));
        }
        catch (Exception exp) {
            throw new IOException("Error sending connect string", exp);
        }
    }

    CompletableFuture<Boolean> sendPing() {
        return this.sendPing(true);
    }

    CompletableFuture<Boolean> softPing() {
        return this.sendPing(false);
    }

    CompletableFuture<Boolean> sendPing(boolean treatAsInternal) {
        int max = this.options.getMaxPingsOut();
        if (!this.isConnectedOrConnecting()) {
            CompletableFuture<Boolean> retVal = new CompletableFuture<Boolean>();
            retVal.complete(Boolean.FALSE);
            return retVal;
        }
        if (!treatAsInternal && !this.needPing.get()) {
            CompletableFuture<Boolean> retVal = new CompletableFuture<Boolean>();
            retVal.complete(Boolean.TRUE);
            this.needPing.set(true);
            return retVal;
        }
        if (max > 0 && this.pongQueue.size() + 1 > max) {
            this.handleCommunicationIssue(new IllegalStateException("Max outgoing Ping count exceeded."));
            return null;
        }
        CompletableFuture<Boolean> pongFuture = new CompletableFuture<Boolean>();
        NatsMessage.ProtocolMessage msg = new NatsMessage.ProtocolMessage(NatsConstants.OP_PING_BYTES);
        this.pongQueue.add(pongFuture);
        if (treatAsInternal) {
            this.queueInternalOutgoing(msg);
        } else {
            this.queueOutgoing(msg);
        }
        this.needPing.set(true);
        this.statistics.incrementPingCount();
        return pongFuture;
    }

    void sendPong() {
        this.queueInternalOutgoing(new NatsMessage.ProtocolMessage(NatsConstants.OP_PONG_BYTES));
    }

    void handlePong() {
        CompletableFuture<Boolean> pongFuture = this.pongQueue.pollFirst();
        if (pongFuture != null) {
            pongFuture.complete(Boolean.TRUE);
        }
    }

    void readInitialInfo() throws IOException {
        int read;
        byte[] readBuffer = new byte[this.options.getBufferSize()];
        ByteBuffer protocolBuffer = ByteBuffer.allocate(this.options.getBufferSize());
        boolean gotCRLF = false;
        boolean gotCR = false;
        block0: while (!gotCRLF && (read = this.dataPort.read(readBuffer, 0, readBuffer.length)) >= 0) {
            int i = 0;
            while (i < read) {
                byte b = readBuffer[i++];
                if (gotCR) {
                    if (b != 10) {
                        throw new IOException("Missed LF after CR waiting for INFO.");
                    }
                    if (i < read) {
                        throw new IOException("Read past initial info message.");
                    }
                    gotCRLF = true;
                    continue block0;
                }
                if (b == 13) {
                    gotCR = true;
                    continue;
                }
                if (!protocolBuffer.hasRemaining()) {
                    protocolBuffer = this.enlargeBuffer(protocolBuffer, 0);
                }
                protocolBuffer.put(b);
            }
        }
        if (!gotCRLF) {
            throw new IOException("Failed to read initial info message.");
        }
        protocolBuffer.flip();
        String infoJson = StandardCharsets.UTF_8.decode(protocolBuffer).toString();
        infoJson = infoJson.trim();
        String[] msg = infoJson.split("\\s");
        String op = msg[0].toUpperCase();
        if (!"INFO".equals(op)) {
            throw new IOException("Received non-info initial message.");
        }
        this.handleInfo(infoJson);
    }

    void handleInfo(String infoJson) {
        ServerInfo serverInfo = new ServerInfo(infoJson);
        this.serverInfo.set(serverInfo);
        List<String> urls = this.serverInfo.get().getConnectURLs();
        if (urls != null && urls.size() > 0) {
            this.processConnectionEvent(ConnectionListener.Events.DISCOVERED_SERVERS);
        }
        if (serverInfo.isLameDuckMode()) {
            this.processConnectionEvent(ConnectionListener.Events.LAME_DUCK);
        }
    }

    void queueOutgoing(NatsMessage msg) {
        if (msg.getControlLineLength() > this.options.getMaxControlLine()) {
            throw new IllegalArgumentException("Control line is too long");
        }
        if (!this.writer.queue(msg)) {
            this.options.getErrorListener().messageDiscarded(this, msg);
        }
    }

    void queueInternalOutgoing(NatsMessage msg) {
        if (msg.getControlLineLength() > this.options.getMaxControlLine()) {
            throw new IllegalArgumentException("Control line is too long");
        }
        this.writer.queueInternalMessage(msg);
    }

    void deliverMessage(NatsMessage msg) {
        this.needPing.set(false);
        this.statistics.incrementInMsgs();
        this.statistics.incrementInBytes(msg.getSizeInBytes());
        NatsSubscription sub = this.subscribers.get(msg.getSID());
        if (sub != null) {
            MessageQueue q;
            msg.setSubscription(sub);
            NatsDispatcher d = sub.getNatsDispatcher();
            NatsConsumer c = d == null ? sub : d;
            MessageQueue messageQueue = q = d == null ? sub.getMessageQueue() : d.getMessageQueue();
            if (c.hasReachedPendingLimits()) {
                this.statistics.incrementDroppedCount();
                c.incrementDroppedCount();
                if (!c.isMarkedSlow()) {
                    c.markSlow();
                    this.processSlowConsumer(c);
                }
            } else if (q != null) {
                c.markNotSlow();
                msg = sub.getBeforeQueueProcessor().apply(msg);
                if (msg != null) {
                    q.push(msg);
                }
            }
        }
    }

    void processOK() {
        this.statistics.incrementOkCount();
    }

    void processSlowConsumer(Consumer consumer) {
        if (!this.callbackRunner.isShutdown()) {
            try {
                this.callbackRunner.execute(() -> {
                    try {
                        this.options.getErrorListener().slowConsumerDetected(this, consumer);
                    }
                    catch (Exception ex) {
                        this.statistics.incrementExceptionCount();
                    }
                });
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }
    }

    void processException(Exception exp) {
        this.statistics.incrementExceptionCount();
        if (!this.callbackRunner.isShutdown()) {
            try {
                this.callbackRunner.execute(() -> {
                    try {
                        this.options.getErrorListener().exceptionOccurred(this, exp);
                    }
                    catch (Exception ex) {
                        this.statistics.incrementExceptionCount();
                    }
                });
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }
    }

    void processError(String errorText) {
        this.statistics.incrementErrCount();
        this.lastError.set(errorText);
        this.connectError.set(errorText);
        String url = this.getConnectedUrl();
        if (this.isConnected() && this.isAuthenticationError(errorText) && url != null) {
            this.serverAuthErrors.put(url, errorText);
        }
        if (!this.callbackRunner.isShutdown()) {
            try {
                this.callbackRunner.execute(() -> {
                    try {
                        this.options.getErrorListener().errorOccurred(this, errorText);
                    }
                    catch (Exception ex) {
                        this.statistics.incrementExceptionCount();
                    }
                });
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }
    }

    void processConnectionEvent(ConnectionListener.Events type) {
        ConnectionListener handler = this.options.getConnectionListener();
        if (handler != null && !this.callbackRunner.isShutdown()) {
            try {
                this.callbackRunner.execute(() -> {
                    try {
                        handler.connectionEvent(this, type);
                    }
                    catch (Exception ex) {
                        this.statistics.incrementExceptionCount();
                    }
                });
            }
            catch (RejectedExecutionException rejectedExecutionException) {
                // empty catch block
            }
        }
    }

    @Override
    public ServerInfo getServerInfo() {
        return this.getInfo();
    }

    ServerInfo getInfo() {
        return this.serverInfo.get();
    }

    @Override
    public Options getOptions() {
        return this.options;
    }

    @Override
    public Statistics getStatistics() {
        return this.statistics;
    }

    NatsStatistics getNatsStatistics() {
        return this.statistics;
    }

    DataPort getDataPort() {
        return this.dataPort;
    }

    int getConsumerCount() {
        return this.subscribers.size() + this.dispatchers.size();
    }

    @Override
    public long getMaxPayload() {
        ServerInfo info = this.serverInfo.get();
        if (info == null) {
            return -1L;
        }
        return info.getMaxPayload();
    }

    @Override
    public Collection<String> getServers() {
        ArrayList<String> servers = new ArrayList<String>();
        this.addOptionsServers(servers);
        this.addDiscoveredServers(servers);
        return servers;
    }

    protected List<String> getServersToTry() {
        if (this.options.getServerListProvider() == null) {
            ArrayList<String> servers = new ArrayList<String>();
            this.addOptionsServers(servers);
            if (!this.options.isIgnoreDiscoveredServers()) {
                this.addDiscoveredServers(servers);
            }
            return this.options.isNoRandomize() ? servers : this.randomize(servers);
        }
        return this.options.getServerListProvider().getServerList(this.currentServer, this.options.getUnprocessedServers(), this.getDiscoveredConnectUrls());
    }

    protected void addOptionsServers(List<String> servers) {
        for (URI uri : this.options.getServers()) {
            String srv = uri.toString();
            if (servers.contains(srv)) continue;
            servers.add(srv);
        }
    }

    protected List<String> getDiscoveredConnectUrls() {
        ServerInfo info = this.serverInfo.get();
        if (info == null) {
            return new ArrayList<String>();
        }
        List<String> urls = info.getConnectURLs();
        if (urls == null) {
            return new ArrayList<String>();
        }
        return info.getConnectURLs();
    }

    protected void addDiscoveredServers(List<String> servers) {
        List<String> urls = this.getDiscoveredConnectUrls();
        for (String url : urls) {
            try {
                String srv = this.options.createURIForServer(url).toString();
                if (servers.contains(srv)) continue;
                servers.add(srv);
            }
            catch (URISyntaxException uRISyntaxException) {}
        }
    }

    private List<String> randomize(List<String> servers) {
        if (servers.size() > 1) {
            if (this.currentServer != null) {
                servers.remove(this.currentServer);
            }
            Collections.shuffle(servers, ThreadLocalRandom.current());
            if (this.currentServer != null) {
                servers.add(this.currentServer);
            }
        }
        return servers;
    }

    @Override
    public String getConnectedUrl() {
        return this.currentServer;
    }

    @Override
    public Connection.Status getStatus() {
        return this.status;
    }

    @Override
    public String getLastError() {
        return this.lastError.get();
    }

    ExecutorService getExecutor() {
        return this.executor;
    }

    void updateStatus(Connection.Status newStatus) {
        Connection.Status oldStatus = this.status;
        this.statusLock.lock();
        try {
            if (oldStatus == Connection.Status.CLOSED || newStatus == oldStatus) {
                return;
            }
            this.status = newStatus;
        }
        finally {
            this.statusChanged.signalAll();
            this.statusLock.unlock();
        }
        if (this.status == Connection.Status.DISCONNECTED) {
            this.processConnectionEvent(ConnectionListener.Events.DISCONNECTED);
        } else if (this.status == Connection.Status.CLOSED) {
            this.processConnectionEvent(ConnectionListener.Events.CLOSED);
        } else if (oldStatus == Connection.Status.RECONNECTING && this.status == Connection.Status.CONNECTED) {
            this.processConnectionEvent(ConnectionListener.Events.RECONNECTED);
        } else if (this.status == Connection.Status.CONNECTED) {
            this.processConnectionEvent(ConnectionListener.Events.CONNECTED);
        }
    }

    boolean isClosing() {
        return this.closing;
    }

    boolean isClosed() {
        return this.status == Connection.Status.CLOSED;
    }

    boolean isConnected() {
        return this.status == Connection.Status.CONNECTED;
    }

    boolean isConnectedOrConnecting() {
        this.statusLock.lock();
        try {
            boolean bl = this.status == Connection.Status.CONNECTED || this.connecting;
            return bl;
        }
        finally {
            this.statusLock.unlock();
        }
    }

    boolean isDisconnectingOrClosed() {
        this.statusLock.lock();
        try {
            boolean bl = this.status == Connection.Status.CLOSED || this.disconnecting;
            return bl;
        }
        finally {
            this.statusLock.unlock();
        }
    }

    boolean isDisconnecting() {
        this.statusLock.lock();
        try {
            boolean bl = this.disconnecting;
            return bl;
        }
        finally {
            this.statusLock.unlock();
        }
    }

    void waitForDisconnectOrClose(Duration timeout) throws InterruptedException {
        this.waitFor(timeout, Void2 -> this.isDisconnecting() && !this.isClosed());
    }

    void waitForConnectOrClose(Duration timeout) throws InterruptedException {
        this.waitFor(timeout, Void2 -> !this.isConnected() && !this.isClosed());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void waitFor(Duration timeout, Predicate<Void> test) throws InterruptedException {
        this.statusLock.lock();
        try {
            long currentWaitNanos = timeout != null ? timeout.toNanos() : -1L;
            long start = System.nanoTime();
            while (currentWaitNanos >= 0L && test.test(null)) {
                if (currentWaitNanos > 0L) {
                    this.statusChanged.await(currentWaitNanos, TimeUnit.NANOSECONDS);
                    long now = System.nanoTime();
                    if ((currentWaitNanos -= now - (start = now)) > 0L) continue;
                    break;
                }
                this.statusChanged.await();
            }
        }
        finally {
            this.statusLock.unlock();
        }
    }

    void waitForReconnectTimeout(long totalTries) {
        long currentWaitNanos = 0L;
        ReconnectDelayHandler handler = this.options.getReconnectDelayHandler();
        if (handler == null) {
            Duration dur = this.options.getReconnectWait();
            if (dur != null) {
                currentWaitNanos = dur.toNanos();
                Duration duration = dur = this.options.isTLSRequired() ? this.options.getReconnectJitterTls() : this.options.getReconnectJitter();
                if (dur != null) {
                    currentWaitNanos += ThreadLocalRandom.current().nextLong(dur.toNanos());
                }
            }
        } else {
            Duration waitTime = handler.getWaitTime(totalTries);
            if (waitTime != null) {
                currentWaitNanos = waitTime.toNanos();
            }
        }
        this.reconnectWaiter = new CompletableFuture();
        long start = System.nanoTime();
        while (!(currentWaitNanos <= 0L || this.isDisconnectingOrClosed() || this.isConnected() || this.reconnectWaiter.isDone())) {
            try {
                this.reconnectWaiter.get(currentWaitNanos, TimeUnit.NANOSECONDS);
            }
            catch (Exception exception) {
                // empty catch block
            }
            long now = System.nanoTime();
            currentWaitNanos -= now - start;
            start = now;
        }
        this.reconnectWaiter.complete(Boolean.TRUE);
    }

    ByteBuffer enlargeBuffer(ByteBuffer buffer, int atLeast) {
        int current = buffer.capacity();
        int newSize = Math.max(current * 2, atLeast);
        ByteBuffer newBuffer = ByteBuffer.allocate(newSize);
        buffer.flip();
        newBuffer.put(buffer);
        return newBuffer;
    }

    NatsConnectionReader getReader() {
        return this.reader;
    }

    NatsConnectionWriter getWriter() {
        return this.writer;
    }

    Future<DataPort> getDataPortFuture() {
        return this.dataPortFuture;
    }

    boolean isDraining() {
        return this.draining.get() != null;
    }

    boolean isDrained() {
        CompletableFuture<Boolean> tracker = this.draining.get();
        try {
            if (tracker != null && tracker.getNow(false).booleanValue()) {
                return true;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return false;
    }

    @Override
    public CompletableFuture<Boolean> drain(Duration timeout) throws TimeoutException, InterruptedException {
        if (this.isClosing() || this.isClosed()) {
            throw new IllegalStateException("A connection can't be drained during close.");
        }
        this.statusLock.lock();
        try {
            if (this.isDraining()) {
                CompletableFuture<Boolean> completableFuture = this.draining.get();
                return completableFuture;
            }
            this.draining.set(new CompletableFuture());
        }
        finally {
            this.statusLock.unlock();
        }
        CompletableFuture<Boolean> tracker = this.draining.get();
        Instant start = Instant.now();
        HashSet<NatsSubscription> pureSubscribers = new HashSet<NatsSubscription>();
        pureSubscribers.addAll(this.subscribers.values());
        pureSubscribers.removeIf(s -> s.getDispatcher() != null);
        HashSet<NatsConsumer> consumers = new HashSet<NatsConsumer>();
        consumers.addAll(pureSubscribers);
        consumers.addAll(this.dispatchers.values());
        NatsDispatcher inboxer = this.inboxDispatcher.get();
        if (inboxer != null) {
            consumers.add(inboxer);
        }
        consumers.forEach(cons -> {
            cons.markDraining(tracker);
            cons.sendUnsubForDrain();
        });
        try {
            this.flush(timeout);
        }
        catch (Exception e) {
            this.close(false);
            throw e;
        }
        consumers.forEach(NatsConsumer::markUnsubedForDrain);
        this.executor.submit(() -> {
            try {
                Instant now = Instant.now();
                while (timeout == null || timeout.equals(Duration.ZERO) || Duration.between(start, now).compareTo(timeout) < 0) {
                    consumers.removeIf(NatsConsumer::isDrained);
                    if (consumers.size() == 0) break;
                    Thread.sleep(1L);
                    now = Instant.now();
                }
                this.blockPublishForDrain.set(true);
                if (timeout == null || timeout.equals(Duration.ZERO)) {
                    this.flush(Duration.ZERO);
                } else {
                    now = Instant.now();
                    Duration passed = Duration.between(start, now);
                    Duration newTimeout = timeout.minus(passed);
                    if (newTimeout.toNanos() > 0L) {
                        this.flush(newTimeout);
                    }
                }
                this.close(false);
                tracker.complete(consumers.size() == 0);
            }
            catch (InterruptedException | TimeoutException e) {
                this.processException(e);
            }
            finally {
                try {
                    this.close(false);
                }
                catch (InterruptedException e) {
                    this.processException(e);
                }
                tracker.complete(false);
            }
        });
        return tracker;
    }

    boolean isAuthenticationError(String err) {
        if (err == null) {
            return false;
        }
        return (err = err.toLowerCase()).startsWith("user authentication") || err.contains("authorization violation");
    }

    @Override
    public void flushBuffer() throws IOException {
        if (!this.isConnected()) {
            throw new IllegalStateException("Connection is not active.");
        }
        this.writer.flushBuffer();
    }

    void lenientFlushBuffer() {
        try {
            this.writer.flushBuffer();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Override
    public JetStream jetStream() throws IOException {
        this.ensureNotClosing();
        return new NatsJetStream(this, null);
    }

    @Override
    public JetStream jetStream(JetStreamOptions options) throws IOException {
        this.ensureNotClosing();
        return new NatsJetStream(this, options);
    }

    @Override
    public JetStreamManagement jetStreamManagement() throws IOException {
        this.ensureNotClosing();
        return new NatsJetStreamManagement(this, null);
    }

    @Override
    public JetStreamManagement jetStreamManagement(JetStreamOptions options) throws IOException {
        this.ensureNotClosing();
        return new NatsJetStreamManagement(this, options);
    }

    @Override
    public KeyValue keyValue(String bucketName) throws IOException {
        Validator.validateKvBucketNameRequired(bucketName);
        this.ensureNotClosing();
        return new NatsKeyValue(this, bucketName, null);
    }

    @Override
    public KeyValue keyValue(String bucketName, KeyValueOptions options) throws IOException {
        Validator.validateKvBucketNameRequired(bucketName);
        this.ensureNotClosing();
        return new NatsKeyValue(this, bucketName, options);
    }

    @Override
    public KeyValueManagement keyValueManagement() throws IOException {
        this.ensureNotClosing();
        return new NatsKeyValueManagement(this, null);
    }

    @Override
    public KeyValueManagement keyValueManagement(KeyValueOptions options) throws IOException {
        this.ensureNotClosing();
        return new NatsKeyValueManagement(this, options);
    }

    private void ensureNotClosing() throws IOException {
        if (this.isClosing() || this.isClosed()) {
            throw new IOException("A JetStream context can't be established during close.");
        }
    }
}

