#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/09/11
@file: user_select.py
@function: 支持用户选择或确认功能的对象
@modify:
"""
import json

from py.fusion_storage.common.context import disk_context_util

USER_SELECT_RESULT_TYPE_TABLE = "table"
TYPE = "type"
COL_LIST = "col_list"
DATA_LIST = "data_list"
SELECT_KEY = "select_key"
TITLE = "title"

USER_CONFIRM_WITH_GUIDE_URI = "user_confirm_with_guide_uri"
MESSAGE = "message"
GUIDE_URI = "guide_uri"


class AbstractUserSelectQueryResult(object):
    def __init__(self, result_type, title):
        self.type = result_type
        self.title = title

    def to_dict(self):
        raise NotImplementedError

    def get_json_result(self):
        return str(json.dumps(self.to_dict())).decode('unicode-escape')


def has_return_result(context):
    """判断脚本是否已经返回了结果

    从任务上下文中读取当前key，判断脚本是否返回了结果，以此确认脚本是否已经执行过一次了
    :param context: 上下文
    :return: True 已经返回了结果/False尚未返回结果
    """
    return bool(context.getEnv().get(disk_context_util.get_result_key(context)))


def set_json_result(context, result):
    """将脚本结果放入任务上下文

    :param context: 上下文
    :param result: AbstractUserSelectQueryResult实例
    """
    context.getEnv()[disk_context_util.get_result_key(context)] = \
        result.get_json_result()


def get_user_select_value(context):
    """获取用户选择的值

    :param context: 上下文
    :return: 用户选择的值
    """
    return context.getEnv().get(disk_context_util.get_select_key(context))


class UserConfirmWithGuide(AbstractUserSelectQueryResult):
    """需用用户确认的交互类型(带指导书)

    message: 信息内容
    title: 信息title
    guide_uri: 指导书Uri
    """
    def __init__(self, message, guide_uri, title):
        super(UserConfirmWithGuide, self).\
            __init__(USER_CONFIRM_WITH_GUIDE_URI, title)
        self.message = message
        self.guide_uri = guide_uri

    def to_dict(self):
        return {
            TYPE: self.type,
            TITLE: self.title,
            MESSAGE: self.message,
            GUIDE_URI: self.guide_uri
        }


class UserSelectTableQueryResult(AbstractUserSelectQueryResult):
    """需要用户从表格数据里进行选择的交互类型

    交互逻辑：
    1. 脚本查询，返回table result, 包含类型，标题，表头，表体，主键
    2. 用户选择后确认，下发选中的主键列表
    """
    def __init__(self, select_key, title):
        super(UserSelectTableQueryResult, self).\
            __init__(USER_SELECT_RESULT_TYPE_TABLE, title)
        self.col_list = []
        self.data_list = []
        self.select_key = select_key

    def set_col_list(self, col_list):
        self.col_list = col_list

    def set_data_list(self, data_list):
        self.data_list = data_list

    def add_data_2_list(self, data):
        self.data_list.append(data)

    def to_dict(self):
        return {
            TYPE: str(self.type),
            COL_LIST: self.col_list,
            DATA_LIST: self.data_list,
            SELECT_KEY: self.select_key,
            TITLE: self.title
        }

    def get_json_result(self):
        return str(json.dumps(self.to_dict())).decode('unicode-escape')


class AbstractUserSelectTableQueryData(object):
    """
    userSelect表格内容抽象类: 包含表头，表体，主键
    方法返回值要为基本类型嵌套
    """

    @staticmethod
    def get_select_key():
        """
        主键
        :return: string类型主键
        """
        raise NotImplementedError

    @staticmethod
    def get_show_col_list():
        """
        表头
        key: 对应属性的key
        name: 显示名称
        show: 是否默认展示
        :return: [ {"key": string, "name": string, "show": boolean}, ...]
        """
        raise NotImplementedError

    def get_show_data_dict(self):
        """
        表体
        :return: [ {key -> value}, ...]
        """
        raise NotImplementedError
