#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.


"""
@version: SmartKit V200R007C00
@time: 2021/05/25
@file: connection_service.py
@function: 连接api层，负责：
        1、连接的建立
        2、连接执行命令的封装，可支持重试，重连等操作
@modify:
"""
from py.common.adapter import java_adapter
from py.common.entity.exception import ConnectionException


class AbstractConnectionService(object):
    def __init__(self, dev_node, extra_success_code=None):
        self._dev_node = dev_node
        self._extra_success_code = extra_success_code

    def _get_connection(self):
        """
        获取连接，通常在java对应连接池中获取
        :return: 返回连接
        """
        raise NotImplementedError

    def _release_connection(self):
        """
        释放连接，在对应java对应连接池中释放连接
        """
        raise NotImplementedError

    def _is_success_response(self, response):
        """
        判断执行的结果是否是成功的
        :param response: 原始执行命令出的结果，根据不通的连接结果类型不同
        :return: True/False
        """
        raise NotImplementedError

    def _get_execute_result(self, response):
        """
        从原始执行结果中转换获取成需要的格式的结果
        :param response: 原始执行结果
        :return: 需要的格式的结果
        """
        raise NotImplementedError

    def _execute(self, retry_times, abnormal_judge_func, func, *params):
        """
        执行命令函数
        :param retry_times: 重试重连次数（重试一次、重连一次...）详见retry装饰器
        :param abnormal_judge_func: 判断结果是否正常的自定义函数
        :param func: 实际连接执行的函数
        :param params: 实例连接执行函数的参数
        :return: 执行的结果，由_get_execute_result函数决定
        """
        @retry(retry_times)
        def execute(_self):
            response = func(*params)
            if not self._is_success_response(response):
                raise ConnectionException("execute failed", response)
            result = self._get_execute_result(response)
            if abnormal_judge_func and abnormal_judge_func(result):
                raise Exception("abnormal result")
            return result
        return execute(self)


def retry(retry_times):
    """
    重试装饰器，支持指定重试次数，先重试一次再重连重试一次
    其中把遇到的java异常转成了python异常，便于脚本捕获
    :param retry_times: 总重试次数
    """

    def do_func(func):
        def raise_exception(exception):
            if isinstance(exception, Exception):
                raise exception
            raise Exception("java exec exception")

        def wrapper(self, *params):
            had_retry_times = 0
            while True:
                try:
                    return func(self, *params)
                except (Exception,
                        java_adapter.get_java_exception_class()) as e:
                    if had_retry_times == retry_times:
                        raise_exception(e)
                    # 奇数次时，进行一次连接的释放，执行的func中获取连接再重建。
                    if had_retry_times % 2 == 1:
                        self._release_connection()
                    had_retry_times += 1
        return wrapper
    return do_func
