#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/07
@file: rest_connection_service.py
@function:
@modify:
"""
import json

from py.common.adapter import java_adapter
from py.common.service import logger_factory
from py.common.service.connection import connection_service


class RestService(connection_service.AbstractConnectionService):

    def _get_connection(self):
        return java_adapter.get_rest_connection_manager_class().getRestConnection(
            self._dev_node)

    def _release_connection(self):
        try:
            java_adapter.get_rest_connection_manager_class().releaseConn(
                self._dev_node)
        except java_adapter.get_java_exception_class():
            logger_factory.create_logger(__file__).error('release conn fail.')

    def release_connect(self):
        self._release_connection()

    def _is_success_response(self, response):
        err_code = response.getContentMap().get("errorCode")
        return str(err_code) == "0" \
               or (err_code is None and json.loads(str(response.getContent())).get("result") == 0) \
               or (err_code is None and self._extra_success_code and json.loads(
            str(response.getContent())).get("result") == self._extra_success_code)

    def _get_execute_result(self, response):
        return json.loads(str(response.getContent()))

    def combine_uri(self, uri):
        return java_adapter.get_rest_util_class().getDstorageUrlHead(
            self._dev_node) + uri

    def exec_get(self, uri, params=None, retry_times=2,
                 abnormal_judge_func=None):
        params_str = json.dumps(params) if params else params
        return self._execute(
            retry_times,
            abnormal_judge_func,
            self._get_connection().execGet,
            self.combine_uri(uri), params_str
        )

    def exec_post(self, uri, params, retry_times=0, abnormal_judge_func=None):
        return self._execute(
            retry_times,
            abnormal_judge_func,
            self._get_connection().execPost,
            self.combine_uri(uri), json.dumps(params)
        )

    def exec_delete(self, uri, retry_times=0, abnormal_judge_func=None):
        return self._execute(
            retry_times,
            abnormal_judge_func,
            self._get_connection().execDelete,
            self.combine_uri(uri), None
        )

    @staticmethod
    def get_description(response):
        """
        result_code等于2时：
        {"result":2,"suggestion":"无。","errorCode":33556998,"description":"
        指定的硬盘所在的服务器硬件不支持硬盘定位灯。","i18n_description":"
        指定的硬盘所在的服务器硬件不支持硬盘定位灯。"}

        result_code等于1时：
        {"result":1,"detail":[{"ip":"xx.xx.xx.xx,"errorCode":1077949005,
        "description":"The system has an internal error.","suggestion":"
        请联系技术支持工程师。","i18n_description":"系统内部错误。",
        "rollBackStepName":null}]}

        有时又不存在i18n_description：
        {"result":2,"errorCode":31000000,"description":"系统内部错误。"}
        :param response: rest的response
        :return: description
        """

        def _get_description(msg_key):
            result = json.loads(str(response.getContent()))
            result_code = result.get("result")
            if result_code == 2:
                return result.get(msg_key)
            elif result_code == 1:
                return "; ".join(
                    [detail.get(msg_key) for detail in
                     result.get("detail", [])])
            return ""

        description = _get_description("i18n_description")
        if description:
            return description
        else:
            return _get_description("description")
