#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/07
@file: ssh_connection_service.py
@function:
@modify:
"""
import time
from py.common.adapter import java_adapter
from py.common.service.connection import connection_service

COMMON_SSH_ENDING = [
    ':/>', ':/', ']', '#', '$', '~>', ':~ #', 'minisystem>',
    'enter y to continue:', 'input y to continue:'
]


class AbstractCmd(object):

    def __init__(self, cmd):
        self.cmd = cmd

    def execute(self, connection):
        raise NotImplementedError


class Cmd(AbstractCmd):

    def execute(self, connection):
        return connection.execCmd(self.cmd)


class CustomEndingJudgeCmd(AbstractCmd):
    def __init__(self, cmd, ssh_judge):
        super(CustomEndingJudgeCmd, self).__init__(cmd)
        self.ssh_judge = ssh_judge

    def execute(self, connection):
        return connection.execCmd(self.cmd, self.ssh_judge)


class NoLogCmd(AbstractCmd):

    def execute(self, connection):
        return connection.execCmdNoLog(self.cmd)


class NoNeedReturnCmd(AbstractCmd):

    def execute(self, connection):
        return connection.execCmdWithOutReturnAutoEnter(self.cmd)


class KeepAliveCmd(AbstractCmd):

    def __init__(self, cmd, time_out, keep_interval=30, is_need_log=True):
        """
        :param cmd: 命令 str
        :param time_out: 超时时间 单位秒 int
        :param keep_interval: 多少秒激活一次 int
        :param is_need_log: 是否打印日志 boolean
        """
        super(KeepAliveCmd, self).__init__(cmd)
        self.time_out = time_out
        self.keep_interval = keep_interval
        self.is_need_log = is_need_log

    def execute(self, connection):
        return connection.execCmdWithKeepAlive(
            self.cmd, self.keep_interval, self.time_out, self.is_need_log)


def _is_login_in_dsware_user(connection):
    who_result = connection.execCmd("whoami")
    for line in who_result.splitlines():
        if line.strip() == "dsware":
            return True
    return False


def _is_login_in_cli_system(su_result):
    for line in su_result.splitlines():
        if line.strip().endswith("cliportal>"):
            return True
    return False


class SshService(connection_service.AbstractConnectionService):
    def __init__(self, dev_node):
        super(SshService, self).__init__(dev_node)
        self.is_mini_system = dev_node.isMiniSystem()
        self.is_sandbox_open = dev_node.isSandboxOpen()

    def _get_connection(self):
        return java_adapter.get_ssh_connection_manager_class().getSshConnection(
            self._dev_node)

    def _release_connection(self):
        java_adapter.get_ssh_connection_manager_class().releaseConnection(
            self._dev_node)

    def _is_success_response(self, response):
        return True

    def _get_execute_result(self, response):
        return response

    def _exec_cmd(self, cmd):
        """
        执行命令
        :param cmd: Cmd对象
        :return: result str
        """
        if not isinstance(cmd, AbstractCmd):
            raise ValueError("cmd should be Cmd object")
        return cmd.execute(self._get_connection())

    def exec_cmd(self, cmd, retry_times=0, abnormal_judge_func=None):
        """
        执行命令
        :param cmd: 命令 Cmd对象
        :param retry_times: 重试次数
        :param abnormal_judge_func: 异常判断方法，入参为_get_execute_result结果
        :return: 结果 str
        """
        return self._execute(
            retry_times,
            abnormal_judge_func,
            self._exec_cmd,
            cmd
        )

    def exec_cmd_handle_res(self, cmd, retry_times=0, abnormal_judge_func=None, handle_res_func=None):
        """
        执行命令
        :param handle_res_func: 回显自定义处理方法
        :param cmd: 命令 Cmd对象
        :param retry_times: 重试次数
        :param abnormal_judge_func: 异常判断方法，入参为_get_execute_result结果
        :return: 结果 str
        """
        res = self._execute(
            retry_times,
            abnormal_judge_func,
            self._exec_cmd,
            cmd
        )
        if handle_res_func:
            return handle_res_func(res)
        return res

    def exec_dsware_cmd(self, cmd, retry_times=0, abnormal_judge_func=None):
        """
        执行dsware命令
        :param cmd: Cmd对象，可以是一个或list，代表一个或多个命令
        :param retry_times: 重试次数
        :param abnormal_judge_func: 异常判断方法，入参为_get_execute_result结果
        :return: 最后一个命令执行的结果 str
        """

        def _exec_dsware_cmd(_cmd):
            connection = self._get_connection()
            connection.execCmd("su - dsware -s /bin/bash")
            if not _is_login_in_dsware_user(connection):
                self._release_connection()
                raise Exception("su dsware failed")
            if isinstance(_cmd, list):
                for cmd in _cmd:
                    result = self._exec_cmd(cmd)
                    # 睡眠一秒防止执行太快未生效
                    time.sleep(1)
            else:
                result = self._exec_cmd(_cmd)
            if _is_login_in_dsware_user(connection):
                connection.execCmd("exit")
                if _is_login_in_dsware_user(connection):
                    self._release_connection()
            return result

        return self._execute(
            retry_times,
            abnormal_judge_func,
            _exec_dsware_cmd,
            cmd
        )

    def exec_dsware_cmd_with_sandbox_open(self, cmd, retry_times=0, abnormal_judge_func=None):
        """
        minisystem执行dsware命令
        :param cmd: Cmd对象，可以是一个或list，代表一个或多个命令
        :param retry_times: 重试次数
        :param abnormal_judge_func: 异常判断方法，入参为_get_execute_result结果
        :return: 最后一个命令执行的结果 str
        """

        def _exec_dsware_cmd(_cmd):
            connection = self._get_connection()
            root_switch_to_cli = False
            # 沙箱开启状态下root直连，需要先切换到cli模式再进入minisystem
            if not self.is_mini_system:
                su_result = connection.execCmd("su - fsadmin")
                if not _is_login_in_cli_system(su_result):
                    self._release_connection()
                    raise Exception("su CLI Portal failed")

                root_switch_to_cli = True
            if isinstance(_cmd, list):
                for cmd in _cmd:
                    result = self._exec_cmd(cmd)
                    # 睡眠一秒防止执行太快未生效
                    time.sleep(1)
            else:
                result = self._exec_cmd(_cmd)
            # 释放从root切换到minisystem的连接
            if root_switch_to_cli:
                self._release_connection()
            return result

        return self._execute(
            retry_times,
            abnormal_judge_func,
            _exec_dsware_cmd,
            cmd
        )
