#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/23
@file: frame_work_device_filed_translator.py
@function: 提供查询设备部件信息后转换成java对应的设备信息对象实例
@modify:
"""


class JavaDeviceFieldTranslator(object):
    """
    将查询到设备信息的某字段转换成框架的object
    """

    def set_field(self, framework_object, origin_value):
        raise NotImplementedError


class EnumFieldTranslator(JavaDeviceFieldTranslator):
    """
    转换为枚举字段，例如Type XXX
    """

    def __init__(self, set_method, enum_class, value_map):
        """
        :param set_method:  设置该属性set方法
        :param enum_class: 若属性值为枚举时需要：定义枚举的class
        :param value_map: 查询的原值与预期值的映射
        """
        self.set_method = set_method
        self.enum_class = enum_class
        self.value_map = value_map

    def set_field(self, framework_object, origin_value):
        trans_value = self.value_map.get(origin_value)
        if trans_value:
            trans_object = self.enum_class.valueOf(trans_value)
            getattr(framework_object, self.set_method)(trans_object)


class ValueFieldTranslator(JavaDeviceFieldTranslator):
    """
    直接转换为值字段, 例如String XXX
    """

    def __init__(self, set_method):
        """
        :param set_method:  设置该属性set方法
        """
        self.set_method = set_method

    def set_field(self, framework_object, origin_value):
        if origin_value is None:
            return
        getattr(framework_object, self.set_method)(origin_value)


class StringValueFieldTranslator(ValueFieldTranslator):
    """
    转为字符串。例如 String XXX
    """
    @staticmethod
    def _trans_2_str(origin_value):
        try:
            return str(origin_value)
        except ValueError:
            return ""

    def set_field(self, framework_object, origin_value):
        getattr(framework_object, self.set_method)(self._trans_2_str(
            origin_value))


class IntValueFieldTranslator(ValueFieldTranslator):
    """
    转为为整型字段。例如 1
    """
    def set_field(self, framework_object, origin_value):
        getattr(framework_object, self.set_method)(self._trans_2_int(
            origin_value))

    @staticmethod
    def _trans_2_int(origin_value):
        try:
            return int(origin_value)
        except ValueError:
            return -1


class ListFieldTranslator(JavaDeviceFieldTranslator):
    """
    转换为list字段。例如List<Integer>
    """

    def __init__(self, set_method, list_key=None):
        """
        :param set_method: 设置该属性set方法
        :param list_key:  若原属性值list中还包含map，则可设该key进行获取
        """
        self.set_method = set_method
        self.list_key = list_key

    def set_field(self, framework_object, origin_value):
        if self.list_key:
            values = [value.get(self.list_key) for value in origin_value]
            getattr(framework_object, self.set_method)(values)
        else:
            getattr(framework_object, self.set_method)(origin_value)
