#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/15
@file: record_nvme_ssd_type.py
@function: 若是nvme，查询记录是nvme盘还是nvme卡，后续更换逻辑有差异
@modify:
"""
import json

import re
from py.fusion_cube.common.context import disk_context_util
from py.fusion_cube.common.service.os_util import disk_info_util
from py.common.service.connection.ssh_connection_service import SshService
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.fusion_cube.common.constant import ClusterRestUri
from py.common.entity.exception import BusinessException
from py.common.service import resource_service
from py.common.service import logger_factory


class NvmeSsdForm(object):
    CARD = "CARD"
    DISK = "DISK"
    # 硬盘不在位时为UN_KNOWN
    UN_KNOWN = "UN_KNOWN"


FORM_2_FORM_NUMBERS = {
    NvmeSsdForm.DISK: ("1", "2", "3"),
    NvmeSsdForm.CARD: ("6", "7")
}

HUAWEI_NVME_FORM_NUMBER_REG = re.compile(r"HUAWEI,HWE\d(\d)")
NVME_SSD_FORM = "NVME_SSD_FORM"


def record(context):
    if disk_context_util.is_nvme_disk(context):
        form = _get_nvme_ssd_form(context)
        logger_factory.create_logger(__file__).info("nvme form: {}".format(
            form))
        context.getEnv()[NVME_SSD_FORM] = form


def _get_nvme_ssd_form(context):
    vmr = _get_disk_vmr(context)
    if not vmr:
        if not _get_disk_slot(context):
            # 如果硬盘不在位了，show_disk.sh将查不到对应得盘信息，也就无法判断是卡
            # 还是盘，这种后面也不需要逻辑下电，故不报错。通过queryAllDisk查看槽位
            # 号是否为空判断
            return NvmeSsdForm.UN_KNOWN
        raise BusinessException(
            resource_service.get_msg("get.nvme.form.failed"))
    form_number = _get_form_number(vmr)
    if not form_number:
        # 其他盘，已知有INTEL NVME,都是盘
        return NvmeSsdForm.DISK
    for form, form_numbers in FORM_2_FORM_NUMBERS.items():
        if form_number in form_numbers:
            return form
    raise BusinessException(resource_service.get_msg("get.nvme.form.failed"))


def _get_disk_slot(context):
    rest_service = CubeRestService(context.getCluster())
    res_dict = rest_service.exec_get(ClusterRestUri.QUERY_ALL_INFO)
    for disk_infos in res_dict.get("disks", {}).values():
        for info in disk_infos:
            if disk_context_util.get_disk_sn(context) == info.get("devEsn"):
                return info.get("devSlot")
            return None


def _get_disk_vmr(context):
    service = SshService(context.getNode())
    infos = disk_info_util.get_show_disk_infos(service)
    disk_sn = disk_context_util.get_disk_sn(context)
    for info in infos:
        if disk_sn == info.get("Esn"):
            return info.get("VMR")
    return ""


def _get_form_number(vmr):
    match = HUAWEI_NVME_FORM_NUMBER_REG.search(vmr)
    if match:
        return match.groups()[0]
    return None


def get_nvme_ssd_form(context):
    return context.getEnv().get(NVME_SSD_FORM)
