#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2022. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/08/13
@file: init_meta_disk_info_service.py
@function: 元数据盘查询实现
@modify:
"""
from py.common.adapter import java_adapter
from py.common.service.java_device_filed_translator import \
    ValueFieldTranslator, EnumFieldTranslator, IntValueFieldTranslator
from py.fusion_cube.common.constant import ClusterRestUri, Module
from py.fusion_cube.common.service.cluster import query_disk_info_service
from py.fusion_cube.common.service.cluster.cluster_info_util import \
    is_ocean_stor_pacific
from py.fusion_cube.common.service.disk_init_factory.disk_info_query_trans_dict import \
    MEDIA_TYPE_TRANS_DICT, DEV_ROLE_TRANS_DICT
from py.fusion_cube.common.service.disk_init_factory.disk_info_service import \
    DiskInitService

DISK_EXIST = 0
DISK_NOT_EXIST = 1
SYS_DISK = 'sys_disk'
M2_DISK = 'm2_disk'


class InitMetaDiskInfoService(DiskInitService):
    """
    当前以控制集群元数据盘为准
    即，复制集群独立使用的元数据盘无法查询
    """
    @staticmethod
    def get_query_all_disk_trans_dict():
        return {
            "devEsn": ValueFieldTranslator("setEsn"),
            "devSlot": IntValueFieldTranslator("setSlotNo"),
            "devType": EnumFieldTranslator("setMediaType",
                                           java_adapter.get_media_type_class(),
                                           MEDIA_TYPE_TRANS_DICT),
            "devRole": EnumFieldTranslator("setMediaRole",
                                           java_adapter.get_media_role_class(),
                                           DEV_ROLE_TRANS_DICT),
            "devTotalCapacity": ValueFieldTranslator("setCapacity"),
            "devStatus": ValueFieldTranslator("setStatus"),
            "sectorSize": ValueFieldTranslator("setSectorSize"),
            "logicSectorSize": ValueFieldTranslator("setLogicSectorSize"),
        }

    @staticmethod
    def get_query_manage_cluster_trans_dict():
        return {
            "diskSn": ValueFieldTranslator("setEsn"),
            "diskSlot": IntValueFieldTranslator("setSlotNo"),
            "diskType": EnumFieldTranslator("setMediaType",
                                            java_adapter.get_media_type_class(),
                                            MEDIA_TYPE_TRANS_DICT),
            "diskStatus": ValueFieldTranslator("setStatus"),
        }

    def __init__(self, context):
        super(InitMetaDiskInfoService, self).__init__(context)
        # 所有在位盘信息字典, key为管理IP, value为盘列表
        # 1. 不包含不在位的盘 2. 复制集群独立使用的元数据盘无法识别为元数据盘角色(8.1后会被识别，将被过滤掉)
        self._all_in_position_disk_dict = {}

    def query_cur_module_disks(self):
        self._all_in_position_disk_dict = self._rest_service \
            .exec_get(ClusterRestUri.QUERY_ALL_INFO).get("disks", {})
        return self.filter_pilot_disks(self._query_ctrl_cluster_all_meta_disks())

    def query_node_disks(self, mgr_ip):
        self._all_in_position_disk_dict = self._rest_service \
            .exec_get(ClusterRestUri.QUERY_ALL_INFO).get("disks", {})
        disk_info_list = self._all_in_position_disk_dict.get(mgr_ip, [])
        disks = []
        for disk_info in disk_info_list:
            disks.append(self._translate_2_java_object(
                disk_info, self.get_query_all_disk_trans_dict(),
                mgr_ip, DISK_EXIST))
        return self.filter_pilot_disks(disks)

    def filter_pilot_disks(self, disks):
        """
        元数据盘需要排除数据导航盘
        :param disks: 元数据盘列表（包含数据导航盘）
        :return: 排除后的元数据盘列表
        """
        if not is_ocean_stor_pacific(self._context):
            return disks
        pilot_esn_list = query_disk_info_service.query_pilot_disks_esn_list(self._context)
        return [disk for disk in disks if disk.getEsn() not in pilot_esn_list]

    def _query_ctrl_cluster_all_meta_disks(self):
        in_position_disks = self._get_ctrl_cluster_in_position_disks()
        conf_disks = self._query_ctrl_cluster_conf_disks()
        return self._parse_ctrl_cluster_all_meta_disks(
            in_position_disks, conf_disks)

    def _parse_ctrl_cluster_all_meta_disks(self, in_position_disks, conf_disks):
        """
        从在位zk_disk盘和配置元数据盘信息中解析出当前设备的所有元数据盘
        in_position_disks中的zk_disk盘可能包含复制集群独立使用的盘(8.1之后)，以conf_disk的范围为准
        :param in_position_disks: 在位的zk_disk盘列表
        :param conf_disks: 控制集群元数据盘配置信息
        :return: 当前设备元数据盘列表
        """
        disks = []
        # 1. 先从in_position_disks中获取在conf_disk范围中的盘
        for in_position_disk in in_position_disks:
            if self._is_contains_disk_esn(conf_disks,
                                          in_position_disk.getEsn()):
                disks.append(in_position_disk)
        # 2. 再从conf_disks中获取不在in_position_disks中的盘（即不在位的控制集群元数据盘）
        for conf_disk in conf_disks:
            if not self._is_contains_disk_esn(
                    in_position_disks, conf_disk.getEsn()):
                disks.append(conf_disk)
        return disks

    @staticmethod
    def _is_contains_disk_esn(disk_list, disk_esn):
        for disk in disk_list:
            if disk.getEsn() == disk_esn:
                return True
        return False

    def _get_ctrl_cluster_in_position_disks(self):
        disks = []
        for mgr_ip, disk_infos in self._all_in_position_disk_dict.items():
            for disk_info in disk_infos:
                if disk_info.get('devType') == M2_DISK:
                    continue
                if disk_info.get('devRole').upper() in \
                        Module.MODULE_2_disk_type.get(Module.META_DISK):
                    disks.append(self._translate_2_java_object(
                        disk_info, self.get_query_all_disk_trans_dict(),
                        mgr_ip, DISK_EXIST))
        return disks

    def _query_ctrl_cluster_conf_disks(self):
        """
        从queryManageCluster接口查询集群配置的元数据盘的信息
        此接口返回的值的盘信息不全，但是不在位的盘也有记录
        :return: 对应Java盘对象列表, 信息统一处理为不在位
        """
        disks = []
        res_dict = self._rest_service.exec_get(
            ClusterRestUri.QUERY_MANAGE_CLUSTER_INFO)
        for node in res_dict.get("nodeInfo", []):
            mgr_ip = node.get('nodeMgrIp')
            disk_info = node.get('diskInfo', {})
            disk_type = disk_info.get('diskType')
            if disk_type != SYS_DISK and disk_type != M2_DISK:
                disks.append(self._translate_2_java_object(
                    disk_info, self.get_query_manage_cluster_trans_dict(),
                    mgr_ip, DISK_NOT_EXIST,))
        return disks
