#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/11/20
@file: init_sys_disk_info_service.py
@function: 系统盘查询服务
@modify:
"""
from py.common.adapter import java_adapter
from py.common.service import logger_factory
from py.common.service.connection.redfish_connection_service import RedfishService
from py.common.service.connection.ssh_connection_service import SshService
from py.common.service.java_device_filed_translator import ValueFieldTranslator, EnumFieldTranslator
from py.fusion_cube.common.constant import ClusterRestUri, RaidCardType
from py.fusion_cube.common.context import common_context_util
from py.fusion_cube.common.service.disk_init_factory.disk_info_query_trans_dict import MEDIA_TYPE_TRANS_DICT
from py.fusion_cube.common.service.disk_init_factory.disk_info_service import DiskInitService
from py.fusion_cube.common.service.hardware_operation_factory.redfish_util import RedfishUtil
from py.fusion_cube.common.service.os_util import raid_card_util
from py.fusion_cube.common.service.os_util.raid_1880_util import RaidCard1880
from py.fusion_cube.common.service.os_util.raid_card_util import RaidCard3508, RaidCard3152, is_1880_supported

SYSTEM_DISK_ROLE = "SYSTEM_DISK"


class InitSysDiskInfoService(DiskInitService):
    @staticmethod
    def get_scan_system_media_trans_dict():
        return {
            "phyDevEsn": ValueFieldTranslator("setEsn"),
            "phySlotId": ValueFieldTranslator("setSlotNo"),
            "mediaType": EnumFieldTranslator("setMediaType",
                                             java_adapter.get_media_type_class(),
                                             MEDIA_TYPE_TRANS_DICT),
            "mediaCapacityForByte": ValueFieldTranslator("setCapacity"),
            "mediaCapacity": ValueFieldTranslator("setCapacityStr"),
            "health": ValueFieldTranslator("setStatus"),
            "devName": ValueFieldTranslator("setDiskLetter"),
        }

    def query_node_disks(self, mgr_ip):
        # 用于软RAID更前后盘信息对比, 返回值不包含系统盘
        return [disk for disk in self._all_disks
                if disk.getBelongNodeIp() == mgr_ip]

    def query_cur_module_disks(self):
        current_disks = list()
        # 查询两大洋节点并通过rest查询disk信息
        current_ip_list = common_context_util.get_proprietary_hardware_storage_ip_list(self._context)
        if current_ip_list and len(current_ip_list) > 0:
            current_disks.extend(self._query_sys_disk_by_ip_list(current_ip_list))
        # 查询鉴权信息并过滤掉两大洋节点
        current_ip_list, node_auth_infos = self._query_auth_node_list(current_ip_list)
        # 已经鉴权的节点查询硬RAID类型，是硬RAID，通过SSH查询disk信息，其他的使用REST信息保存
        current_disks.extend(self._query_raid_disk_info(node_auth_infos))
        # 将当前查询的硬盘信息和原有的ModuleData数据进行对比，在当前信息中不存在则加入当前信息
        origin_disks = self._context.getAllModuleData()
        for origin_disk in origin_disks:
            if origin_disk.getBelongNodeIp() not in current_ip_list:
                current_disks.append(origin_disk)
        return current_disks

    def _query_auth_node_list(self, ip_list):
        node_list = common_context_util.get_task_auth_nodes_list(self._context)
        node_auth_infos = dict()
        if node_list and len(node_list) > 0:
            for value in node_list:
                if InitSysDiskInfoService.need_query_sys_disk(ip_list, value):
                    ip_list.append(value.getClusterNode().getManagementIp())
                    node_auth_infos[value.getClusterNode().getManagementIp()] = {
                        "bmcNode": value.getBmcNode(),
                        "osNode": value.getOsNode()
                    }
        else:
            mgr_ip = common_context_util.get_current_node_management_ip(self._context)
            if mgr_ip not in ip_list and common_context_util.is_node_has_storage_role(self._context.getClusterNode()):
                ip_list.append(mgr_ip)
                node_auth_infos[mgr_ip] = {
                    "bmcNode": self._context.getBmcNode(),
                    "osNode": self._context.getNode()
                }
        return ip_list, node_auth_infos

    @staticmethod
    def need_query_sys_disk(ip_list, dev_node):
        return dev_node.isBothAuthSuccess() and dev_node.getClusterNode().getManagementIp() not in ip_list \
               and common_context_util.is_node_has_storage_role(dev_node.getClusterNode())

    def query_node_sys_disks(self, mgr_ip):
        return [disk for disk in self.query_cur_module_disks()
                if disk.getBelongNodeIp() == mgr_ip]

    def _query_raid_disk_info(self, auth_node_dict):
        if not auth_node_dict:
            return list()
        logger = logger_factory.create_logger(__file__)
        res = self._rest_service.exec_post(
            ClusterRestUri.SCAN_SYSTEM_MEDIA,
            params={
                "nodeIpList": auth_node_dict.keys()
            })
        disks = list()
        for mgr_ip, disk_info_list in res.get("servers", {}).items():
            for disk_info in disk_info_list:
                # 获取节点硬RAID型号
                auth_node = auth_node_dict.get(mgr_ip)
                redfish_service = RedfishService(auth_node.get("bmcNode"))
                raid_card_type = RedfishUtil(redfish_service).get_hard_raid_card_type()
                # 没有硬RAID时，直接使用restful接口拼接硬盘信息
                if not raid_card_type:
                    disks.append(
                        self._get_base_disk_from_disk_res(disk_info, mgr_ip))
                    continue
                # 硬RAID节点，通过SSH查询硬盘信息
                ret_disks = self._query_hard_raid_disk_info(raid_card_type, disk_info.get("devName"),
                                                            auth_node.get("osNode"), logger)
                for disk in ret_disks:
                    disks.append(self._get_base_disk_from_disk_res(disk, mgr_ip))
        return disks

    @staticmethod
    def _query_hard_raid_disk_info(raid_card_type, disk_letter, os_node, logger):
        ssh_service = SshService(os_node)
        if RaidCardType.SAS3152 in raid_card_type:
            return RaidCard3152(ssh_service, disk_letter, logger).get_sys_disk_info()
        elif raid_card_util.is_3508_supported(raid_card_type):
            return RaidCard3508(ssh_service, disk_letter, logger).get_sys_disk_info()
        elif is_1880_supported(raid_card_type):
            return RaidCard1880(ssh_service, disk_letter, logger).get_sys_disk_info()
        return []

    def _query_sys_disk_by_ip_list(self, ip_list):
        res = self._rest_service.exec_post(
            ClusterRestUri.SCAN_SYSTEM_MEDIA,
            params={
                "nodeIpList": ip_list
            })
        disks = list()
        for mgr_ip, disk_info_list in res.get("data", {}).get("servers", {}).items():
            for disk_info in disk_info_list:
                disks.append(
                    self._get_base_disk_from_disk_res(disk_info, mgr_ip))
        return disks

    def _get_base_disk_from_disk_res(self, disk_res, ip_info):
        disk = self._translate_2_java_object(disk_res, self.get_scan_system_media_trans_dict(), ip_info)
        trans_object = java_adapter.get_media_role_class().valueOf(
            SYSTEM_DISK_ROLE)
        getattr(disk, "setMediaRole")(trans_object)
        return disk
