#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/16
@file: disk_out_storage_pool.py
@function:
@modify:
"""
from py.fusion_cube.common.constant import ClusterRestUri
from py.fusion_cube.common.service.os_util import mdc_info_util, disk_info_util
from py.common.service.connection.ssh_connection_service import Cmd, CustomEndingJudgeCmd
from py.fusion_cube.common.record import record_cache_disk_match_main_disk
from py.common.service.connection.ssh_connection_service import SshService
from py.common.entity.exception import BusinessException
from py.common.service import resource_service, logger_factory
from py.fusion_cube.common.context import disk_context_util, common_context_util
from py.fusion_cube.common.service.cluster.cluster_info_util \
    import is_ocean_stor_pacific

SUCCESS_CODE = 'success'
FAILED_CODE = 'failed'
HIGH_USAGE_CODE = 'ret[50090029]'


def get_storage_pool_ids(context, service):
    # 根据硬盘池id，获取硬盘池对应的存储池id
    disk_pool_ids = disk_context_util.get_disk_belong_pool_ids(context)
    return get_storage_pool_ids_by_disk_pool_ids(disk_pool_ids, service, context)


def get_storage_pool_ids_by_disk_pool_ids(pool_ids, service, context):
    # 根据版本型号判断，如果是OceanStor Pacific，pool_ids是硬盘池id，其他则是存储池id
    if not is_ocean_stor_pacific(context):
        return pool_ids
    storage_pool_ids = []
    # 查询所有存储池
    # 获取获取每个存储池下的硬盘池
    # 解析出硬盘池id和存储id对应关系,id都是集群唯一
    storage_pools_dict = service.exec_get(
        ClusterRestUri.QUERY_STORAGE_POOL_INFO)
    for pool_info in storage_pools_dict.get("storagePools", []):
        pool_id = pool_info.get("poolId")
        if pool_id is None:
            continue
        disk_pools_dict = service.exec_get(ClusterRestUri.QUERY_DISK_POOL,
                                           params={"storagePoolId": pool_id})
        for disk_pools_info in disk_pools_dict.get("diskPools", []):
            disk_pool_id = disk_pools_info.get("poolId", "")
            if disk_pool_id in pool_ids:
                storage_pool_ids.append(pool_id)
    return set(storage_pool_ids)


def get_disk_pool_id_2_storage_pool_id(context, pool_ids, service):
    storage_pool_id_2_disk_pool_id = {}
    storage_pools_dict = service.exec_get(
        ClusterRestUri.QUERY_STORAGE_POOL_INFO)
    for pool_info in storage_pools_dict.get("storagePools", []):
        storage_pool_id = pool_info.get("poolId")
        if storage_pool_id is None:
            continue
        disk_pools_dict = service.exec_get(ClusterRestUri.QUERY_DISK_POOL,
                                           params={
                                               "storagePoolId": storage_pool_id})
        for disk_pools_info in disk_pools_dict.get("diskPools", []):
            disk_pool_id = disk_pools_info.get("poolId", "")
            if disk_pool_id in pool_ids:
                storage_pool_id_2_disk_pool_id[disk_pool_id] = storage_pool_id
    return storage_pool_id_2_disk_pool_id


def had_disk_outed_pool(connection_service, pool_ids, disk_esn, disk_slot):
    for pool_id in pool_ids:
        disk_infos = _get_pool_infos(connection_service, pool_id)
        disk_esns = [info.get("diskSn") for info in disk_infos]
        disk_slots = [info.get("diskSlot") for info in disk_infos]
        if disk_esn in disk_esns and disk_slot in disk_slots:
            return False
    return True


def _get_pool_infos(service, pool_id):
    res_dict = service.exec_get(
        ClusterRestUri.QUERY_DISK_INFO,
        params={"poolId": pool_id})
    disk_infos = list()
    for node_info in res_dict.get("nodeInfo", []):
        disk_infos.extend(node_info.get("mediaInfo", []))
    return disk_infos


def out_disk_to_pool(ssh_service, disk_sns, pool_id, context):
    """
    将盘踢出存储池
    执行命令 dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1807 [pool_id] 1 [disk_nid] down盘
            dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1817 [pool_id] 0 [disk_nid] 踢盘
    如果踢盘失败 dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1807 [pool_id] 0 [disk_nid] 取消down盘
    1. 此命令mdc_id 与 mdc_ip 不需要与盘节点匹配，下发后会自动匹配
    2. 10530为mdc_port，为固定值
    :param ssh_service: ssh_service
    :param disk_sns: 踢盘的sn list
    :param pool_id: 存储池id
    :param context: 上下文
    """
    if not disk_sns:
        return

    # 查询mdc_id, mdc_ip mdc_cmd.sh 181
    infos = mdc_info_util.get_mdc_cmd_181_infos(ssh_service)
    mdc_nid = None
    mdc_ip = None
    for info in infos:
        if int(info.get('STATUS')) == 0:
            mdc_nid = info.get('MDC NID')
            mdc_ip = info.get('STORAGE IP_0')
            break
    if not mdc_nid or not mdc_ip:
        raise BusinessException(err_msg=resource_service.get_msg(
            "mdc.id.query.failed"))

    # 查询disk_nid, mdc_cmd.sh 101
    disk_infos = disk_info_util.get_mdc_cmd_101_infos(ssh_service, pool_id)
    disk_sn_2_nid = dict()
    for disk_info in disk_infos:
        if disk_info.get("OSD ESN") in disk_sns:
            disk_sn_2_nid[disk_info.get("OSD ESN")] = disk_info.get("OSD NID")

    # 设置硬盘down，然后踢出
    option_cmd = 'dsware_insight 0 {0} {1} 10530 8 {{}} {2} {{}} {{}}'.format(mdc_nid, mdc_ip, pool_id)
    for disk_sn, disk_nid in disk_sn_2_nid.items():
        _try_out_disk(option_cmd, disk_nid, disk_sn, ssh_service, context)


def _try_out_disk(option_cmd, disk_nid, disk_sn, ssh_service, context):
    """
     执行命令 dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1807 [pool_id] 1 [disk_nid] down盘
            dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1817 [pool_id] 0 [disk_nid] 踢盘
    如果踢盘失败 dsware_insight 0 [mdc_id] [mdc_ip] 10530 8 1807 [pool_id] 0 [disk_nid] 取消down盘
    :param option_cmd: 执行命令
    :param disk_nid: 盘id
    :param disk_sn: 盘序列号
    :param ssh_service: shell连接
    :param context: 上下文
    """
    set_disk_status_code = 1807
    set_disk_down = 1
    clear_disk_down = 0
    out_disk_code = 1817
    remaining_quantity_limit = 0

    set_disk_down_cmd = option_cmd.format(set_disk_status_code, set_disk_down, disk_nid)
    _set_disk_down(disk_sn, set_disk_down_cmd, ssh_service, context)

    out_disk_cmd = option_cmd.format(out_disk_code, remaining_quantity_limit, disk_nid)
    error_code = _out_disk(ssh_service, out_disk_cmd)
    if not error_code:
        return
    # 踢盘失败，清除硬盘down状态
    clear_disk_down_cmd = option_cmd.format(set_disk_status_code, clear_disk_down, disk_nid)
    _clear_disk_down(disk_sn, error_code, clear_disk_down_cmd, ssh_service)


def _set_disk_down(disk_sn, set_disk_down_cmd, ssh_service, context):
    if not _set_disk_status(ssh_service, set_disk_down_cmd):
        common_context_util.set_cur_task_suggestion(context, resource_service.get_msg("set.disk.down.error").format(
            set_disk_down_cmd))
        raise BusinessException(err_msg=resource_service.get_msg("set.disk.down.error").format(disk_sn))


def _clear_disk_down(disk_sn, error_code, option_cmd, ssh_service):
    err_msg = resource_service.get_msg(error_code).format(disk_sn)
    if _set_disk_status(ssh_service, option_cmd):
        raise BusinessException(err_msg=err_msg)
    raise BusinessException(err_msg=err_msg + resource_service.get_msg("clear.disk.down.error"))


def _set_disk_status(ssh_service, down_disk_cmd):
    execute_res = ssh_service.exec_cmd(Cmd(down_disk_cmd))
    return _handle_result(execute_res, ssh_service)


def _handle_result(execute_res, ssh_service):
    if '[Yes/No]' in execute_res:
        execute_res = ssh_service.exec_cmd(Cmd('Yes'))
    for line in execute_res.splitlines():
        if 'result' in line and SUCCESS_CODE not in line:
            return False
    return True


def _out_disk(ssh_service, out_disk_cmd):
    end_strs = ['please input y to continue:', 'Yes/No? ', ':/>', ':/', ']', '#', '$', '~>', ':~ #']
    execute_res = ssh_service.exec_cmd(CustomEndingJudgeCmd(out_disk_cmd, end_strs))
    if 'please input y to continue' in execute_res:
        execute_res = ssh_service.exec_cmd(Cmd('y'))
    if SUCCESS_CODE in execute_res:
        return ''
    if FAILED_CODE in execute_res and HIGH_USAGE_CODE in execute_res:
        return "out.disk.high.usage.error"
    return 'out.disk.error'


def get_out_pool_main_disk(ssh_service, main_disks):
    """
    获取未加入存储池的主存盘名单
    :param ssh_service: ssh_service
    :param main_disks:
    :return:
    """
    infos = disk_info_util.get_mdc_cmd_101_infos(ssh_service)
    all_pool_disk_sns = [info.get("OSD ESN") for info in infos]
    not_add_pool_main_disks = list()
    for disk in main_disks:
        if disk.getEsn() not in all_pool_disk_sns:
            not_add_pool_main_disks.append(disk)
    return not_add_pool_main_disks


def get_out_pool_cache_match_main_disk(context):
    match_disks = record_cache_disk_match_main_disk.get_recorded_cache_disk_match_main_disks(
        context)
    return get_out_pool_main_disk(SshService(context.getNode()), match_disks)


def is_disks_add_in_pool(service, disk_esn_list):
    """
    用于判断盘是否全都成功加入了存储池（进程已经被拉起）
    1. 新主存盘接入后判断是否自动加盘成功 2. 缓存盘对应主存盘加盘后判断是否加成功
    :param service: 连接
    :param disk_esn_list: 硬盘列表
    :return: true/false
    """
    # nid -> esn的映射（传入的盘）
    infos = disk_info_util.get_mdc_cmd_101_infos(service)
    disk_osd_id_2_sn = dict()
    for info in infos:
        osd_esn = info.get("OSD ESN")
        if osd_esn in disk_esn_list:
            disk_osd_id_2_sn[info.get("OSD NID")] = osd_esn

    # 存在盘在101中没有查询到
    if len(disk_osd_id_2_sn) != len(disk_esn_list):
        logger_factory.create_logger(__file__).info('exist disk which not in mdc101')
        return False

    # nid -> up/down状态的映射
    up_down_infos = disk_info_util.get_mdc_cmd_102_infos(service)
    osd_id_2_up_down = dict()
    for up_down_info in up_down_infos:
        osd_id_2_up_down[up_down_info.get('NID')] = up_down_info.get('UP/DOWN')

    # 判断是否所有nid都为up状态
    for n_id, esn in disk_osd_id_2_sn.items():
        if osd_id_2_up_down.get(n_id) == 'DOWN':
            return False
    return True
