#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import os
import re

from py.common.entity.exception import BusinessException
from py.common.service import resource_service, ssh_info_handle_util
from py.common.service.connection.ssh_connection_service import Cmd
from py.common.service.ssh_info_handle_util import find_title_line_index
from py.fusion_cube.common.constant import DiskStatus, RaidStatus
from py.fusion_cube.common.service.os_util.raid_card_common import RaidCard, NO_FILE, SATA_DISK, SAS_DISK, UNKNOWN


class RaidCard1880(RaidCard):
    """
    1880raid卡查询操作类，主要用于系统盘更换过程中进行查询和点灯操作，执行命令依赖hiraidadm工具
    [术语说明]：
    vd - virtual disk -虚拟磁盘
    rg - raid group   -硬盘组
    pd - physical disk -物理硬盘
    control id - RAID控制卡ID
    enclosure_id, slot_id - 物理硬盘在RAID卡下的定位信息：框号，槽位号
    [对应关系]：
    分布式存储场景下，自上而下:
    1个节点有1块RAID卡(专有硬件和部分通用ARM服务器的通过软RAID组RAID，无RAID卡) -> 对应1个control id
    1个节点有1个系统盘(vd) -> 对应1个硬盘组(rg) -> 由两块物理盘组成(pd)
    """
    def __init__(self, ssh_connection, disk_letter, logger, slot_id=""):
        self._ssh_service = ssh_connection
        self._logger = logger
        self._disk_letter = disk_letter
        self._cur_slot_id = slot_id

        self._tool_path = self._query_tool_path()
        self._controller_id = self._query_control_id()
        self._rg_id = self._query_rg_id()
        self._enclosure_id, self._cur_node_slot_ids = \
            self._query_enc_and_slot_ids()

    def get_pre_check_status_standard(self):
        """
        更换前检查中检查raid状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("raid.1880.sync.complete.judgements")

    def get_post_check_status_standard(self):
        """
        更换后检查中检查重构状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("check.1880.card.rebuilding.suggestion")

    def get_light_on_cmd(self, slot_id):
        return "{} c{}:e{}:s{} set led type=locate sw=on".format(
            self._tool_path, self._controller_id, self._enclosure_id,
            slot_id)

    def has_raid1(self):
        """
        判断是否是raid1
        :return: 是否是raid1
        """
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} c{}:rg{} show".format(
            self._tool_path, self._controller_id, self._rg_id)))
        match = re.findall(r"RAID\s+Level\s+\|\s+(\d+)", ssh_ret)
        if not match:
            return False
        return str(match[0]) == "1"

    def check_raid_card_status(self):
        """
        获取raid卡的状态
        状态在重构中时，返回检查的命令
        :return: (状态, 命令)
        """
        cmd = "{} c{}:rg{} show".format(self._tool_path, self._controller_id, self._rg_id)
        ssh_ret = self._ssh_service.exec_cmd(Cmd(cmd))
        match_status = re.findall(r"Status\s+\|\s+(\S+)", ssh_ret)
        match_background_task = re.findall(r"Current\s+Background\s+Task\s+\|\s+(\S+)", ssh_ret)
        if not match_status or not match_background_task:
            return RaidStatus.UNKNOWN, cmd
        if match_status[0] == "normal" and match_background_task[0] == "NA":
            return RaidStatus.ACTIVE, cmd
        if match_status[0] == "degrade" and match_background_task[0] == "Rebuild":
            return RaidStatus.REBUILDING, cmd
        return RaidStatus.UNKNOWN, cmd

    def is_raid_card_rebuilding(self):
        """
        判断raid卡是否在重构中
        :return: (是否重构中，[命令])
        """
        status, cmd = self.check_raid_card_status()
        return status != RaidStatus.ACTIVE, [cmd]

    def get_sys_disk_info(self):
        """
        查询系统盘物理盘
        :return: 物理盘列表
        """
        return self._query_pdlist()

    def _query_tool_path(self):
        cmd = "find / -name hiraidadm"
        # 优先调用工具路径
        default_path = "/usr/bin/hiraidadm"
        # 模块内部预置工具，特定用户才有权限运行
        black_path = "/opt/dfv/oam/public/hotpatch/plogserver/general/hiraidadm"
        ret = self._ssh_service.exec_cmd(Cmd(cmd))
        path_lines = ret.replace(cmd, "").splitlines()
        real_path = ""
        for line in path_lines:
            if NO_FILE in line or "hiraidadm" not in line.lower() or black_path in line.lower():
                continue
            if default_path in line:
                return line
            real_path = line
        return real_path

    def _query_control_id(self):
        if not self._tool_path:
            self._logger.error("The hiraidadm tool is not found.")
            raise BusinessException(resource_service.get_msg('cant.find.hiraidadm.tool.path.and.query.sys.failed'))
        ssh_str = self._ssh_service.exec_cmd(Cmd("{} show allctrl".format(self._tool_path)))
        match = re.findall(r"Controller\s+Id\s+\|\s+(\d+)", ssh_str)
        if not match:
            self._logger.error("Failed to obtain the controller ID.")
            raise BusinessException(resource_service.get_msg("query.raid.controller.failed"))
        if len(match) > 1:
            raise BusinessException(resource_service.get_msg("check.raid.more.than.one.suggestion"))
        return match[0]

    def _query_rg_id(self):
        vd_rg_dict = self._get_vd_rg_dict()
        for vd_id, rg_id in vd_rg_dict.items():
            drive = self._query_driver_letter_by_vd(vd_id)
            if drive == "/dev/{}".format(self._disk_letter):
                return rg_id
        raise BusinessException(resource_service.get_msg("query.raid.disk.group.failed"))

    def _get_vd_rg_dict(self):
        vd_rg_dict = dict()
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} c{} show vdlist".format(
            self._tool_path, self._controller_id)))
        ret_lines = ssh_ret.splitlines()
        title_line = find_title_line_index(ret_lines, ["VDID", "RGID", "Name"])
        table_data = ssh_info_handle_util.parse_horizontal_table_by_chars(
            os.linesep.join(ret_lines[title_line: len(ret_lines) - 1]), " ")
        for single_data in table_data:
            vd_rg_dict[single_data.get("VDID")] = single_data.get("RGID")
        return vd_rg_dict

    def _query_driver_letter_by_vd(self, vd_id):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} c{}:vd{} show".format(
            self._tool_path, self._controller_id, vd_id)))
        match = re.findall(r"OS\s+Drive\s+Letter\s+\|\s+(\S+)", ssh_ret)
        return match[0] if match else ""

    def _query_enc_and_slot_ids(self):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} c{}:rg{} show".format(
            self._tool_path, self._controller_id, self._rg_id)))
        match = re.findall(r"Enc\s+Slot\s+(\d+):(\d+)\s+(\d+):(\d+)", ssh_ret)
        if not match or len(match[0]) != 4:
            raise BusinessException(resource_service.get_msg("query.raid.disk.group.failed"))
        return match[0][0], [match[0][1], match[0][3]]

    def _query_pdlist(self):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} c{} show pdlist".format(
            self._tool_path, self._controller_id)))
        ssh_ret_lines = ssh_ret.splitlines()
        title_index = find_title_line_index(ssh_ret_lines, ["Index", "Enc",
                                                            "Slot", "DID"])
        table_data = ssh_info_handle_util.parse_horizontal_table_by_chars(
            os.linesep.join(ssh_ret_lines[title_index: len(ssh_ret_lines) - 1]), " ")
        disks = list()
        for data in table_data:
            if data.get("Enc") == self._enclosure_id and data.get("Slot") in self._cur_node_slot_ids:
                disks.append(self._init_disk_by_data(data))
        return disks

    def _init_disk_by_data(self, data):
        disk = dict()
        disk["phySlotId"] = int(data.get("Slot"))
        disk["mediaCapacity"] = data.get("Capacity")
        disk["mediaType"] = self._get_media_type(data.get("Intf"))
        disk["health"] = self._get_health_status(data.get("Status"))
        disk["devName"] = self._disk_letter
        disk["phyDevEsn"] = self._query_esn_by_phy_slot(data.get("Slot"))
        return disk

    @staticmethod
    def _get_media_type(ret_intf):
        if "SATA" == ret_intf:
            return SATA_DISK
        elif "SAS" == ret_intf:
            return SAS_DISK
        else:
            return UNKNOWN

    @staticmethod
    def _get_health_status(ret_status):
        if "online" in ret_status:
            return DiskStatus.HEALTH
        elif "rebuild" in ret_status.lower() or "rbld" in ret_status.lower():
            return DiskStatus.REBUILDING
        else:
            return DiskStatus.FAULT

    def _query_esn_by_phy_slot(self, slot):
        ssh_ret = self._ssh_service.exec_cmd(
            Cmd("{} c{}:e{}:s{} show".format(self._tool_path,
                                             self._controller_id,
                                             self._enclosure_id, slot)))
        match = re.findall(r"Serial\s+Number\s+\|\s+(\S+)", ssh_ret)
        if match:
            return match[0]
        raise BusinessException(resource_service.get_msg("query.raid.disk.group.failed"))
