#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/15
@file: disk_info_util.py
@function:
@modify:
"""
import re

from py.common.entity.exception import BusinessException
from py.common.service import resource_service
from py.common.service.connection.ssh_connection_service import Cmd
from py.common.service.connection.ssh_connection_service import SshService
from py.fusion_cube.common.constant import RaidStatus, RaidCardType, DiskStatus, NODE_SYS_DISKS
from py.fusion_cube.common.context import disk_context_util
from py.fusion_cube.common.service.os_util.raid_1880_util import RaidCard1880

CMD_INVALID_KEYS = ("command not found", "No such file or directory",
                    "COMMAND NOT SUPPORTED")
LSI_SAS_3008_RAID_CARD = "3008"
MSCC_SMARTRAID_3152 = "3152"
# ARM的回文中是首字母大写，忽略大小写对比
RAID_CARD_CONTROLLER_5 = "mixed"
SAS_DISK = "sas_disk"
SATA_DISK = "sata_disk"
UNKNOWN = "unknown"
NO_FILE = "No such file or directory"


def raid_card_set_work_mode(context):
    '''
    设置raid卡的工作模式，设置为5，一般都是5，Mixed模式
    如果不是3152卡，不需要设置，直接返回True
    :param connection:
    :return:
    '''
    connection = SshService(context.getNode())
    if not raid_card_version_is_smart_raid_3152(connection):
        return True, ""
    flag, raid_card_id = get_raid_card_id(connection)
    if not flag:
        return False, resource_service.get_msg("query.raid.card.id.failed")
    cmd = "arcconf setcontrollermode {} 5".format(raid_card_id)
    connection.exec_cmd(Cmd(cmd))
    if raid_card_controller_mode_is_mixed(connection, raid_card_id):
        return True, ""
    return False, resource_service.get_msg("set.raid.card.mode.failed").format(
        raid_card_id)


def raid_card_controller_mode_is_mixed(connection, raid_card_id):
    cmd = 'arcconf getconfig {} ad |grep -i "controller mode" |cat'.format(
        raid_card_id)
    ssh_ret = connection.exec_cmd(Cmd(cmd))
    match = re.findall("\scontroller mode\s*:\s*(\w+)\s?", ssh_ret.lower())
    if match:
        return match[0] == RAID_CARD_CONTROLLER_5
    else:
        return False


def get_raid_card_id(connection):
    cmd = "arcconf list"
    ssh_ret = connection.exec_cmd(Cmd(cmd))
    match = re.findall(r"(?<=Controller )\d+", ssh_ret)
    if match:
        # 一个节点一个raid卡，只需要取第一个
        return True, match[0]
    else:
        return False, ""


def raid_card_version_is_smart_raid_3152(connection):
    return check_raid_card_version(connection,
                                   "lspci -m |grep -i Adaptec |cat",
                                   r"smartraid\s?(\d+)",
                                   MSCC_SMARTRAID_3152)


def raid_card_version_is_lsi_sas_3008(connection):
    # LSI是厂商
    return check_raid_card_version(connection, "lspci | grep -i LSI |cat",
                                   r"\ssas\s?(\d+)\s",
                                   LSI_SAS_3008_RAID_CARD)


def raid_card_version_is_1880(connection):
    ssh_ret = connection.exec_cmd(Cmd("lspci | grep -i RAID|cat"))
    return "Device 3758" in ssh_ret


def check_raid_card_version(connection, cmd, reg, version):
    ssh_ret = connection.exec_cmd(Cmd(cmd))
    if is_invalid_cmd(ssh_ret) or is_none_cmd_info(ssh_ret):
        return False
    match = re.findall(reg, ssh_ret.lower())
    if match:
        raid_card_type = match[0]
        if not raid_card_type:
            return False
        return int(raid_card_type) == int(version)
    return False


def is_invalid_cmd(ssh_ret):
    lines = ssh_ret.splitlines()
    for invalid_key in CMD_INVALID_KEYS:
        if invalid_key in lines[-2]:
            return True
    return False


def is_none_cmd_info(ssh_ret):
    lines = ssh_ret.splitlines()
    no_info_line_num = 2
    if len(lines) < no_info_line_num:
        return True
    # 回显可能异常，结束符在正常回显之后，而未另起一行
    if len(lines) == no_info_line_num and lines[1].startswith("["):
        return True
    return False


def get_raid_card(context, raid_card_type, logger):
    ssh_service = SshService(context.getNode())
    if RaidCardType.SAS3152 in raid_card_type:
        return RaidCard3152(ssh_service, disk_context_util.get_disk_letter(context), logger)
    elif is_3508_supported(raid_card_type):
        slot_ids = [disk.getSlotNo() for disk in context.getEnv().get(NODE_SYS_DISKS)]
        return RaidCard3508(ssh_service, disk_context_util.get_disk_letter(context), logger,
                            disk_context_util.get_disk_slot(context), slot_ids)
    elif is_1880_supported(raid_card_type):
        return RaidCard1880(ssh_service,
                            disk_context_util.get_disk_letter(context), logger,
                            disk_context_util.get_disk_slot(context))
    return None


def has_soft_raid1(ssh_connection):
    ssh_ret = ssh_connection.exec_cmd(Cmd("cat /proc/mdstat"))
    split_lines = ssh_ret.splitlines()
    for line in split_lines:
        if "Personalities" in line and "raid1" in line:
            return True
    return False


def is_3508_supported(raid_card_type):
    return RaidCardType.SAS3408 in raid_card_type \
           or RaidCardType.SAS3108 in raid_card_type \
           or RaidCardType.SAS3416 in raid_card_type \
           or RaidCardType.SAS3508 in raid_card_type \
           or RaidCardType.SAS3908 in raid_card_type


def is_3152_supported(raid_card_type):
    return RaidCardType.SAS3152 in raid_card_type


def is_1880_supported(raid_card_type):
    return RaidCardType.RAID1880 in raid_card_type


class RaidCard(object):
    def has_raid1(self):
        raise NotImplementedError

    def check_raid_card_status(self):
        raise NotImplementedError

    def is_raid_card_rebuilding(self):
        raise NotImplementedError


class RaidCard3008(RaidCard):
    def __init__(self, ssh_connection, tool_path, logger):
        self._ssh_service = ssh_connection
        self._tool_path = tool_path
        self._logger = logger

    def check_raid_card_status(self):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("ls"))
        str_list = ssh_ret.split()
        if len(str_list) > 1:
            return RaidStatus.UNKNOWN
        ssh_ret = self._ssh_service.exec_cmd(Cmd("cat /sys/class/raid_devices/{}/state".format(str_list[0])))
        if "active" in ssh_ret:
            return RaidStatus.ACTIVE
        return RaidStatus.REBUILDING

    def get_controller_id(self):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} list".format(self._tool_path)))
        ret_lines = ssh_ret.splitlines()
        range_list = range(len(ret_lines))
        for index in range_list:
            if "Index" in ret_lines[index] and index + 2 < len(ret_lines):
                return ret_lines[index + 2].split()[0]
        self._logger.info("get controller id failed.")
        return ""

    def has_raid1(self):
        controller_id = self.get_controller_id()
        if not controller_id:
            return False
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} {} display".format(self._tool_path, controller_id)))
        ret_lines = ssh_ret.splitlines()
        for line in ret_lines:
            if "RAID level" in line:
                match = re.findall(r"RAID\d+", line)
                if match and match[0] == "RAID1":
                    return True
        return False

    def is_raid_card_rebuilding(self):
        raise NotImplementedError


class RaidCard3508(RaidCard):
    _SLOT_INDEX = 0
    _HEALTH_INDEX = 2
    _MEDIA_CAPATICY = 4
    _MEDIA_CAPATICY_UNIT = 5
    _MEDIA_TYPE_INDEX = 6

    def __init__(self, ssh_connection, disk_letter, logger, slot_id="", slot_ids=None):
        if slot_ids is None:
            slot_ids = []
        self._ssh_service = ssh_connection
        self._tool_path = RaidCard3508._get_tool_path(ssh_connection)
        self._controller_id = ""
        self._vd_id = ""
        self._enclosure_id = ""
        # 当前更换盘槽位号
        self._cur_slot_id = slot_id
        # 当前节点对应所有系统盘槽位号列表
        self._cur_node_slot_ids = slot_ids
        self._disk_letter = disk_letter
        self._unit_id = ""
        self._device_info = list()
        self._logger = logger

    def get_pre_check_status_standard(self):
        """
        更换前检查中检查raid状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("raid.3508.sync.complete.judgements")

    def get_post_check_status_standard(self):
        """
        更换后检查中检查重构状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("check.3508.card.rebuilding.suggestion")

    def get_light_on_cmd(self, slot_id):
        """
        获取点亮指定硬盘的定位指示灯的命令
        :param slot_id: 硬盘槽位号
        :return: 点灯命令
        """
        return "{} /c{}/e{}/s{} start locate".format(
            self._tool_path, self._get_controller_id(),
            self._get_enclosure_id(), slot_id)

    @staticmethod
    def _get_tool_path(ssh_connection):
        cmd = "find / -name storcli64"
        default_path = "/opt/MegaRAID/storcli/"
        ret = ssh_connection.exec_cmd(Cmd(cmd))
        path_lines = ret.splitlines()
        real_path = ""
        for line in path_lines:
            if NO_FILE in line:
                continue
            if default_path in line:
                return line
            real_path = line
        return real_path

    def _get_controller_id(self):
        if self._controller_id:
            return self._controller_id
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} /call show".format(self._tool_path)))
        match = re.findall(r"(?<=Controller = )\d+", ssh_ret)
        if match:
            self._controller_id = match[0]
            return self._controller_id
        self._logger.info("Failed to obtain the controller ID.")
        return self._controller_id

    def _get_unit_id(self):
        if self._unit_id:
            return True, self._unit_id
        ssh_ret = self._ssh_service.exec_cmd(Cmd("smartctl -a /dev/{}".format(self._disk_letter)))
        ret_lines = ssh_ret.splitlines()
        for line in ret_lines:
            match = re.findall(r"Logical Unit id:(.*)", line)
            if match:
                self._unit_id = match[0].strip()[2:]
                return True, self._unit_id
        self._logger.info("Failed to obtain the unit ID.")
        return False, ""

    def _get_vd_id(self):
        if self._vd_id:
            return True, self._vd_id
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} /call /vall show all".format(self._tool_path)))
        ret_lines = ssh_ret.splitlines()
        is_find_start = False
        is_success, unit_id = self._get_unit_id()
        if not is_success:
            self._vd_id = ""
            return False, self._vd_id
        for line in ret_lines:
            if not is_find_start:
                match = re.findall(r"VD(\d+) Properties", line)
                if not match:
                    continue
                self._vd_id = match[0]
                is_find_start = True
            if "SCSI NAA Id" not in line:
                continue
            if unit_id in line:
                return True, self._vd_id
            else:
                is_find_start = False
                self._vd_id = ""
        self._logger.info("get vd id failed.")
        raise BusinessException(err_msg=resource_service.get_msg(
            'query.sys.disk.because.storcli.version'))

    def _get_enclosure_id(self):
        if self._enclosure_id:
            return self._enclosure_id
        match_reg = r"(\d+):{}".format(self._cur_slot_id)
        device_info = self._get_device_info()
        for line in device_info:
            match = re.findall(match_reg, line)
            if match:
                self._enclosure_id = match[0]
                return self._enclosure_id
        self._logger.info("get enclosure id failed.")
        return ""

    def _get_device_info(self):
        if self._device_info:
            return self._device_info
        is_success, vd_id = self._get_vd_id()
        if not is_success:
            return ""
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} /call /vall show all".format(self._tool_path)))
        pd_infos = ssh_ret.split("PDs for VD {}".format(vd_id))
        if len(pd_infos) < 2:
            return ""
        split_lines = pd_infos[1].splitlines()
        is_find_start = False
        skip_line = False
        ret_lines = list()
        for line in split_lines:
            if not is_find_start:
                if "EID:Slt" not in line:
                    continue
                skip_line = True
                is_find_start = True
                continue
            if skip_line:
                skip_line = False
                continue
            if "----" not in line:
                ret_lines.append(line)
                continue
            break
        return ret_lines

    def _is_optimal(self, c_id, v_id):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} /c{}/v{} show".format(self._tool_path, c_id, v_id)))
        split_lines = ssh_ret.splitlines()
        range_list = range(len(split_lines))
        for index in range_list:
            if "State" in split_lines[index] and index + 2 < len(split_lines):
                return "Optl" in split_lines[index + 2]
        return False

    def _is_in_progress(self, c_id, e_id, s_id):
        cmd = "{} /c{}/e{}/s{} show rebuild".format(self._tool_path, c_id, e_id, s_id)
        ssh_ret = self._ssh_service.exec_cmd(
            Cmd(cmd))
        return (True, cmd) if "In progress" in ssh_ret else (False, cmd)

    def _get_disk_esn(self, slot_id):
        ssh_ret = self._ssh_service.exec_cmd(
            Cmd("{} /c{}/e{}/s{} show all".format(self._tool_path, self._get_controller_id(), self._get_enclosure_id(),
                                                  slot_id)))
        device_attributes = ssh_ret.split("Device attributes")
        if len(device_attributes) < 2:
            self._logger.error("get esn failed.")
            return ""
        ret_lines = device_attributes[1].splitlines()
        for line in ret_lines:
            match = re.findall(r"SN = (.*)", line)
            if not match:
                continue
            return match[0]
        self._logger.error("get esn failed.")
        return ""

    def _init_disk(self, device_info):
        disk_info = device_info.split()
        if len(disk_info) < self._MEDIA_TYPE_INDEX + 1:
            return None
        disk = dict()
        disk["mediaCapacity"] = disk_info[self._MEDIA_CAPATICY] + disk_info[self._MEDIA_CAPATICY_UNIT]
        if "SAS" in disk_info[self._MEDIA_TYPE_INDEX]:
            disk["mediaType"] = SAS_DISK
        elif "SATA" in disk_info[self._MEDIA_TYPE_INDEX]:
            disk["mediaType"] = SATA_DISK
        else:
            disk["mediaType"] = UNKNOWN
        match = re.findall(r"\d+:(\d+)", disk_info[self._SLOT_INDEX])
        disk["phySlotId"] = int(match[0] if match else "-1")
        disk["phyDevEsn"] = self._get_disk_esn(disk.get("phySlotId"))
        if "Onln" in disk_info[self._HEALTH_INDEX]:
            disk["health"] = DiskStatus.HEALTH
        elif "Rbld" in disk_info[self._HEALTH_INDEX]:
            disk["health"] = DiskStatus.REBUILDING
        else:
            disk["health"] = DiskStatus.FAULT
        disk["devName"] = self._disk_letter
        return disk

    def get_sys_disk_info(self):
        disks = list()
        is_success, vd_id = self._get_vd_id()
        if not is_success:
            return disks
        device_info = self._get_device_info()
        disks = list()
        for info in device_info:
            disks.append(self._init_disk(info))
        return disks

    def check_raid_card_status(self):
        controller_id = self._get_controller_id()
        if not controller_id:
            return RaidStatus.UNKNOWN, ""
        is_success, vd_id = self._get_vd_id()
        if not is_success:
            return RaidStatus.UNKNOWN, ""
        if self._is_optimal(controller_id, vd_id):
            return RaidStatus.ACTIVE, ""
        flag, cmd = self._is_in_progress(controller_id, self._get_enclosure_id(), self._cur_slot_id)
        if flag:
            return RaidStatus.REBUILDING, cmd
        return RaidStatus.UNKNOWN, ""

    def has_raid1(self):
        controller_id = self._get_controller_id()
        if not controller_id:
            return False
        is_success, vd_id = self._get_vd_id()
        if not is_success:
            return False
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} /c{}/v{} show all".format(self._tool_path, controller_id, vd_id)))
        match = re.findall(r"RAID\d+", ssh_ret)
        if match and match[0] == "RAID1":
            return True
        return False

    def is_raid_card_rebuilding(self):
        result_cmd = []
        for slot_id in self._cur_node_slot_ids:
            flag, cmd = self._is_in_progress(self._get_controller_id(), self._get_enclosure_id(), slot_id)
            result_cmd.append(cmd)
            if flag:
                return True, result_cmd
        return False, result_cmd


class RaidCard3152(RaidCard):
    CONTROLLER_LD = "{} getconfig {} ld"
    _SLOT_INDEX = 5
    _MEDIA_CAPATICY = 0
    _MEDIA_TYPE_INDEX = 1

    def __init__(self, ssh_connection, disk_letter, logger):
        self._ssh_service = ssh_connection
        self._tool_path = RaidCard3152._get_tool_path(ssh_connection)
        self._controller_id = ""
        self._disk_letter = disk_letter
        self._logger = logger

    def get_pre_check_status_standard(self):
        """
        更换前检查中检查raid状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("raid.3152.sync.complete.judgements")

    def get_post_check_status_standard(self):
        """
        更换后检查中检查重构状态的检查标准描述
        :return: 标准描述
        """
        return resource_service.get_msg("check.3152.card.rebuilding.suggestion")

    def get_light_on_cmd(self, slot_id):
        """
        获取点亮指定硬盘的定位指示灯的命令
        :param slot_id: 硬盘槽位号
        :return: 点灯命令
        """
        return "{} identify {} device {} time {}".format(
            self._tool_path, self._get_controller_id(),
            self._get_physical_id(slot_id), 1800)

    def _get_physical_id(self, slot_id):
        ssh_ret = self._ssh_service.exec_cmd(
            Cmd("{} list {}".format(self._tool_path,
                                    self._get_controller_id())))
        for line in ssh_ret.splitlines():
            if "Slot {}".format(slot_id) in line:
                match_id = re.search(r"Physical (\d+,\d+)", line)
                if match_id:
                    return match_id.group(1).replace(',', ' ')
        raise BusinessException('cant find physical slot.')

    @staticmethod
    def _get_tool_path(ssh_connection):
        cmd = "find / -name arcconf"
        default_path = "/usr/Arcconf"
        ret = ssh_connection.exec_cmd(Cmd(cmd))
        path_lines = ret.replace(cmd, "").splitlines()
        real_path = ""
        for line in path_lines:
            if NO_FILE in line or "arcconf" not in line.lower():
                continue
            if default_path in line:
                return line
            real_path = line
        return real_path

    def _get_controller_id(self):
        if self._controller_id:
            return self._controller_id
        if not self._tool_path:
            self._logger.error("The arrconf tool is not found.")
            raise BusinessException(resource_service.get_msg(
                'cant.find.arcconf.tool.path.and.query.sys.failed'))
        ssh_str = self._ssh_service.exec_cmd(Cmd("{} list".format(self._tool_path)))
        match = re.findall(r"(?<=Controller )\d+", ssh_str)
        if not match:
            self._logger.error("Failed to obtain the controller ID.")
            return ""
        if len(match) > 1:
            raise BusinessException(
                resource_service.get_msg("check.raid.more.than.one.suggestion"))
        self._controller_id = match[0]
        return self._controller_id

    def _get_sys_disk_health_status(self, slot_id):
        ssh_ret = self._ssh_service.exec_cmd(Cmd("{} list {}".format(self._tool_path, self._get_controller_id())))
        ret_lines = ssh_ret.splitlines()
        for line in ret_lines:
            if not re.findall(r"Slot {}".format(slot_id), line):
                continue
            if "Online" in line:
                return DiskStatus.HEALTH
            if "Rebuilding" in line:
                return DiskStatus.REBUILDING
        return DiskStatus.FAULT

    def _init_disk(self, disk_info, origin_line):
        disk = dict()
        if len(disk_info) < self._SLOT_INDEX + 1:
            return None
        disk["mediaCapacity"] = disk_info[self._MEDIA_CAPATICY]
        if "SAS" in disk_info[self._MEDIA_TYPE_INDEX]:
            disk["mediaType"] = SAS_DISK
        elif "SATA" in disk_info[self._MEDIA_TYPE_INDEX]:
            disk["mediaType"] = SATA_DISK
        else:
            disk["mediaType"] = UNKNOWN
        disk["phySlotId"] = int(re.findall(r"(?<=Slot:)\d+", disk_info[self._SLOT_INDEX])[0])
        disk["phyDevEsn"] = origin_line.split()[-1]
        disk["health"] = self._get_sys_disk_health_status(disk.get("phySlotId"))
        disk["devName"] = self._disk_letter
        return disk

    def get_sys_disk_info(self):
        controller_id = self._get_controller_id()
        ssh_ret = self._ssh_service.exec_cmd(Cmd(self.CONTROLLER_LD.format(self._tool_path, controller_id)))
        # 通过盘符获取具体的逻辑设备信息所在的字符串
        logical_device_infos = ssh_ret.split("Logical Device number")
        device_info = ""
        for info in logical_device_infos:
            if re.findall(r"Disk Name( )+: /dev/{}".format(self._disk_letter), info):
                device_info = info
                break
        ret_lines = device_info.splitlines()
        disks = list()
        for line in ret_lines:
            match = re.findall(r"Present \((.*)\)", line)
            if not match:
                continue
            disk_info = str(match[0]).split(",")
            disk = self._init_disk(disk_info, line)
            if disk:
                disks.append(disk)
        return disks

    def check_raid_card_status(self):
        controller_id = self._get_controller_id()
        if not controller_id:
            return RaidStatus.UNKNOWN, ""
        cmd = self.CONTROLLER_LD.format(self._tool_path, controller_id)
        ssh_ret = self._ssh_service.exec_cmd(Cmd(cmd))
        ret_lines = ssh_ret.splitlines()
        for line in ret_lines:
            if "Status of Logical Device" in line and "Optimal" in line:
                return RaidStatus.ACTIVE, ""
        return RaidStatus.REBUILDING, cmd

    def has_raid1(self):
        controller_id = self._get_controller_id()
        if not controller_id:
            return False
        ssh_ret = self._ssh_service.exec_cmd(Cmd(self.CONTROLLER_LD.format(self._tool_path, controller_id)))
        ret_lines = ssh_ret.splitlines()
        for line in ret_lines:
            if "RAID level" not in line:
                continue
            match = re.findall(r"\d+", line)
            if match and match[0] == "1":
                return True
        return False

    def is_raid_card_rebuilding(self):
        status, cmd = self.check_raid_card_status()
        return status == RaidStatus.REBUILDING, [cmd]
