#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2022/01/06
@file: raid_sync_complete_check_util.py
@function: 检查raid同步状态相关方法工具类
@modify:
"""
from py.common.entity.exception import BusinessException
from py.common.service.connection.ssh_connection_service import Cmd


class RaidSyncStatus(object):
    ACTIVATE = 'activate'
    REBUILDING = 'rebuilding'
    UNKNOWN = 'unknown'


def get_raid_sync_query_cmds(disk_info):
    """
    根据盘符信息，组织查询raid同步状态的命令
    :param disk_info: 盘符信息 DiskInfo对象
    :return: raid查询命令list
    """
    res = list()
    cmd = 'mdadm --detail /dev/{}'
    for disk in disk_info:
        if disk.children:
            _format_cmd(disk, cmd, res)
    return res


def _format_cmd(disk, cmd, res):
    for part in disk.children:
        if part.children:
            for raid in part.children:
                res.append(cmd.format(raid.name))


def get_raid_sync_status(cmds, ssh, logger, handle_res_func):
    """
    查看raid是否为数据同步完成状态
    :param cmds: 需要执行的命令
    :param ssh: shell连接
    :param logger: 日志
    :param handle_res_func: 回显处理类
    :return: boolean
    """
    if not isinstance(cmds, list):
        logger.error('cmds should be list type:{}'.format(cmds))
        raise BusinessException('cmds should be list:{}'.format(cmds))
    is_activate = True
    for item in cmds:
        res = ssh.exec_cmd_handle_res(Cmd(item), handle_res_func=handle_res_func)
        logger.info('check sync complete res:{}'.format(res))
        if not res:
            return False
        res_list = res.strip().splitlines()
        if len(res_list) < 2:
            return False
        res_list = res_list[-2:]
        for raid_stat in res_list:
            if RaidSyncStatus.REBUILDING in raid_stat:
                return RaidSyncStatus.REBUILDING
            if 'active' not in raid_stat:
                is_activate = False
    return RaidSyncStatus.ACTIVATE if is_activate else RaidSyncStatus.UNKNOWN


def get_cmd_show_string(cmd_list):
    """
    raid同步状态命令list组织成界面显示的提示信息
    :param cmd_list: 命令集合
    :return: 提示字符串
    """
    if not isinstance(cmd_list, list):
        return '1.' + str(cmd_list)
    cmd_res = list()
    for index, item in enumerate(cmd_list):
        cmd_res.append(str((index + 1)) + '.' + item)
    return '\n'.join(cmd_res)
