#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/03
@file: check_whether_disk_added_to_storage_pool.py
@function: 更换后盘是否加入更换前的存储池
@modify:
"""
import traceback
import time

from py.fusion_cube.common.context import disk_context_util
from py.fusion_cube.common.constant import ClusterRestUri
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.common.service.connection.ssh_connection_service import SshService
from py.common.entity.item_status import ItemStatus
from py.common.service import logger_factory
from py.common.service import resource_service
from py.common.entity.exception import ConnectionException
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.fusion_cube.common.context.common_context_util import \
    set_cur_task_suggestion
from py.fusion_cube.common.service import add_disk_to_pool_service
from py.fusion_cube.common.record import record_cache_disk_replace_type
from py.fusion_cube.common.service import disk_out_storage_pool_service
from py.fusion_cube.common.record import record_main_disk_out_pool
from py.fusion_cube.common.service.storage_pool_status_service import \
    check_storage_pool_is_target, POOL_STATUS_REFACTORING
from py.fusion_cube.common.service.disk_out_storage_pool_service import \
    get_storage_pool_ids_by_disk_pool_ids


@auto_brush_progress(300)
def execute(context):
    return AddDiskToStoragePoolItem(context, 300, 30).check()


class AddDiskToStoragePoolItem(object):

    def __init__(self, context, wait_time, query_interval_time):
        if query_interval_time <= 0:
            raise ValueError
        self.wait_time = wait_time
        self.query_interval_time = query_interval_time
        self.context = context
        self.logger = logger_factory.create_logger(__file__)
        self.ssh_service = SshService(context.getNode())
        self.rest_service = CubeRestService(self.context.getCluster())
        self._replace_esn = disk_context_util.get_replace_disk_sn(context)

    def check(self):
        waited_time = 0
        while waited_time < self.wait_time:
            try:
                # 判断新盘是否已经加入了存储池
                is_disk_already_add = \
                    disk_out_storage_pool_service.is_disks_add_in_pool(
                        SshService(self.context.getNode()), [self._replace_esn])
                if is_disk_already_add:
                    return ItemStatus.NOT_INVOLVED, ""
                time.sleep(self.query_interval_time)
                waited_time += self.query_interval_time
            except Exception:
                self.logger.error("query is disk already add failed:{}".format(
                    traceback.format_exc()))
        need_add_pool_ids = disk_context_util.get_disk_belong_pool_ids(self.context)
        self.logger.info("need_add_pool_ids: {}".format(need_add_pool_ids))
        try:
            # 判断存储池状态是否为重构, 重构则提示等待重构完成后重试
            err_msgs = self._check_is_refactoring(need_add_pool_ids)
            if err_msgs:
                set_cur_task_suggestion(self.context, resource_service.get_msg(
                        'main.pool.status.refactoring.retry'))
                return ItemStatus.FAILED, "\n".join(err_msgs)
            # 新盘自动加入失败，需要再次查询旧盘是否踢盘（主要考虑旧盘由未踢盘变为已踢盘场景）
            record_main_disk_out_pool.record(self.context)
            if record_main_disk_out_pool.is_main_disk_out_pool(self.context):
                add_disk_to_pool_service.add_disk_to_pool(
                    self.rest_service, need_add_pool_ids,
                    self.context.getData(),
                    self.context.getDataAfterReplace())
            else:
                add_disk_to_pool_service.force_add_main_disk_to_pool(
                    self.rest_service,
                    need_add_pool_ids,
                    self.context.getData(),
                    self.context.getDataAfterReplace())
            return ItemStatus.SUCCESS, ""
        except ConnectionException as ex:
            return ItemStatus.FAILED, resource_service.get_msg(
                "add.storage.pool.failed").format(
                str(need_add_pool_ids),
                CubeRestService.get_description(ex.response))

    def _check_is_refactoring(self, need_add_pool_ids):
        storage_pool_ids = get_storage_pool_ids_by_disk_pool_ids(
            need_add_pool_ids, self.rest_service, self.context)
        refactoring_pool_ids, not_queried_pool_ids = \
            check_storage_pool_is_target(
                self.rest_service, storage_pool_ids, [POOL_STATUS_REFACTORING])
        err_msgs = list()
        if not_queried_pool_ids:
            err_msgs.append(resource_service.get_msg(
                "not.queried.pool.status").format(str(not_queried_pool_ids)))
        if refactoring_pool_ids:
            err_msgs.append(resource_service.get_msg(
                "pool.status.refactoring").format(str(refactoring_pool_ids)))
        return err_msgs

    def get_need_add_pool_ids(self):
        pool_infos = self.get_replace_disk_belong_pool_infos()
        replace_disk_belong_pool_ids = self.parse_pool_ids(pool_infos)
        return self.find_not_add_pool_ids(replace_disk_belong_pool_ids)

    def get_replace_disk_belong_pool_infos(self):
        res_dict = self.rest_service.exec_get(ClusterRestUri.QUERY_ALL_INFO)
        for disk_infos in res_dict.get("disks", {}).values():
            for info in disk_infos:
                if disk_context_util.get_replace_disk_sn(
                        self.context) == info.get("devEsn"):
                    return info.get("devPools")
        raise Exception("not queried replace info")

    def parse_pool_ids(self, pool_infos):
        return [pool_info.get("poolId") for pool_info in pool_infos]

    def find_not_add_pool_ids(self, replace_disk_belong_pool_ids):
        old_disk_belong_pool_ids = disk_context_util.get_disk_belong_pool_ids(
            self.context)
        not_add_pool_ids = list()
        for old_disk_belong_pool_id in old_disk_belong_pool_ids:
            if old_disk_belong_pool_id not in replace_disk_belong_pool_ids:
                not_add_pool_ids.append(old_disk_belong_pool_id)
        return not_add_pool_ids


@auto_brush_progress(300)
def execute_add_cache_match_main_disks(context):
    """
    离线更换，当缓存盘对应主存盘被踢出存储池时执行，需要在恢复亚健康之后
    """
    if record_cache_disk_replace_type.is_online_replace_type(context):
        return ItemStatus.NOT_INVOLVED, ""
    out_pool_main_disks = disk_out_storage_pool_service.get_out_pool_cache_match_main_disk(
        context)
    if not out_pool_main_disks:
        return ItemStatus.NOT_INVOLVED, ""
    rest_service = CubeRestService(context.getCluster())
    old_disk = context.getData()
    try:
        add_disk_to_pool_service.add_single_disks_to_pool(
            rest_service,
            old_disk.getBelongPoolIds(),
            old_disk.getBelongNodeIp(),
            out_pool_main_disks
        )
        return ItemStatus.SUCCESS, ""
    except ConnectionException as ex:
        return ItemStatus.FAILED, resource_service.get_msg(
            "add.storage.pool.failed").format(
            str(old_disk.getBelongPoolIds()),
            CubeRestService.get_description(ex.response))
    except Exception:
        # 加盘命令执行失败, 轮询15分钟。
        rest_service.release_connect()
        logger_factory.create_logger(__file__)\
            .info('Add cache match main disk failed. polling query 10min')
        start_ime = time.time()
        while time.time() - start_ime < 15 * 60:
            if disk_out_storage_pool_service.is_disks_add_in_pool(
                    SshService(context.getNode()),
                    [disk.getEsn() for disk in out_pool_main_disks]):
                return ItemStatus.SUCCESS, ""
            time.sleep(20)
        return ItemStatus.FAILED, resource_service.get_msg(
            'add.storage.pool.failed.time.out')
