#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/20
@file: change_disk_light.py
@function: 设置硬盘定位灯状态
@modify:
"""
import traceback

from py.common.entity.item_status import ItemStatus
from py.common.service import logger_factory, resource_service
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.common.service.connection.ssh_connection_service import SshService, Cmd
from py.fusion_cube.common.constant import ClusterRestUri
from py.fusion_cube.common.context import disk_context_util
from py.fusion_cube.common.record import record_raid_card_type
from py.fusion_cube.common.service.cluster import query_disk_info_service
from py.fusion_cube.common.service.os_util import raid_card_util

ACTION_LIGHT_ON = "on"
ACTION_LIGHT_OFF = "off"


def light_on(context):
    # 系统盘
    if disk_context_util.is_sys_module(context):
        return ChangeDiskLightByRaidCard(context).light_on()
    return ChangeDiskLightByRest(context).change_disk_light(ACTION_LIGHT_ON)


class ChangeDiskLightByRest(object):
    def __init__(self, context):
        self._context = context
        self._rest_service = CubeRestService(context.getCluster())
        self._node_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._slot_id = disk_context_util.get_disk_slot(context)
        self._logger = logger_factory.create_logger(__file__)
        self._disk_esn = disk_context_util.get_disk_sn(context)
        self._is_nvme_disk = disk_context_util.is_nvme_disk(context)
        self.is_disk_exist = disk_context_util.is_disk_exist(context)

    def change_disk_light(self, action):
        try:
            # 不在位盘不支持点灯
            if not self.is_disk_exist:
                return ItemStatus.NOT_INVOLVED, ""
            # 无RAID卡等情况，不支持点灯，通过接口查询是否支持;
            is_support_disk_light = \
                query_disk_info_service.is_disk_support_light(
                    self._context, self._node_ip, self._disk_esn)
            if not is_support_disk_light:
                return ItemStatus.NOT_INVOLVED, ""

            self._rest_service.exec_post(ClusterRestUri.CHANGE_DISK_LIGHT, {
                "slotId": self._slot_id,
                "action": action,
                "ip": self._node_ip,
                "devEsn": self._disk_esn
            })
        except Exception as ex:
            self._logger.error(
                "failed to change disk light, action: {}, error: {}".format(
                    action, ex))
        return ItemStatus.SUCCESS, ""


class ChangeDiskLightByRaidCard(object):
    def __init__(self, context):
        self._context = context
        self._logger = logger_factory.create_logger(__file__)

    def light_on(self):
        try:
            card_type = record_raid_card_type.get_raid_card_type(self._context)
            # 软RAID不涉及
            if not card_type:
                return ItemStatus.NOT_INVOLVED, ""
            card_service = raid_card_util.get_raid_card(
                self._context, card_type, self._logger)
            light_on_cmd = card_service.get_light_on_cmd(
                disk_context_util.get_disk_slot(self._context))
            SshService(self._context.getNode()).exec_cmd(Cmd(light_on_cmd))
        except Exception:
            self._logger.error(
                "failed to light on disk by raid card tool, "
                "error: {}".format(traceback.format_exc()))
        return ItemStatus.SUCCESS, ""
